// Author: Dave Pape
// Revision: 6/1/04 Ben Chang - added support for pfShadows
// Revision: 08/01/03 Alex Hill - changed ygSound references to use ygSoundServer

#include <stdlib.h>
#include <iostream>
#include <list>
#include <Performer/pf/pfChannel.h>
#include <Performer/pf/pfEarthSky.h>
#include <Performer/pr/pfFog.h>
#include <Performer/pf/pfScene.h>
#include <Performer/pf/pfShadow.h>
#include <Performer/pf/pfVolFog.h>
#include <pfcave.h>
#include "ygUtil.h"
#include "ygNodeDB.h"
#include "ygNet.h"
#include "ygDebugFlags.h"
#include "ygSoundServer.h"
#include "ygCAVEViewer.h"
#include "ygWorld.h"

using namespace std;

#define MAX_SHADOW_CASTERS 32

class ygDelayMsg
	{
	public:
	 ygMessage msg;
	 float time;
	 friend inline bool operator< (const ygDelayMsg& a,const ygDelayMsg& b)
			{ return a.time < b.time; };
	};


struct _ygWorldPrivateData
	{
	bool sentMessageDebug;
	list<ygDelayMsg*> messageQueue;
	pfScene* pfscene;
	pfEarthSky* pfsky;
	pfFog* pffog;
	vector<ygUser*> users;
	ygNode* viewer;
	pfShadow *pfshadow;
	ygNode *shadowcaster[MAX_SHADOW_CASTERS];
	int numShadowCasters;	
	pfVolFog* pfvolfog;
	};



float ygWorld::FrameTime = 0;
float ygWorld::FrameDeltaTime = 1.0;
int ygWorld::FrameNumber = 0;
ygWorld* ygWorld::World = NULL;


ygWorld::ygWorld(void)
	{
	ygString dummyName;
	if (World)
		cerr << "ERROR: More than one ygWorld has been created -"
			" there should be only one!\n";
	World = this;
	p_ = new struct _ygWorldPrivateData;
	p_->pfscene = new pfScene;
	p_->pfsky = new pfEarthSky;
	p_->pffog = new pfFog;
	pfCAVEMasterChan()->setESky(p_->pfsky);
	p_->sentMessageDebug = ygDebugFlags::checkDebugEnv("world.sentmessages");
	ygNode::createDummyName(dummyName);
	p_->viewer = new ygCAVEViewer(dummyName.c_str(),true);
	ygNode::createDummyName(dummyName);
	localRoot_ = ygNodeDB::create("group",dummyName);
	ygSoundServer::initServer();

	p_->pfshadow = new pfShadow;
	p_->pfshadow->addChannel(pfCAVEMasterChan());
	p_->pfshadow->setNumSources(1);
	p_->pfshadow->setSourcePos(0,0,0,-1,0);
	p_->pfshadow->setAmbientFactor (0,0.6);
	p_->pfshadow->setVal(0,0,PFSHD_PARAM_TEXTURE_SIZE,512);
	p_->pfshadow->setVal(0,0,PFSHD_PARAM_NUM_TEXTURES,32);
	p_->pfshadow->setFlags(PFSHD_BLEND_TEXTURES,1);
	p_->pfshadow->apply();
	
	p_->numShadowCasters=0;
	}


ygWorld::~ygWorld(void)
	{
	while (!p_->messageQueue.empty())
		{
		ygDelayMsg *dm = p_->messageQueue.front();
		p_->messageQueue.pop_front();
		delete dm;
		}
	delete p_;
	ygSoundServer::closeServer();
	}


void ygWorld::createRoot(void)
	{
	root_ = ygNodeDB::create("group","root");
	root_->addChild(localRoot_);
	p_->pfscene->addChild(root_->pfnode());
	}


void ygWorld::setRoot(ygNode *r)
	{
	root_ = r;
	if (root_)
		{
		root_->addChild(localRoot_);
		p_->pfscene->addChild(root_->pfnode());
		}
	}


pfScene* ygWorld::pfscene(void) const
	{
	return p_->pfscene;
	}


ygNode* ygWorld::viewer(void) const
	{
	return p_->viewer;
	}
	
pfShadow* ygWorld::pfshadow(void) const
	{
	return p_->pfshadow;
	}
	
pfVolFog* ygWorld::pfvolfog(void) const
	{
	return p_->pfvolfog;
	}
void ygWorld::reset(void)
	{
	FrameTime = ygGetTime();
	while (!p_->messageQueue.empty())
		{
		ygDelayMsg *dm = p_->messageQueue.front();
		p_->messageQueue.pop_front();
		delete dm;
		}
	if (localRoot_)
		localRoot_->resetTree();
	}


void ygWorld::frame(void)
	{
	float t = ygGetTime();
	FrameDeltaTime = t - FrameTime;
	FrameTime = t;
	if (FrameDeltaTime <= 0)
		{
		cerr << "ERROR (ygWorld::app): Bogus delta time ("
			<< FrameDeltaTime << "); kludging to .01\n";
		FrameDeltaTime = 0.01;
		}
	FrameNumber++;
	while ((!p_->messageQueue.empty()) &&
		(p_->messageQueue.front()->time <= FrameTime))
		{
		ygDelayMsg *dm = p_->messageQueue.front();
		sendMessage(dm->msg);
		p_->messageQueue.pop_front();
		delete dm;
		}
	if (root_)
		root_->appTraverse();
	if (p_->viewer)
		p_->viewer->app();
	ygSoundServer::updateServer();
	ygNet::process();
	}


void ygWorld::scheduleMessage(ygMessage& msg)
	{
	if (msg.delay <= 0)
		sendMessage(msg);
	else
		{
		ygDelayMsg * dm = new ygDelayMsg;
		dm->time = FrameTime + msg.delay;
		dm->msg = msg;
		list<ygDelayMsg*>::iterator i = p_->messageQueue.begin();
		for ( ; i != p_->messageQueue.end(); ++i)
			if (dm->time < (*i)->time)
				break;
		p_->messageQueue.insert(i,dm);
		}
	}


void ygWorld::sendMessage(ygMessage& msg)
	{
	if (p_->sentMessageDebug)
		cout << "ygWorld: sending message " << msg << endl;
	if (!msg.node)
		msg.node = ygNodeDB::find(msg.nodeName,false);
	if (msg.node)
		{
		if (msg.node->isNetMaster())
			msg.node->message(msg);
		else
			msg.node->sendNetMessage(msg);
		}
	else
		cerr << "ERROR: could not find recipient node for message '" << msg << "'\n";
	}


void ygWorld::addUser(ygUser* user)
	{
	p_->users.push_back(user);
	}


void ygWorld::removeUser(ygUser* user)
	{
	vector<ygUser*>::iterator i = p_->users.begin();
	while ((i != p_->users.end()) && (*i != user))
		++i;
	if (i != p_->users.end())
		p_->users.erase(i);
	}


int ygWorld::numUsers(void) const
	{
	return p_->users.size();
	}


ygUser* ygWorld::user(int index) const
	{
	if ((index >= 0) && (index < p_->users.size()))
		return p_->users[index];
	else
		return NULL;
	}


void ygWorld::setClip(float near,float far)
	{
	CAVENear = near;
	CAVEFar = far;
	}


void ygWorld::setSkyColor(const pfVec3& sky)
	{
	p_->pfsky->setColor(PFES_CLEAR, sky[0], sky[1], sky[2], 0.0);
	}


void ygWorld::setEarthSkyGradient (const pfVec3& top,const pfVec3& bottom,const pfVec3& horizon, const pfVec3& far, const pfVec3& near)
	{
	p_->pfsky->setColor(PFES_SKY_TOP,top[0],top[1],top[2], 0.0);
	p_->pfsky->setColor(PFES_SKY_BOT,bottom[0],bottom[1],bottom[2],0.0);
	p_->pfsky->setColor(PFES_HORIZ,horizon[0],horizon[1],horizon[2],0.0);
	p_->pfsky->setColor(PFES_GRND_FAR,far[0],far[1],far[2],0.0);
	p_->pfsky->setColor(PFES_GRND_NEAR,near[0],near[1],near[2],0.0);	
	}
void ygWorld::setHorizonAngle (float angle)
{
	p_->pfsky->setAttr (PFES_HORIZ_ANGLE,angle);
}
void ygWorld::setGroundHeight (float height)
{
	p_->pfsky->setAttr (PFES_GRND_HT,height);
}
void ygWorld::setBackgroundMode (int mode)
{
	p_->pfsky->setMode (PFES_BUFFER_CLEAR,mode);
}
void ygWorld::fogOff(void)
	{
	p_->pfsky->setFog(PFES_GENERAL, NULL);
	}


void ygWorld::setFog(int fogtype,const pfVec3& color,float onset,float opaque)
	{
	p_->pffog->setFogType(fogtype);
	p_->pffog->setColor(color[0],color[1],color[2]);
	p_->pffog->setRange(onset,opaque);
	p_->pfsky->setFog(PFES_GENERAL, p_->pffog);
	}


void ygWorld::unlinkLocalScene(void)
	{
	unlinkTree(localRoot());
	frame();
	}


void ygWorld::unlinkTree(ygNode *node)
	{
	if (!node)
		return;
	while (node->numChildren() > 0)
		unlinkTree(node->child(0));
	if (node->parent())
		node->parent()->removeChild(node);
	}

	
void ygWorld::addShadowCaster (ygNode *node)
{
	
	pfMatrix casterMat;

	if (p_->numShadowCasters<MAX_SHADOW_CASTERS)
	{
		
		p_->numShadowCasters++;
		p_->pfshadow->setNumCasters(p_->numShadowCasters);
		
		node->getTransform(casterMat);
		p_->pfshadow->setShadowCaster(p_->numShadowCasters-1,node->pfnode(),casterMat);
		p_->pfshadow->apply();
		p_->shadowcaster[p_->numShadowCasters-1] = node;
	}
}

void ygWorld::updateShadows ()
{
	pfMatrix casterMat;
	p_->pfshadow->updateView();
	

	for (int i=0;i<p_->numShadowCasters;i++)
	{
		p_->shadowcaster[i]->getTransform(casterMat);
		p_->pfshadow->updateCaster(i,casterMat);
	}
	
}
