// Revision: 11/01/01 Dave Pape

#include <Performer/pf/pfGroup.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pfdu.h>
#include "ygMessage.h"
#include "ygVolume.h"
#include "ygBox.h"
#include "ygCylinder.h"
#include "ygSphere.h"
#include "ygPointVolume.h"
#include "ygInfiniteVolume.h"
#include "ygUtil.h"


#define SWAPFLOAT(a,b)  { float swap_temp; swap_temp=a; a=b; b=swap_temp; }


/***** Parsing a message to create a volume *****/

static ygVolume* parseBoxMessage(const ygMessage& msg)
	{
	pfVec3 min,max;
	if (msg.args.size() != 7)
		{
		msg.error("","(wrong number of arguments -"
			" should be 'box MINX MINY MINZ MAXX MAXY MAXZ')");
		return NULL;
		}
	msg.getVec3Args(min,1);
	msg.getVec3Args(max,4);
	if (min[0] > max[0])
		SWAPFLOAT(min[0],max[0]);
	if (min[1] > max[1])
		SWAPFLOAT(min[1],max[1]);
	if (min[2] > max[2])
		SWAPFLOAT(min[2],max[2]);
	ygBox * box = new ygBox;
	box->min = min;
	box->max = max;
	return box;
	}


static ygVolume* parseSphereMessage(const ygMessage& msg)
	{
	if (msg.args.size() != 5)
		{
		msg.error("","(wrong number of arguments -"
			" should be 'sphere CENTERX CENTERY CENTERZ RADIUS')");
		return NULL;
		}
	ygSphere * sphere = new ygSphere;
	msg.getVec3Args(sphere->center,1);
	sphere->radius = msg.floatArg(4);
	return sphere;
	}


static ygVolume* parseCylinderMessage(const ygMessage& msg)
	{
	pfVec3 p0,p1,axis;
	float length;
	if (msg.args.size() != 8)
		{
		msg.error("","(wrong number of arguments - should be "
			"'cylinder BOTTOMX BOTTOMY BOTTOMZ TOPX TOPY TOPZ RADIUS')");
		return NULL;
		}
	msg.getVec3Args(p0,1);
	msg.getVec3Args(p1,4);
	ygCylinder *cylinder = new ygCylinder;
	cylinder->center = (p0+p1)/2.0f;
	cylinder->radius = msg.floatArg(7);
	axis = p1 - p0;
	length = axis.length();
	if (length < 0.0000001f)
		axis.set(0,0,1);
	else
		axis /= length;
	cylinder->axis = axis;
	cylinder->halfLength = length/2.0f;
	return cylinder;
	}


static ygVolume* parsePointVolumeMessage(const ygMessage& msg)
	{
	if (msg.args.size() != 4)
		{
		msg.error("","(wrong number of arguments -"
			" should be 'point X Y Z')");
		return NULL;
		}
	ygPointVolume * point = new ygPointVolume;
	msg.getVec3Args(*point,1);
	return point;
	}


ygVolume* ygParseVolumeMessage(const ygMessage& msg)
	{
	if (msg.args.size() < 1)
		{
		msg.error("","(missing arguments)");
		return NULL;
		}
	if (msg.args[0] == "box")
		return parseBoxMessage(msg);
	else if (msg.args[0] == "sphere")
		return parseSphereMessage(msg);
	else if (msg.args[0] == "cylinder")
		return parseCylinderMessage(msg);
	else if (msg.args[0] == "point")
		return parsePointVolumeMessage(msg);
	else if (msg.args[0] == "infinite")
		return new ygInfiniteVolume;
	return NULL;
	}


/******* Creating a message string from a volume ******/

static void writeBoxVolumeMessage(ygBox *box, ygString& str)
	{
	char s[256];
	sprintf(s,"volume(box %f %f %f %f %f %f)",
		box->min[0], box->min[1], box->min[2],
		box->max[0], box->max[1], box->max[2]);
	str = s;
	}


static void writeSphereVolumeMessage(ygSphere *sphere, ygString& str)
	{
	char s[256];
	sprintf(s,"volume(sphere %f %f %f %f)",
		sphere->center[0], sphere->center[1], sphere->center[2],
		sphere->radius);
	str = s;
	}


static void writeCylinderVolumeMessage(ygCylinder *cyl, ygString& str)
	{
	char s[256];
	pfVec3 bot, top;
	bot = cyl->center - cyl->axis * cyl->halfLength;
	top = cyl->center + cyl->axis * cyl->halfLength;
	sprintf(s,"volume(cylinder %f %f %f  %f %f %f  %f)",
		bot[0], bot[1], bot[2], top[0], top[1], top[2], cyl->radius);
	str = s;
	}


static void writePointVolumeMessage(ygPointVolume *point, ygString& str)
	{
	char s[256];
	sprintf(s,"volume(point %f %f %f)",
		point->vec[0], point->vec[1], point->vec[2]);
	str = s;
	}

void ygWriteVolumeMessage(ygVolume* vol, ygString& str)
	{
	if (vol->volumeType() == YGVOL_BOX)
		writeBoxVolumeMessage((ygBox*)vol,str);
	else if (vol->volumeType() == YGVOL_SPHERE)
		writeSphereVolumeMessage((ygSphere*)vol,str);
	else if (vol->volumeType() == YGVOL_CYLINDER)
		writeCylinderVolumeMessage((ygCylinder*)vol,str);
	else if (vol->volumeType() == YGVOL_POINT)
		writePointVolumeMessage((ygPointVolume*)vol,str);
	else
		str = "volume(infinite)";
	}


/****** GeoSet creation *****/

static pfGeoSet * makeBoxGeoset(ygBox * box,int style);
static pfGeoSet * makeBoxWireGeoset(ygBox * box);
static pfGeoSet * makeBoxSolidGeoset(ygBox * box);
static pfGeoSet * makeSphereGeoset(ygSphere * sphere,int style);
static pfGeoSet * makeCylinderGeoset(ygCylinder * cyl,int style);

pfGeoSet * ygCreateVolumeGeoset(ygVolume * vol, int style)
	{
	if (vol->volumeType() == YGVOL_BOX)
		return makeBoxGeoset((ygBox*)vol,style);
	else if (vol->volumeType() == YGVOL_SPHERE)
		return makeSphereGeoset((ygSphere*)vol,style);
	else if (vol->volumeType() == YGVOL_CYLINDER)
		return makeCylinderGeoset((ygCylinder*)vol,style);
	return NULL;
	}


static pfGeoSet * makeBoxGeoset(ygBox * box,int style)
	{
	if (style == YG_STYLE_WIREFRAME)
		return makeBoxWireGeoset(box);
	else
		return makeBoxSolidGeoset(box);
	}


static pfGeoSet * makeBoxWireGeoset(ygBox * box)
	{
	pfGeoSet *gset = new pfGeoSet;
	pfVec3 * verts = (pfVec3 *) pfMalloc(8 * sizeof(pfVec3),pfGetSharedArena());
	int *lengths = (int *) pfMalloc(8 * sizeof(int),pfGetSharedArena());
	ushort *ilist_v = (ushort *) pfMalloc(24 * sizeof(ushort),pfGetSharedArena());
	pfVec4 *color = (pfVec4 *) pfMalloc(sizeof(pfVec4),pfGetSharedArena());
	ushort *ilist_c = (ushort *) pfMalloc(1 * sizeof(ushort),pfGetSharedArena());
	gset->setPrimType(PFGS_LINESTRIPS);
	gset->setNumPrims(8);
	lengths[0] = 10;
	lengths[1] = 2;
	lengths[2] = 2;
	lengths[3] = 2;
	lengths[4] = 2;
	lengths[5] = 2;
	lengths[6] = 2;
	lengths[7] = 2;
	gset->setPrimLengths(lengths);
	verts[0] = box->min;
	verts[1][0] = box->max[0];  verts[1][1] = box->min[1];
	  verts[1][2] = box->min[2];
	verts[2][0] = box->max[0];  verts[2][1] = box->max[1];
	  verts[2][2] = box->min[2];
	verts[3][0] = box->min[0];  verts[3][1] = box->max[1];
	  verts[3][2] = box->min[2];
	verts[4][0] = box->min[0];  verts[4][1] = box->min[1];
	  verts[4][2] = box->max[2];
	verts[5][0] = box->max[0];  verts[5][1] = box->min[1];
	  verts[5][2] = box->max[2];
	verts[6] = box->max;
	verts[7][0] = box->min[0];  verts[7][1] = box->max[1];
	  verts[7][2] = box->max[2];
	ilist_v[0] = 0;   ilist_v[1] = 1;   ilist_v[2] = 2;   ilist_v[3] = 3;
	  ilist_v[4] = 0;   ilist_v[5] = 4;   ilist_v[6] = 5;   ilist_v[7] = 6;
	  ilist_v[8] = 7;   ilist_v[9] = 4;
	ilist_v[10] = 1;   ilist_v[11] = 5;
	ilist_v[12] = 2;   ilist_v[13] = 6;
	ilist_v[14] = 3;   ilist_v[15] = 7;
	ilist_v[16] = 4;   ilist_v[17] = 1;
	ilist_v[18] = 5;   ilist_v[19] = 2;
	ilist_v[20] = 6;   ilist_v[21] = 3;
	ilist_v[22] = 7;   ilist_v[23] = 0;
	gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, verts, ilist_v);
	color->set(1,1,1,1);
	ilist_c[0] = 0;
	gset->setAttr(PFGS_COLOR4, PFGS_OVERALL, color, ilist_c);
	gset->setLineWidth(1.0);
	gset->setGState(new pfGeoState);
	return gset;
	}


static pfGeoSet * makeBoxSolidGeoset(ygBox * box)
	{
	pfGeoSet *gset = new pfGeoSet;
	pfVec3 *verts = (pfVec3 *) pfMalloc(8*sizeof(pfVec3),pfGetSharedArena());
	int *lengths = (int *) pfMalloc(3 * sizeof(int),pfGetSharedArena());
	ushort *ilist_v = (ushort *) pfMalloc(18 * sizeof(ushort),pfGetSharedArena());
	pfVec4 *color = (pfVec4 *) pfMalloc(sizeof(pfVec4),pfGetSharedArena());
	ushort *ilist_c = (ushort *) pfMalloc(1 * sizeof(ushort),pfGetSharedArena());
	gset->setPrimType(PFGS_TRISTRIPS);
	gset->setNumPrims(3);
	lengths[0] = 10;
	lengths[1] = 4;
	lengths[2] = 4;
	gset->setPrimLengths(lengths);
	verts[0].set(0,0,0);  verts[1].set(0,0,0);  verts[2].set(0,0,0);
	  verts[3].set(0,0,0);  verts[4].set(0,0,0);  verts[5].set(0,0,0);
	  verts[6].set(0,0,0);  verts[7].set(0,0,0);
	ilist_v[0] = 1;   ilist_v[1] = 0;   ilist_v[2] = 2;   ilist_v[3] = 3;
	  ilist_v[4] = 6;  ilist_v[5] = 7;   ilist_v[6] = 5;   ilist_v[7] = 4;
	  ilist_v[8] = 1;   ilist_v[9] = 0;
	ilist_v[10] = 2;  ilist_v[11] = 6;  ilist_v[12] = 1;  ilist_v[13] = 5;
	ilist_v[14] = 7;  ilist_v[15] = 3;  ilist_v[16] = 4;  ilist_v[17] = 0;
	gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, verts, ilist_v);
	color->set(1,1,1,1);
	ilist_c[0] = 0;
	gset->setAttr(PFGS_COLOR4, PFGS_OVERALL, color, ilist_c);
	return gset;
	}

static pfGeoSet * makeSphereGeoset(ygSphere * sphere,int style)
	{
	pfGeoSet *gset;
	pfVec4 *colorAttr;
	pfMatrix matrix;
	gset = pfdNewSphere(32,pfGetSharedArena());
	colorAttr = (pfVec4 *) pfMalloc(sizeof(pfVec4),pfGetSharedArena());
	colorAttr->set(1,1,1,1);
	gset->setAttr(PFGS_COLOR4,PFGS_OVERALL,colorAttr,NULL);
	gset->setGState(new pfGeoState);
	if (style == YG_STYLE_WIREFRAME)
		{
		gset->setLineWidth(1.0);
		gset->setDrawMode(PFGS_WIREFRAME,1);
		gset->getGState()->setMode(PFSTATE_ENWIREFRAME,1);
		}
	matrix.makeScale(sphere->radius,sphere->radius,sphere->radius);
	matrix.postTrans(matrix,sphere->center[0],sphere->center[1],
				sphere->center[2]);
	pfdXformGSet(gset,matrix);
	return gset;
	}


static pfGeoSet * makeCylinderGeoset(ygCylinder * cyl,int style)
	{
	pfGeoSet *gset;
	pfVec4 *colorAttr;
	pfVec3 upVec(0,0,1);
	pfMatrix matrix, rotMatrix;
	gset = pfdNewPipe(1.0, 1.0, 24,pfGetSharedArena());
	colorAttr = (pfVec4 *) pfMalloc(sizeof(pfVec4),pfGetSharedArena());
	colorAttr->set(1,1,1,1);
	gset->setAttr(PFGS_COLOR4,PFGS_OVERALL,colorAttr,NULL);
	gset->setGState(new pfGeoState);
	if (style == YG_STYLE_WIREFRAME)
		{
		gset->setLineWidth(1.0);
		gset->setDrawMode(PFGS_WIREFRAME,1);
		gset->getGState()->setMode(PFSTATE_ENWIREFRAME,1);
		}
	matrix.makeScale(cyl->radius,cyl->radius,cyl->halfLength);
	rotMatrix.makeVecRotVec(upVec, cyl->axis);
	matrix *= rotMatrix;
	matrix.postTrans(matrix, cyl->center[0], cyl->center[1],
				 cyl->center[2]);
	pfdXformGSet(gset,matrix);
	return gset;
	}
