// Description: aggregates the user related functions of wands, navigators, and trackers
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: user
//<li> all messages are passed on to the associated navigator
//<li> this node should a parent of any <a href="ygCAVEHead.html">ygCAVEHead</a> or <a href="ygCAVENavigator.html">ygCAVENavigator</a> nodes
//<li> this node is required for userTrigger and wandTrigger detection
//</ul>
//
// Category: User
// Author: Dave Pape
//           11/01/01
// Revision: 12/20/02 Alex Hill - added ygNode message filtering
//
#include <vector>
#include <Performer/pf/pfGroup.h>
#include <pfcave.h>
#include "ygNodeDB.h"
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "ygNavigator.h"
#include "ygUser.h"

using namespace std;

extern "C" ygNode* construct_ygUser(const char* name,bool master) { return new ygUser(name,master); }

struct _ygUserPrivateData
		{
		ygNavigator * navigator;
		ygNode * head;
		vector<ygNode *> wands;
		ygString netHead, netWands;
		ygString filterMessages;
		};


ygUser::ygUser(const char* name,bool master) : ygSimpleTransform(name,master)
	{
	ygString str;
	setClassName("ygUser",true);
	p_ = new struct _ygUserPrivateData;
	p_->head = NULL;
	p_->navigator = NULL;
	//distribute the name of the head node
	addNetKey("head", &p_->netHead, YG_NET_STRING);
	//distribute the names of all wand nodes
	addNetKey("wand", &p_->netWands, YG_NET_STRING);
	ygWorld::World->addUser(this);
	p_->filterMessages = " when addChild removeChild debug print reset event unreliableKey reliableKey resetTree position orientation updateInterval ";
	}


ygUser::~ygUser(void)
	{
	ygWorld::World->removeUser(this);
	delete p_;
	}


void ygUser::message(const ygMessage& msg)
	{
	ygString paddedMessage = " ";
	paddedMessage += msg.message;
	paddedMessage += " ";
	//hide all avatar geometry at the local site
	if (msg == "hideLocal")
		{
		if (msg.args.size() > 0)
			hideLocal(msg.boolArg(0));
		else
			hideLocal();
		}
	//pass all other messages on to the associated navigator
	else if (navigator() && !p_->filterMessages.contains(paddedMessage))
		navigator()->message(msg);
	else
		ygSimpleTransform::message(msg);
	}


void ygUser::hideLocal(bool hide)
	{
	if (hide)
		pfnode()->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF, PF_SET);
	else
		pfnode()->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF, PF_SET);
	}


void ygUser::app(void)
	{
	ygSimpleTransform::app();
	}


void ygUser::setHead(ygNode* head)
	{
	p_->head = head;
	storeHeadKey();
	}


void ygUser::addWand(ygNode* wand)
	{
	int i;
	for (i=0; i < p_->wands.size(); i++)
		if (p_->wands[i] == wand)
			return;
	p_->wands.push_back(wand);
	storeWandKey();
	}


void ygUser::setNavigator(ygNavigator *nav)
	{
	p_->navigator = nav;
	}


ygNavigator * ygUser::navigator(void) const
	{
	return p_->navigator;
	}


ygNode * ygUser::head(void) const
	{
	return p_->head;
	}


int ygUser::numWands(void) const
	{
	return p_->wands.size();
	}


ygNode * ygUser::wand(int i) const
	{
	if ((0 <= i) && (i < numWands()))
		return p_->wands[i];
	else
		return NULL;
	}


ygUser* ygUser::findUserAncestor(ygNode * node)
	{
	ygUser * user = NULL;
	while ((node) && (!user))
		{
		if (node->isOfClass("ygUser"))
			user = (ygUser*) node;
		else
			node = node->parent();
		}
	return user;
	}


/***  Net functions  ***/

void ygUser::acceptNetKey(const ygString& key)
	{
	if (key == "head")
		p_->head = ygNodeDB::find(p_->netHead);
	else if (key == "wand")
		{
		ygString *wandName;
		int i=0;
		p_->wands.clear();
		while (wandName = p_->netWands.nextToken(" ",&i))
			{
			ygNode * n = ygNodeDB::find(*wandName);
			if (n)
				p_->wands.push_back(n);
			delete wandName;
			}
		}
	else
		ygSimpleTransform::acceptNetKey(key);
	}


void ygUser::storeWandKey(void)
	{
	int i;
	p_->netWands.clear();
	for (i=0; i < numWands(); i++)
		{
		p_->netWands += wand(i)->name();
		p_->netWands += " ";
		}
	netKeyChanged("wand");
	}


void ygUser::storeHeadKey(void)
	{
	if (head())
		{
		p_->netHead = head()->name();
		netKeyChanged("head");
		}
	}
