// Description: creates a transformation matrix with position, orientation and scaling
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: transform
//<li> the size message accepts values for the X, Y, and Z axis to enable non-proportional scaling
//<li> creates a pfDCS and maintains state with a pfMatrix
//<li> the orientation message is measured in degrees and allows you to control the heading, pitch, and roll of the child node
//</ul>
//
// Category: Transformation
// Author: Dave Pape
// Revision: 11/01/01
//
#include <Performer/pf/pfDCS.h>
#include "ygNetKeys.h"
#include "ygTransform.h"

extern "C" ygNode* construct_ygTransform(const char* name,bool master) { return new ygTransform(name,master); }

#define MATRIX_NETKEY "m"


struct _ygTransformPrivateData
	{
	pfDCS * dcs;
	pfMatrix netMatrix;
	};


ygTransform::ygTransform(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("ygTransform",true);
	p_ = new struct _ygTransformPrivateData;
	//create new pfDCS and set as pfnode
	p_->dcs = new pfDCS;
	setPfNode(p_->dcs);
	p_->netMatrix.makeIdent();
	//create a full matrix network key
	addNetKey(MATRIX_NETKEY, &p_->netMatrix, YG_NET_MATRIX);
	}


ygTransform::~ygTransform(void)
	{
	delete p_;
	}


void ygTransform::reset(void)
	{
	//if net master then set matrix to identity
	if (isNetMaster())
		{
		pfMatrix mat;
		mat.makeIdent();
		setMatrix(mat);
		}
	ygNode::reset();
	}


/*****  DCS functions *****/

void ygTransform::getMatrix(pfMatrix& mat) const
	{
	p_->dcs->getMat(mat);
	}


void ygTransform::setMatrix(pfMatrix& mat)
	{
	p_->dcs->setMat(mat);
	storeMatrixKey();
	}


void ygTransform::setPosition(float x,float y,float z)
	{
	p_->dcs->setTrans(x,y,z);
	storeMatrixKey();
	}


void ygTransform::setOrientation(float x,float y,float z)
	{
	p_->dcs->setRot(z,x,y);
	storeMatrixKey();
	}


void ygTransform::setSize(float s)
	{
	p_->dcs->setScale(s);
	storeMatrixKey();
	}


void ygTransform::setSize(float x,float y,float z)
	{
	p_->dcs->setScale(x,y,z);
	storeMatrixKey();
	}


/***** Message functions *****/

void ygTransform::message(const ygMessage& msg)
	{
	//set the translational component
	if (msg == "position")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setPosition(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}	
	//set the rotational component applied before translation
	else if (msg == "orientation")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setOrientation(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}	
	//set the scaling component
	else if (msg == "size")
		{
		pfVec3 v;
		if (msg.args.size() == 1)
			setSize(msg.floatArg(0));
		else if (msg.getVec3Args(v))
			setSize(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}	
	//set the entire 4x4 matrix
	else if (msg == "matrix")
		{
		if (msg.args.size() == 16)
			{
			pfMatrix m;
			int i,j,argNum;
			for (i=0, argNum=0; i < 4; i++)
				for (j=0; j < 4; j++, argNum++)
					m[i][j] = msg.floatArg(argNum);
			setMatrix(m);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygNode::message(msg);
	}

/***** Net functions *****/

void ygTransform::acceptNetKey(const ygString& key)
	{
	if (key == MATRIX_NETKEY)
		setMatrix(p_->netMatrix);
	else
		ygNode::acceptNetKey(key);
	}


void ygTransform::storeMatrixKey(void)
	{
	if (isNetMaster())
		{
		getMatrix(p_->netMatrix);
		netKeyChanged(MATRIX_NETKEY);
		}
	}
