#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "ygString.h"
#include "ygUtil.h"
#include "ygTokenStream.h"

using namespace std;


#ifdef __sgi
#define CPP_COMMAND "/bin/cc -E"
#else
#define CPP_COMMAND "/usr/bin/cpp"
#endif


ygTokenStream::ygTokenStream(const ygString& filename)
	{
	ygString fullPath;
	if (ygFindFileInYgPath(filename,fullPath))
		{
		ygString cmd;
		noPreprocessor_ = false;
		if (getenv("YG_NOCPP_COMMAND"))
			noPreprocessor_ = true;
		else if (getenv("YG_CPP_COMMAND"))
			cmd = getenv("YG_CPP_COMMAND");
		else
			cmd = CPP_COMMAND;
		if (noPreprocessor_)
			{
			fp_ = fopen(fullPath.c_str(),"r");
			if (!fp_)
				{
				cerr << "ERROR: ygTokenStream could not open '"
					<< filename << "'\n";
				perror(fullPath.c_str());
				}
			}
		else
			{
			cmd += " ";
			cmd += fullPath;
			fp_ = popen(cmd.c_str(),"r");
			if (!fp_)
				{
				cerr << "ERROR: ygTokenStream failed to popen"
					" command '" << cmd << "'\n";
				perror(filename.c_str());
				}
			}
		}
	else
		{
		cerr << "ERROR: ygTokenStream could not find file '"
			<< filename << "'\n";
		fp_ = NULL;
		}
	filename_ = filename;
	linebufptr_ = NULL;
	lineNumber_ = 0;
	}


ygTokenStream::~ygTokenStream(void)
	{
	if (fp_)
		{
		if (noPreprocessor_)
			fclose(fp_);
		else
			pclose(fp_);
		}
	}


void ygTokenStream::pushback(const ygString& str)
	{
	ygString * s = new ygString(str);
	tokenStack_.push(s);
	}


bool ygTokenStream::next(ygString & str)
	{
	if (!tokenStack_.empty())
		{
		ygString *tmp = tokenStack_.top();
		tokenStack_.pop();
		str = *tmp;
		delete tmp;
		return true;
		}
	else
		{
		return readNextToken(str);
		}
	}


bool ygTokenStream::readNextToken(ygString& token)
	{
	int c;
	token.clear();
	/* Find the next non-whitespace character */
	c = nextChar();
	while ((c != -1) && ((c == ' ') || (c == '\t')))
		c = nextChar();
	/* If it's a brace, just return that */
	if ((c == '{') || (c == '}'))
		token += (char)c;
	/* If it's an open paren, return everything up to the close paren */
	else if (c == '(')
		{
		int parenDepth=1, quoting=0, startLine=lineNumber_;
		token += (char)c;
		while (parenDepth)
			{
			c = nextChar();
			if ((c == -1) ||
			    ((!quoting) && ((c == '{') || (c == '}'))))
				{
				cerr << "ERROR (ygTokenStream): unclosed "
				    "parenthesis in file " << filename_ 
				    << " line " << startLine << endl;
				if (c > -1)
					pushbackChar(c);
				return true;
				}
			if (c == '\\')
				{
				token += (char)c;
				c = nextChar();
				if (c == -1)
					{
					cerr << "ERROR (ygTokenStream): "
						"unclosed parenthesis in file "
						<< filename_ << endl;
					return true;
					}
				}
		/* Don't count any parentheses inside quoted strings */
			else if (c == '"')
				quoting = !quoting;
			else if ((!quoting) && (c == '('))
				parenDepth++;
			else if ((!quoting) && (c == ')'))
				parenDepth--;
			token += (char)c;
			}
		}
	/* Catch mismatched parentheses */
	else if (c == ')')
		{
		cerr << "ERROR: mismatched parentheses in " << filename_
			<< " line " << lineNumber_ << endl;
		return false;
		}
	/* For anything else, return up to the next paren, brace, or space */
	else if (c > -1)
		{
		while ((c > -1) && (!strchr("(){} \t",(char)c)))
			{
			token += (char)c;
			c = nextChar();
			}
		if (c > -1)
			pushbackChar(c);
		}
	else
		return false;
	return true;
	}


int ygTokenStream::nextChar(void)
	{
	if (!charStack_.empty())
		{
		int c = charStack_.top();
		charStack_.pop();
		return c;
		}
	else
		{
		int c;
		if ((!linebufptr_) || (!*linebufptr_))
			{
			if (!getNextLine())
				return -1;
			linebufptr_ = linebuf_;
			}
		c = *linebufptr_;
		linebufptr_++;
		return c;
		}
	}


void ygTokenStream::pushbackChar(int c)
	{
	charStack_.push(c);
	}


bool ygTokenStream::getNextLine(void)
	{
	if (!fp_)
		return false;
	while (fgets(linebuf_,sizeof(linebuf_)-1,fp_))
		{
		/* An initial # indicates it's a line & file note from cpp */
		if (linebuf_[0] == '#')
			{
			char tmpname[256];
			sscanf(linebuf_, "%*s %d \"%s", &lineNumber_, tmpname);
			filename_ = tmpname;
			if (filename_[filename_.length()-1] == '"')
				filename_.erase(filename_.length()-1);
			/* lineNumber_ will be incremented when we read the actual line */
			lineNumber_--;	
			}
		else
			{
			char *p;
			lineNumber_++;
			for (p=linebuf_; *p; p++)
				if (*p == '\n')
					*p = ' ';
			return true;
			}
		}
	fclose(fp_);
	fp_ = NULL;
	return false;
	}
