// Description: load a texture and place it on a polygon
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: texture
//<li> the initial polygon size is 1x1 oriented in the X-Y plane
//<li> image dimensions should be in powers of two: 32,64,128,256,512, or 1024
//<li> Currently the only format supported by Performer is:
//<b> .rgb</b> - SGI Iris RGB format
//<li> The imgcopy utility can be used to convert many formats to IRIS RGB format
//<ol> 
//<li> the imgcopy utility is located in /usr/sbin/
//<li> type imgcopy then the name of the file you want to convert, followed by the name of the converted file
//<li> <b>ex:</b> imgcopy oldfile.gif newfile.rgb
//<li> see the man pages on imgcopy for more information
//</ol>
//</ul>
//
// Category: Geometry
// Author: Alex Hill
//           11/01/01
//           10/15/02 Dave Pape - add optional cache flag to "file" msg, "reload" message, and included cache-state in netFilename
// Revision: 04/01/04 Alex Hill - corrected material properties to work with materialProperty node
//
#include <Performer/pf/pfChannel.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pr/pfTexture.h>
#include <Performer/pr/pfMaterial.h>
#include <ygPFTextureCache.h>
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "ygTexture.h"

extern "C" ygNode* construct_ygTexture(const char *name, bool master) { return new ygTexture(name, master); }

ygTexture::ygTexture(const char *name, bool master) : ygTransform(name, master)
	{
	// Set this instance's class type
	setClassName("ygTexture",true);
	gstate = NULL;
	geode = NULL;
	geoset = NULL;
	//Create geometry if not already created
	createGeometry();
	isectMask = YG_ISECT_ALLMASK;
	drawn = true;
	cache = true;
	show = true;
	addNetKey("file",&filename,YG_NET_STRING);
	addNetKey("isect",&isectMask,YG_NET_INT);
	addNetKey("drawn",&drawn,YG_NET_BOOL);
	}

ygTexture::~ygTexture(void)
	{
	}

void ygTexture::reset(void)
	{
	//reset intersection mask
	isectMask = YG_ISECT_ALLMASK;
	netKeyChanged("isect");
	//set drawn to true
	drawn = true;
	//set cache to true
	cache = true;
	//set show local to true
	show = true;
	//if a texture has been loaded
	if (gstate)
		{
		//reset intersection traversal mask
		geode->setTravMask(PFTRAV_ISECT, isectMask, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		//if drawn and show are true then allow drawing
		if (drawn && show)
			geode->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		else
			geode->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	ygTransform::reset();
	}


void ygTexture::message(const ygMessage& msg)
	{
	//load given texture file with optional caching
	if (msg == "file")
		{
		if (msg.args.size() == 1 || msg.args.size() == 2)
			{
			ygString file = msg.args[0];
			if (msg.args.size() == 2)
				{
				bool saveCache = cache;
				cache = msg.boolArg(1);
				setTexture(file);
				cache = saveCache;
				}
			else
				setTexture(file);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the dimensions of the texture
	else if (msg == "size")
		{
		if (msg.args.size() == 1)
			{
			float size = msg.floatArg(0);
			setSize(size,size,1.0);
			}
		else if (msg.args.size() == 2 || msg.args.size() == 3)
			{
			pfVec2 v;
			msg.getVec2Args(v);
			setSize(v[0],v[1],1.0);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}	
	//set collision detection status
	else if (msg == "wall")
		{
		if (msg.args.size() > 0)
			setWall(msg.boolArg(0));
		else
			setWall();
		}
	//set walk navigation status
	else if (msg == "floor")
		{
		if (msg.args.size() > 0)
			setFloor(msg.boolArg(0));
		else
			setFloor();
		}
	//set if the object is drawn
	else if (msg == "draw")
		{
		if (msg.args.size() > 0)
			setDraw(msg.boolArg(0));
		else
			setDraw();
		}
	//set whether the object is drawn locally
	else if (msg == "showLocal")
		{
		if (msg.args.size() > 0)
			showLocal(msg.boolArg(0));
		else
			showLocal();
		}
	//set the default texture caching mode
	else if (msg == "cache")
		{
		if (msg.args.size() > 0)
			cache = msg.boolArg(0);
		else
			cache = true;
		}
	//reload the currently loaded texture
	else if (msg == "reload")
		{
		/* This is slightly contorted just to avoid changing the class interface */
		bool saveCache = cache;
		cache = false;
		ygString *file;
		int index=0;
		file = filename.nextToken(" ",&index);
		setTexture(*file);
		cache = saveCache;
		}
    else
		ygTransform::message(msg);
	}
	
void ygTexture::setWall(bool val)
	{
	if (val)
		isectMask |= YG_ISECT_WALLMASK;
	else
		isectMask &= ~YG_ISECT_WALLMASK;
	netKeyChanged("isect");
	geode->setTravMask(PFTRAV_ISECT, isectMask, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	}

void ygTexture::setFloor(bool val)
	{
	if (val)
		isectMask |= YG_ISECT_FLOORMASK;
	else
		isectMask &= ~YG_ISECT_FLOORMASK;
	netKeyChanged("isect");
	geode->setTravMask(PFTRAV_ISECT, isectMask, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	}
	
void ygTexture::setDraw(bool val)
	{
	drawn = val;
	netKeyChanged("drawn");
	if (drawn && show)
		geode->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	else
		geode->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	}

void ygTexture::showLocal(bool val)
	{
	show = val;
	if (drawn && show)
		geode->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	else
		geode->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	}

void ygTexture::setTexture(const ygString& file)
	{
	pfTexture* texture;
	//if cache mode is true then retrieve from texture cache
	if (cache)
		texture = ygPFTextureCache::textureReference(file.c_str());
	//else, load the file using loadFile
	else
		{
		texture = new pfTexture;
		texture->loadFile(file.c_str());
		}	
	if (texture)
		{
		filename = file;
		gstate->setAttr(PFSTATE_TEXTURE, texture);
                int comp;
                //get number of image components
                texture->getImage(NULL,&comp,NULL,NULL,NULL);
		//if alpha channel then set transparency
		if (comp == 4)
			{
			gstate->setMode(PFSTATE_TRANSPARENCY,PF_ON|PFTR_NO_OCCLUDE|PFTR_BLEND_ALPHA);
			geoset->setDrawBin(PFSORT_TRANSP_BIN);
			}
		else
			{
			gstate->setMode(PFSTATE_TRANSPARENCY,PF_OFF);
			geoset->setDrawBin(PFSORT_OPAQUE_BIN);
			}
		//if drawn and show are true then allow drawing
		if (drawn && show)
			geode->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		else
			geode->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		//set intersection traversal mask
		geode->setTravMask(PFTRAV_ISECT, isectMask, PFTRAV_SELF|PFTRAV_DESCEND|PFTRAV_IS_CACHE, PF_SET);
		char cacheStr[32];
		sprintf(cacheStr," %d",cache);
		filename += cacheStr;
		netKeyChanged("file");
		setPFChanged();
		}
	}

void ygTexture::createGeometry(void)
	{		
	//create a new pfGeode and add it to the scene
	geode = new pfGeode;
	
	//allocate (from shared memory) an array of vertices for the polygon
	pfVec3 *vertex = (pfVec3 *) pfMalloc(4*sizeof(pfVec3),pfGetSharedArena());
	vertex[0].set( 0.5, -0.5, 0.0);
	vertex[1].set( 0.5, 0.5, 0.0);
	vertex[2].set(-0.5, 0.5, 0.0);
	vertex[3].set(-0.5, -0.5, 0.0);

	//create an array of normals for the polygon
	pfVec3 *normal = (pfVec3 *) pfMalloc(4*sizeof(pfVec3),pfGetSharedArena());
	normal[0].set(0.0, 0.0, 1.0);
	normal[1].set(0.0, 0.0, 1.0);
	normal[2].set(0.0, 0.0, 1.0);
	normal[3].set(0.0, 0.0, 1.0);

	//create an array of texture coordinates for the polygon
	pfVec2 *tCoord = (pfVec2 *) pfMalloc(4*sizeof(pfVec2),pfGetSharedArena());
	tCoord[0].set(1.0, 0.0);
	tCoord[1].set(1.0, 1.0);
	tCoord[2].set(0.0, 1.0);
	tCoord[3].set(0.0, 0.0);

	//create an array of color coordinates for the polygon
	pfVec4 *color = (pfVec4 *) pfMalloc(sizeof(pfVec4),pfGetSharedArena());
	color[0].set(1,1,1,1);

	//create the geometry set and add the geometry data to it
	geoset = new pfGeoSet;
	geoset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, vertex, NULL);
	geoset->setAttr(PFGS_NORMAL3, PFGS_PER_VERTEX, normal, NULL);
	geoset->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, tCoord, NULL);
	geoset->setAttr(PFGS_COLOR4, PFGS_PER_PRIM, color, NULL);
	geoset->setPrimType(PFGS_QUADS);
	geoset->setNumPrims(1);
	geode->addGSet(geoset);
	
	//Create a new material
	pfMaterial *material = new pfMaterial;
	material->setColorMode(PFMTL_BOTH,PFMTL_CMODE_DIFFUSE);
	
	//Create a pfGeostate, add the material, and then assign it to our geometry
	gstate = new pfGeoState;
	gstate->setAttr(PFSTATE_FRONTMTL, material);
	gstate->setAttr(PFSTATE_BACKMTL,  material);
	gstate->setMode(PFSTATE_CULLFACE, PF_OFF);
	gstate->setMode(PFSTATE_ENTEXTURE, PF_ON);
	gstate->setAttr(PFSTATE_TEXENV, new pfTexEnv);
	geoset->setGState(gstate);
	pfnode()->addChild(geode);
	setPFChanged();
	}
	
void ygTexture::acceptNetKey(const ygString& key)
	{
	if (key == "file")
		{
		ygString *file,*cacheStr;
		int index=0;
		file = filename.nextToken(" ",&index);
		cacheStr = filename.nextToken(" ",&index);
		if (cacheStr)
			cache = atoi(cacheStr->c_str());
		setTexture(*file);
		}
	else if (key == "drawn")
		setDraw(drawn);
	else if (key == "isect")
		{
		if (geode)
			geode->setTravMask(PFTRAV_ISECT, isectMask, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	else
		ygTransform::acceptNetKey(key);
	}
