// Description: toggles the child for node processing and Performer traversal
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: switch
//<li> creates a switch using a Performer pfSwitch
//<li> this node is frequently used to toggle geometry on and off
//</ul>
//
// Category: Selection
// Author: Dave Pape
// Revision: 11/01/01
//
#include <Performer/pf/pfSwitch.h>
#include "ygNetKeys.h"
#include "ygSwitch.h"

extern "C" ygNode* construct_ygSwitch(const char* name,bool master) { return new ygSwitch(name,master); }

struct _ygSwitchPrivateData
	{
	pfSwitch * sw;
	bool prevOn;
	bool netOn;
	};


ygSwitch::ygSwitch(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("ygSwitch",true);
	p_ = new struct _ygSwitchPrivateData;
	//create pfSwitch for the pfnode
	p_->sw = new pfSwitch;
	setPfNode(p_->sw);
	addNetKey("on", &p_->netOn, YG_NET_BOOL);
	reset();
	}


ygSwitch::~ygSwitch(void)
	{
	delete p_;
	}


void ygSwitch::reset(void)
	{
	//if net master then set the switch to on
	if (isNetMaster())
		{
		on();
		p_->prevOn = true;
		}
	ygNode::reset();
	}


void ygSwitch::app(void)
	{
	bool ison = isOn();
	if ((ison) && (!p_->prevOn))
		{
		//the switch has turned on
		eventOccurred("SwitchOn");
		}
	else if ((!ison) && (p_->prevOn))
		{
		//the switch has turned off
		eventOccurred("SwitchOff");
		}
	p_->prevOn = ison;
	ygNode::app();
	}


void ygSwitch::appTraverse(void)
	{
	//start timing
	timingBegin();
	//execute app method
	app();
	//if switch is on
	if (isOn())
		{
		//call appTraverse recursively on all children
		for (int i=0; i < numChildren(); i++)
			child(i)->appTraverse();
		}
	//end timing
	timingEnd();
	}


/*****  Switch functions *****/

bool ygSwitch::isOn(void) const
	{
	return (p_->sw->getVal() == PFSWITCH_ON);
	}


void ygSwitch::on(void)
	{
	if (!isOn())
		{
		p_->sw->setVal(PFSWITCH_ON);
		p_->netOn = true;
		netKeyChanged("on");
		}
	}


void ygSwitch::off(void)
	{
	if (isOn())
		{
		p_->sw->setVal(PFSWITCH_OFF);
		p_->netOn = false;
		netKeyChanged("on");
		}
	}


void ygSwitch::toggle(void)
	{
	if (isOn())
		off();
	else
		on();
	}


/***** Message functions *****/

void ygSwitch::message(const ygMessage& msg)
	{
	//set the switch to on
	if (msg == "on")
		on();
	//set the switch to off
	else if (msg == "off")
		off();
	//toggle the state of the switch
	else if (msg == "toggle")
		toggle();
	else
		ygNode::message(msg);
	}


/***** Net functions *****/

void ygSwitch::acceptNetKey(const ygString& key)
	{
	if (key == "on")
		{
		if (p_->netOn)
			on();
		else
			off();
		}
	else
		ygNode::acceptNetKey(key);
	}
