// Revision: 11/01/01 Dave Pape

#include <math.h>
#include <Performer/pf/pfGroup.h>
#include <Performer/pf/pfSCS.h>
#include <Performer/pf/pfSwitch.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pr/pfGeoMath.h>


static int sphereIsectTrav(pfNode *node,pfSphere *sphere,uint mask,pfMatrix xform);
static int isectGSet(pfGeoSet *gset,pfSphere *sphere);
static int isectGSetBBox(pfGeoSet *gset,pfSphere *sphere);
static int isectTristrips(pfGeoSet *gset,pfSphere *sphere);
static int isectTris(pfGeoSet *gset,pfSphere *sphere);
static int isectQuads(pfGeoSet *gset,pfSphere *sphere);
static int isectPolys(pfGeoSet *gset,pfSphere *sphere);
static int isectTriangle(pfSphere *sphere,pfVec3& v0,pfVec3& v1,pfVec3& v2);


int ygSphereIsect(pfNode *node,pfSphere *sphere,uint mask)
{
 pfMatrix mat;
 mat.makeIdent();
 return sphereIsectTrav(node,sphere,mask,mat);
}


static int sphereIsectTrav(pfNode *node,pfSphere *sphere,uint mask,pfMatrix xform)
{
 pfSphere boundSphere;
 int i;
 node->getBound(&boundSphere);
 boundSphere.orthoXform(sphere,xform);
 if (sphere->contains(&boundSphere) & PFIS_FALSE)
	return 0;
 if (node->isOfType(pfGeode::getClassType()))
	{
	pfGeode *geode = (pfGeode *)node;
	if (geode->getTravMask(PFTRAV_ISECT) & mask)
		{
		pfMatrix invXform;
		pfSphere localSphere;
		invXform.invertFull(xform);
		localSphere.orthoXform(sphere,invXform);
		for (i=0; i < geode->getNumGSets(); i++)
			{
			if (isectGSet(geode->getGSet(i),&localSphere))
				return 1;
			}
		}
	}
 else if (node->isOfType(pfSwitch::getClassType()))
	{
	int val = (int)((pfSwitch *)node)->getVal();
	pfGroup *group = (pfGroup *)node;
	if (val == PFSWITCH_ON)
		{
		for (i=0; i < group->getNumChildren(); i++)
			if (sphereIsectTrav(group->getChild(i),sphere,mask,xform))
				return 1;
		}
	else if (val != PFSWITCH_OFF)
		{
		if (sphereIsectTrav(group->getChild(val),sphere,mask,xform))
			return 1;
		}
	}
 else if (node->isOfType(pfGroup::getClassType()))
	{
	pfGroup *group = (pfGroup *)node;
	if (node->isOfType(pfSCS::getClassType()))
		{
		pfMatrix scsmat;
		((pfSCS *)node)->getMat(scsmat);
		xform = scsmat * xform;
		}
	for (i=0; i < group->getNumChildren(); i++)
		if (sphereIsectTrav(group->getChild(i),sphere,mask,xform))
			return 1;
	}
 return 0;
}


static int isectGSet(pfGeoSet *gset,pfSphere *sphere)
{
 if (isectGSetBBox(gset,sphere))
	{
	int primType = gset->getPrimType();
	if ((primType == PFGS_TRISTRIPS) || (primType == PFGS_FLAT_TRISTRIPS))
		return isectTristrips(gset,sphere);
	else if (primType == PFGS_TRIS)
		return isectTris(gset,sphere);
	else if (primType == PFGS_QUADS)
		return isectQuads(gset,sphere);
	else if (primType == PFGS_POLYS)
		return isectPolys(gset,sphere);
	}
 return 0;
}


static int isectTristrips(pfGeoSet *gset,pfSphere *sphere)
{
 int numPrims = gset->getNumPrims(), *primLengths = gset->getPrimLengths();
 int prim, tri, offset, madeIlist=0;
 int returnVal = 0;
 pfVec3 *verts;
 ushort *ilist=NULL;
 gset->getAttrLists(PFGS_COORD3,(void **)&verts,&ilist);
 if (!ilist)
	{
	int numVerts = gset->getAttrRange(PFGS_COORD3,NULL,NULL), i;
	ilist = (ushort *) pfMalloc(numVerts * sizeof(ushort));
	for (i = 0; i < numVerts; i++)
		ilist[i] = i;
	madeIlist = 1;
	}
 for (prim = 0, offset = 0; (prim < numPrims) && (!returnVal);
		offset += primLengths[prim], prim++)
	{
	for (tri=0; tri < primLengths[prim]-2; tri++)
		if (returnVal = isectTriangle(sphere,verts[ilist[offset+tri]],
				verts[ilist[offset+tri+1]],verts[ilist[offset+tri+2]]))
			break;
	}
 if (madeIlist)
	pfFree(ilist);
 return returnVal;
}


static int isectTris(pfGeoSet *gset,pfSphere *sphere)
{
 int numPrims = gset->getNumPrims();
 int prim, madeIlist=0;
 int returnVal = 0;
 pfVec3 *verts;
 ushort *ilist=NULL;
 gset->getAttrLists(PFGS_COORD3,(void **)&verts,&ilist);
 if (!ilist)
	{
	int numVerts = numPrims * 3, i;
	ilist = (ushort *) pfMalloc(numVerts * sizeof(ushort));
	for (i = 0; i < numVerts; i++)
		ilist[i] = i;
	madeIlist = 1;
	}
 for (prim = 0; (prim < numPrims) && (!returnVal); prim++)
	{
	if (returnVal = isectTriangle(sphere,verts[ilist[prim*3]],
				verts[ilist[prim*3+1]],verts[ilist[prim*3+2]]))
		break;
	}
 if (madeIlist)
	pfFree(ilist);
 return returnVal;
}


static int isectQuads(pfGeoSet *gset,pfSphere *sphere)
{
 int numPrims = gset->getNumPrims();
 int prim, madeIlist=0;
 int returnVal = 0;
 pfVec3 *verts;
 ushort *ilist=NULL;
 gset->getAttrLists(PFGS_COORD3,(void **)&verts,&ilist);
 if (!ilist)
	{
	int numVerts = numPrims * 4, i;
	ilist = (ushort *) pfMalloc(numVerts * sizeof(ushort));
	for (i = 0; i < numVerts; i++)
		ilist[i] = i;
	madeIlist = 1;
	}
 for (prim = 0; (prim < numPrims) && (!returnVal); prim++)
	{
	if (returnVal = isectTriangle(sphere,verts[ilist[prim*4]],
				verts[ilist[prim*4+1]],verts[ilist[prim*4+2]]))
		break;
	if (returnVal = isectTriangle(sphere,verts[ilist[prim*4+2]],
				verts[ilist[prim*4+3]],verts[ilist[prim*4]]))
		break;
	}
 if (madeIlist)
	pfFree(ilist);
 return returnVal;
}


static int isectPolys(pfGeoSet *gset,pfSphere *sphere)
{
 int numPrims = gset->getNumPrims(), *primLengths = gset->getPrimLengths();
 int prim, tri, offset, madeIlist=0;
 int returnVal = 0;
 pfVec3 *verts;
 ushort *ilist=NULL;
 gset->getAttrLists(PFGS_COORD3,(void **)&verts,&ilist);
 if (!ilist)
	{
	int numVerts = gset->getAttrRange(PFGS_COORD3,NULL,NULL), i;
	ilist = (ushort *) pfMalloc(numVerts * sizeof(ushort));
	for (i = 0; i < numVerts; i++)
		ilist[i] = i;
	madeIlist = 1;
	}
 for (prim = 0, offset = 0; (prim < numPrims) && (!returnVal);
		offset += primLengths[prim], prim++)
	{
	for (tri=0; tri < primLengths[prim]-2; tri++)
		if (returnVal = isectTriangle(sphere,verts[ilist[offset]],
				verts[ilist[offset+tri+1]],verts[ilist[offset+tri+2]]))
			break;
	}
 if (madeIlist)
	pfFree(ilist);
 return returnVal;
}


static int isectTriangle(pfSphere *sphere,pfVec3& v0,pfVec3& v1,pfVec3& v2)
{
 pfVec3 side1, side2, side3, norm, projCenter, crossProd;
 float normDotCenter, distance;
 side1 = v1 - v0;
 side2 = v2 - v0;
 norm.cross(side1,side2);
 norm.normalize();
 normDotCenter = norm.dot(sphere->center);
 distance = normDotCenter - norm.dot(v0);
/* Test distance to plane - if too great, fail */
 if (fabs(distance) > sphere->radius)
	return 0;
/* Project sphere center onto plane of triangle */
 projCenter = sphere->center - distance * norm;
/* Test if projected center is inside triangle - if so, succeed */
 crossProd.cross(side1,projCenter-v0);
 if (crossProd.dot(norm) > 0)
	{
	side3 = v2 - v1;
	crossProd.cross(side3,projCenter-v1);
	if (crossProd.dot(norm) > 0)
		{
		crossProd.cross(-side2,projCenter-v2);
		if (crossProd.dot(norm) > 0)
			return 1;
		}
	}
/* If not, test distances to edges */
 pfSeg edge;
 edge.makePts(v0,v1);
 if (sphere->isect(&edge,NULL,NULL) & PFIS_TRUE)
	return 1;
 edge.makePts(v1,v2);
 if (sphere->isect(&edge,NULL,NULL) & PFIS_TRUE)
	return 1;
 edge.makePts(v0,v2);
 if (sphere->isect(&edge,NULL,NULL) & PFIS_TRUE)
	return 1;
 return 0;
}


static int isectGSetBBox(pfGeoSet *gset,pfSphere *sphere)
{
 pfBox boundBox;
 float dx,dy,dz;
 gset->getBound(&boundBox);
 if (sphere->center[0] >= boundBox.min[0])
	if (sphere->center[0] <= boundBox.max[0])
		dx = 0;
	else
		dx = sphere->center[0] - boundBox.max[0];
 else
	dx = boundBox.min[0] - sphere->center[0];
 if (sphere->center[1] >= boundBox.min[1])
	if (sphere->center[1] <= boundBox.max[1])
		dy = 0;
	else
		dy = sphere->center[1] - boundBox.max[1];
 else
	dy = boundBox.min[1] - sphere->center[1];
 if (sphere->center[2] >= boundBox.min[2])
	if (sphere->center[2] <= boundBox.max[2])
		dz = 0;
	else
		dz = sphere->center[2] - boundBox.max[2];
 else
	dz = boundBox.min[2] - sphere->center[2];
 return (dx*dx + dy*dy + dz*dz <= sphere->radius*sphere->radius);
}
