// Description: defines a space that is rectangular, spherical, cylindrical, point or infinite in volume
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: space
//<li> a volume message defines a box, sphere, cylinder, point or infinite volume as follows:
//<ul>
//<li> volume(<a href="box.gif">box</a> X Y Z X Y Z) - two coordinate points define the extent of the box
//<li> volume(sphere X Y Z radius) - one coordinate point represents the origin of the sphere
//<li> volume(cylinder X Y Z X Y Z radius) - two coordinate points represent the origin of circles at the top and bottom of the cylinder
//<li> volume(point X Y Z) - one coordinate triplet defines the point
//<li> volume(infinite) - an infinite volume
//</ul>
//</ul>
//
// Category: Foundation
// Author: Dave Pape
// Revision: 11/01/01
//
#include <Performer/pf/pfGroup.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pfdu.h>
#include "ygUtil.h"
#include "ygNetKeys.h"
#include "ygInfiniteVolume.h"
#include "ygSpace.h"

using namespace std;

extern "C" ygNode* construct_ygSpace(const char* name,bool master) { return new ygSpace(name,master); }

struct _ygSpacePrivateData
	{
	bool debugVolume;
	pfGeode * debugGeode;
	ygString netString;
	};


ygSpace::ygSpace(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("ygSpace",true);
	p_ = new struct _ygSpacePrivateData;
	volume_ = new ygInfiniteVolume;
	p_->debugGeode = NULL;
	//show a wireframe of the volume
	debugFlag("volume", &p_->debugVolume);
	ygWriteVolumeMessage(volume_,p_->netString);
	//distribute the volume message
	addNetKey("volume", &p_->netString, YG_NET_STRING);
	}

ygSpace::~ygSpace(void)
	{
	if (volume_)
		delete volume_;
	}


void ygSpace::reset(void)
	{
	//set volume to infinite
	setVolume(new ygInfiniteVolume);
	ygNode::reset();
	}


void ygSpace::message(const ygMessage& msg)
	{
	//set the shape and size of the volume enclosed
	if (msg == "volume")
		setVolume(ygParseVolumeMessage(msg));
	else
		ygNode::message(msg);
	}


void ygSpace::setVolume(ygVolume * vol)
	{
	if (volume_)
		delete volume_;
	if (vol)
		volume_ = vol;
	else
		volume_ = new ygInfiniteVolume;
	if (p_->debugVolume)
		makeDebugOutline();
	ygWriteVolumeMessage(volume_,p_->netString);
	netKeyChanged("volume");
	}


void ygSpace::makeDebugOutline(void)
	{
	pfGeoSet * gset;
	if (p_->debugGeode)
		{
		gset = p_->debugGeode->getGSet(0);
		if (gset)
			{
			p_->debugGeode->removeGSet(gset);
#ifdef __sgi
			pfDelete(gset);
#endif
			}
		}
	else
		{
		p_->debugGeode = new pfGeode;
		pfnode()->addChild(p_->debugGeode);
		if (!p_->debugGeode)
			{
			cerr << "ERROR: ygSpace::makeDebugOutline: failed to"
				" create debugGeode\n";
			return;
			}
		}
	gset = ygCreateVolumeGeoset(volume_,YG_STYLE_WIREFRAME);
	if (gset)
		{
		p_->debugGeode->addGSet(gset);
		p_->debugGeode->setTravMask(PFTRAV_ISECT, 0, 
					PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	}


void ygSpace::acceptNetKey(const ygString& key)
	{
	if (key == "volume")
		{
		ygMessage msg;
		msg.parseString(p_->netString);
		setVolume(ygParseVolumeMessage(msg));
		}
	else
		ygNode::acceptNetKey(key);
	}
