// Description: creates a bergen sound server and manages sound clients
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node is the base class for most sound related nodes
//<li> the volume of a sound is the region in which the sound will play at 100% amplitude(loudness)
//<li> falloffDistance is a region outside the volume in which the amplitude is diminished as the user moves further away from the source(volume)
//<li> Common file formats supported by snerd:
//<b> .aiff</b> - Audio Interchange File Format
//<b> .wav</b> - MS RIFF WAVE Format
//<li> sound files should be saved using the <b>.aiff</b> format for best performance
//<ol> 
//<li> the sfconvert utility is located in /usr/sbin/
//<li> type sfconvert then the name of the file you want to convert, followed by the name of the converted file
//<li> <b>ex:</b> sfconvert oldfile.wav newfile.aif
//<li> see the man pages on sfconvert for more information
//</ol>
//<li> the sound server, snerd, must be run on the a local or remote machine
//<li> all sounds should be included in the YG_PATH environmental variable
//<li> sounds under a switch will turn off but must be restarted when the switch is toggled back on
//</ul>
//
// Category: Foundation
// Author: Alex Hill
// Revision: 08/01/03
//
#include <stdio.h>
#include <pfcave.h>
#include <bergenServer.h>
#include <vector>
#include "ygVolume.h"
#include "ygUtil.h"
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "ygSoundServer.h"

using namespace std;

extern "C" ygNode* construct_ygSoundServer(const char* name,bool master) { return new ygSoundServer(name,master); }

struct _ygSoundServerPrivateData
	{
	float falloffDistance;
	float amplitude;
	pfVec3 currPosition;
	unsigned int lastHeartbeat;
	};


struct _ygSoundServerGlobalData
	{
	bergenServer * server;
	vector<ygSoundServer*> sounds;
	unsigned int currentHeartbeat;
	float updateInterval, lastUpdate;
	};

struct _ygSoundServerGlobalData * ygSoundServer::global_ = NULL;



ygSoundServer::ygSoundServer(const char* name,bool master) : ygSpace(name,master)
	{
	setClassName("ygSoundServer",true);
	if (!global_)
		initServer();
	p_ = new struct _ygSoundServerPrivateData;
	p_->falloffDistance = 10;
	p_->amplitude = 1;
	addSound(this);
	addNetKey("ampl", &p_->amplitude, YG_NET_FLOAT);
	addNetKey("dist", &p_->falloffDistance, YG_NET_FLOAT);
	}

ygSoundServer::~ygSoundServer(void)
	{
	delete p_;
	}


void ygSoundServer::reset(void)
	{
	//set fall off distance to 10.0
	p_->falloffDistance = 10;
	//set amplitude to 1.0
	p_->amplitude = 1;
	ygSpace::reset();
	}


void ygSoundServer::message(const ygMessage& msg)
	{
	ygSpace::message(msg);
	}


void ygSoundServer::setFalloffDistance(float val)
	{
	p_->falloffDistance = val;
	netKeyChanged("dist");
	}


void ygSoundServer::setAmplitude(float val)
	{
	p_->amplitude = val;
	netKeyChanged("ampl");
	}


bool ygSoundServer::isPlaying(void)
	{
	return 0;
	}


void ygSoundServer::play(void)
	{
	return;
	}


void ygSoundServer::stop(bool sendStopCommand)
	{
	return;
	}


void ygSoundServer::heartbeat(void)
	{
	if (global_)
		p_->lastHeartbeat = global_->currentHeartbeat;
	if(p_->lastHeartbeat == 0)
		p_->lastHeartbeat = 1;
	}


void ygSoundServer::app(void)
	{
	ygSpace::app();
	}


float ygSoundServer::computeAmplitude(void)
	{
	float amp = p_->amplitude;
	float maxDist = p_->falloffDistance;
	ygNode * viewer = ygWorld::World->viewer();
	if ((maxDist >= 0) && (viewer))
		{
		viewer->app();
		pfVec3 userPos = viewer->origin(this);
		float distSq;
		if (volume())
			distSq = volume()->sqrDistance(userPos);
		else
			distSq = userPos.dot(userPos);
		if (distSq > maxDist*maxDist)
			amp = 0;
		else if (maxDist > 0)
			amp *= (1.0f - sqrtf(distSq) / maxDist);
		}
	return amp;
	}


void ygSoundServer::acceptNetKey(const ygString& key)
	{
	ygSpace::acceptNetKey(key);
	}


/****** Global functions for connection to server & heartbeat *******/

void ygSoundServer::initServer(void)
	{
	if (!global_)
		{
		global_ = new struct _ygSoundServerGlobalData;
		global_->server = new bergenServer;
		if (!getenv("BERGEN_PATH"))
			{
			if (getenv("YG_PATH"))
				global_->server->addPath(getenv("YG_PATH"));
			}
		global_->currentHeartbeat = 0;
		global_->updateInterval = 0.1;
		if (getenv("YG_SERVER_UPDATE_INTERVAL"))
			global_->updateInterval = atof(getenv("YG_SERVER_UPDATE_INTERVAL"));
		global_->lastUpdate = 0;
		}
	}


void ygSoundServer::addSound(ygSoundServer* s)
	{
	if (global_)
		global_->sounds.push_back(s);
	}


void ygSoundServer::removeSound(ygSoundServer* s)
	{
	if (global_)
		{
		vector<ygSoundServer*>::iterator iter;
		for (iter = global_->sounds.begin(); iter != global_->sounds.end();
		    ++iter)
			if (*iter == s)
				{
				global_->sounds.erase(iter);
				break;
				}
		}
	}


void ygSoundServer::updateServer(void)
	{
	vector<ygSoundServer*>::iterator iter;
	for (iter = global_->sounds.begin(); iter != global_->sounds.end(); ++iter)
		if (((*iter)->isPlaying()) &&
		    ((*iter)->p_->lastHeartbeat < global_->currentHeartbeat))
			(*iter)->stop();
	global_->currentHeartbeat++;
	//if update interval has passed then update sound server
	if ((ygWorld::FrameTime - global_->lastUpdate) > global_->updateInterval)
		{
		//get position relative to CAVE
		float* pos = new float[3];
		CAVEGetPosition(CAVE_HEAD,pos);
		global_->server->listener()->setPosition(pos[0],pos[2],-pos[1]);
		global_->lastUpdate = ygWorld::FrameTime;
		}
	}


void ygSoundServer::closeServer(void)
	{
	delete global_->server;
	delete global_;
	global_ = NULL;
	}


bergenServer * ygSoundServer::server(void)
	{
	return global_->server;
	}
