// Description: creates a lightweight transformation matrix with position and orientation
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: simpleTransform
//<li> the orientation message is measured in degrees and allows you to control the heading, pitch, and roll of the child node
//<li> creates a pfDCS and maintains state with a pfCoord
//<li> this node should be used when possible to reduce network traffic between multiple users
//</ul>
//
// Category: Transformation
// Author: Dave Pape
// Revision: 11/01/01
//
#include <Performer/pf/pfDCS.h>
#include "ygNetKeys.h"
#include "ygWorld.h"
#include "ygSimpleTransform.h"

extern "C" ygNode* construct_ygSimpleTransform(const char* name,bool master) { return new ygSimpleTransform(name,master); }

#define COORD_NETKEY "x"


struct _ygSimpleTransformPrivateData
	{
	pfDCS * dcs;
	pfCoord netCoord;
	float updateInterval, lastUpdate;
	};


ygSimpleTransform::ygSimpleTransform(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("ygSimpleTransform",true);
	p_ = new struct _ygSimpleTransformPrivateData;
	//create new pfDCS and set as pfnode
	p_->dcs = new pfDCS;
	setPfNode(p_->dcs);
	p_->netCoord.xyz.set(0,0,0);
	p_->netCoord.hpr.set(0,0,0);
	p_->updateInterval = 0;
	p_->lastUpdate = 0;
	//create a simple coordinate network key
	addNetKey(COORD_NETKEY, &p_->netCoord, YG_NET_COORD);
	//create an unreliable key for coordinate values
	unreliableKey(COORD_NETKEY);
	}


ygSimpleTransform::~ygSimpleTransform(void)
	{
	delete p_;
	}


void ygSimpleTransform::reset(void)
	{
	//if net master
	if (isNetMaster())
		{
		//set matrix to identity
		pfMatrix mat;
		mat.makeIdent();
		setMatrix(mat);
		}
	ygNode::reset();
	}


/*****  DCS functions *****/

void ygSimpleTransform::getMatrix(pfMatrix& mat) const
	{
	p_->dcs->getMat(mat);
	}


void ygSimpleTransform::setMatrix(pfMatrix& mat)
	{
	p_->dcs->setMat(mat);
	}


void ygSimpleTransform::setPosition(float x,float y,float z)
	{
	p_->dcs->setTrans(x,y,z);
	}


void ygSimpleTransform::setOrientation(float x,float y,float z)
	{
	p_->dcs->setRot(z,x,y);
	}


/***** Message functions *****/

void ygSimpleTransform::message(const ygMessage& msg)
	{
	//set the translational component
	if (msg == "position")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setPosition(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the rotational component applied before translation
	else if (msg == "orientation")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setOrientation(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}	
	//set the key update interval for remote users
	else if (msg == "updateInterval")
		{
		if (msg.args.size() > 0)
			p_->updateInterval = msg.floatArg(0);
		}
	else
		ygNode::message(msg);
	}

/***** Net functions *****/

void ygSimpleTransform::acceptNetKey(const ygString& key)
	{
	if (key == COORD_NETKEY)
		{
		setPosition(p_->netCoord.xyz);
		setOrientation(p_->netCoord.hpr[1],p_->netCoord.hpr[2],p_->netCoord.hpr[0]);
		}
	else
		ygNode::acceptNetKey(key);
	}


void ygSimpleTransform::app(void)
	{
	//if net master
	if (isNetMaster())
		{
		//if update interval has passed then update key
		if ((ygWorld::FrameTime - p_->lastUpdate) > p_->updateInterval)
			{
			pfMatrix matrix;
			pfCoord coord;
			getMatrix(matrix);
			matrix.getOrthoCoord(&coord);
			if ((coord.xyz != p_->netCoord.xyz) ||
			    (coord.hpr != p_->netCoord.hpr))
				{
				p_->netCoord = coord;
				netKeyChanged(COORD_NETKEY);
				p_->lastUpdate = ygWorld::FrameTime;
				}
			}
		}
	ygNode::app();
	}
