// Description: the base class from which all scripted Ygdrasil nodes are derived
//
//<b>notes:</b>
//<ul>
//<li> see reset method for default settings
//<li> this node can also be created with the aliases: node, ygGroup, and group
//<li> handles basic node functions such as maintaining child/parent pointers, handling events, and processing initial messages
//<li> this node can be used to group items in a scene
//</ul>
//
// Category: Foundation
// Author: Dave Pape
//           11/01/01
//           10/01/02 Alex Hill - added pfChanged and setPFChanged
// Revision: 10/10/02 Alex Hill - changed event message to take multiple arguments
//
#include <stdlib.h>
#include <unistd.h>
#include <Performer/pf/pfGroup.h>
#include <Performer/pf/pfSCS.h>
#include <iostream>
#include <vector>
#include <queue>
#include <map>
#include "ygNodeDB.h"
#include "ygNet.h"
#include "ygNetKeys.h"
#include "ygWorld.h"
#include "ygDebugFlags.h"
#include "ygEvent.h"
#include "ygEventMessages.h"
#include "ygUtil.h"
#include "ygNode.h"

using namespace std;


struct _ygNodePrivateData
	{
	ygString name;
	ygString className;
	ygString classDerivation;
	ygString netClassName;
	vector<ygMessage*> initMessages;
	ygEventMessages eventMessages;
	std::queue<ygEvent*> events;
	ygDebugFlags debug;
	vector<ygNode*> children;
	ygNode* parent;
	pfGroup* pfnode;
	bool isMaster;
	bool pfChange;
	bool debugUnknownKeys;
	bool debugNetMessages;
	ygNetKeys netKeys;
	ygString netChildren;
	ygString netMessage;
	int frameCount, eventCount;
	double frameStart, frameTimeTotal, appTimeTotal;
	};



extern "C" ygNode* construct_ygNode(const char* name,bool master)
	{ return new ygNode(name,master); }


ygNode::ygNode(const char* name,bool master,bool createPfNode)
	{
	p_ = new struct _ygNodePrivateData;
	p_->isMaster = master;
	setClassName("ygNode",true);
	p_->name = name;
	p_->eventMessages.setNode(this);
	p_->parent = NULL;
	//if createPfNode flag set then create a new group node
	if (createPfNode)
		setPfNode(new pfGroup);
	else
		setPfNode(NULL);
	//print out all unknown keys received
	debugFlag("UnknownKeys",&p_->debugUnknownKeys);
	//print out all messages sent over the network
	debugFlag("NetMessages",&p_->debugNetMessages);
	p_->netKeys.setNodeName(p_->name);
	//distribute list of children nodes
	addNetKey("children",&p_->netChildren,YG_NET_STRING);
	//distribute network messages
	addNetKey("message",&p_->netMessage,YG_NET_STRING);
	p_->frameCount = 0;
	p_->eventCount = 0;
	p_->frameTimeTotal = 0;
	p_->appTimeTotal = 0;
	p_->pfChange = false;
	}


ygNode::~ygNode(void)
	{
	vector<ygMessage*>::const_iterator iter;
	for (iter=p_->initMessages.begin(); iter != p_->initMessages.end();
	     ++iter)
		delete *iter;
	delete p_;
	}


void ygNode::reset(void)
	{
	//clear event messages
	p_->eventMessages.clear();
	}


void ygNode::sendInitMessages(void)
	{
	vector<ygMessage*>::const_iterator iter;
	for (iter=p_->initMessages.begin(); iter != p_->initMessages.end();
	     ++iter)
		ygWorld::World->scheduleMessage(**iter);
	}


void ygNode::message(const ygMessage& msg)
	{
	//configure messages to send in response to an event
	if (msg == "when")
		{
		int i;
		if (msg.args.size() < 2)
			msg.error(name(),"(wrong number of arguments)");
		ygString eventName;
		eventName = msg.args[0];
		for (i=1; i < msg.args.size(); i++)
			when(eventName, msg.args[i]);
		}
	//add the given node as a child
	else if (msg == "addChild")
		{
		if (msg.args.size() == 1)
			{
			ygNode * node = ygNodeDB::find(msg.args[0]);
			if (node)
				addChild(node);
			else
				cerr << "WARNING: " << name() << ".addChild("
					<< msg.args[0] << ") - could not find child node\n";
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//remove the given node as a child
	else if (msg == "removeChild")
		{
		if (msg.args.size() == 1)
			{
			ygNode * node = ygNodeDB::find(msg.args[0]);
			if (node)
				removeChild(node);
			else
				cerr << "WARNING: " << name() << ".removeChild("
					<< msg.args[0] << ") - could not find child node\n";
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//turn debuging on and off
	else if (msg == "debug")
		{
		if (msg.args.size() > 0)
			{
			bool val;
			ygString flag = msg.args[0];
			if (msg.args.size() > 1)
				val = msg.boolArg(1);
			else
				val = true;
			p_->debug.setFlag(flag,val);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//print the arguments to standard output
	else if (msg == "print")
		{
		int i;
		ygString data;
		for (i=0; i < msg.args.size(); i++)
			{
			data += msg.args[i];
			data += " ";
			}
		cout << name() << ": " << data << endl;
		}
	//reset and send initial messages
	else if (msg == "reset")
		{
		reset();
		sendInitMessages();
		}
	//generate an event with associated arguments
	else if (msg == "event")
		{
		if (msg.args.size() > 0)
			{
			ygString eventName = msg.args[0];
			if (msg.args.size() == 1)
				{
				//process event messages without arguments
				eventOccurred(eventName);
				}
			else
				{
				ygString args;
				for (int i=1;i<msg.args.size();i++)
					{
					args += msg.args[i];
					args += " ";
					}
				//process event messages with arguments
				eventOccurred(eventName,args);
				}
			}
		}
	//create an unreliable key
	else if (msg == "unreliablekey")
		{
		if (msg.args.size() > 0)
			unreliableKey(msg.args[0]);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//create a reliable key
	else if (msg == "reliablekey")
		{
		if (msg.args.size() > 0)
			reliableKey(msg.args[0]);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//reset the tree
	else if (msg == "resetTree")
		resetTree();
	else
		cout << "WARNING: " << name() << " received unknown message " << msg << endl;
	}


void ygNode::when(const ygString& event,const ygString& message)
	{
	p_->eventMessages.add(event,message);
	}


void ygNode::unreliableKey(const ygString& key)
	{
	p_->netKeys.unreliableKey(key);
	}


void ygNode::reliableKey(const ygString& key)
	{
	p_->netKeys.reliableKey(key);
	}


void ygNode::resetTree(void)
	{
	//reset this node
	reset();
	//send initial messages
	sendInitMessages();
	//call resetTree recursively on all children
	for (int i=0; i < numChildren(); i++)
		child(i)->resetTree();
	}


void ygNode::appTraverse(void)
	{
	//start timing
	timingBegin();
	//execute app method
	app();
	//call appTraverse recursively on all children
	for (int i=0; i < numChildren(); i++)
		child(i)->appTraverse();
	//end timing
	timingEnd();
	}


void ygNode::eventOccurred(const ygString& event,const ygString& args)
	{
	//add new event to events queue
	p_->events.push(new ygEvent(event,args));
	}


void ygNode::app(void)
	{
	//while events queue not empty
	while (!p_->events.empty())
		{
		ygEvent * event = p_->events.front();
		//respond to front event, pop, and delete
		p_->eventMessages.respondToEvent(*event);
		p_->events.pop();
		delete event;
		}
	ygNet::midframeProcess();
	timingAppDone();
	}


ygNode* ygNode::parent(void) const
	{
	return p_->parent;
	}


int ygNode::numChildren(void) const
	{
	return p_->children.size();
	}


ygNode * ygNode::child(int index) const
	{
	if ((index >= 0) && (index < p_->children.size()))
		return p_->children[index];
	else
		return NULL;
	}


void ygNode::addChild(ygNode *child,bool doLocal)
	{
	if (!child)
		{
		cerr << "ERROR (ygNode::addChild()): attempted to add a NULL"
			" pointer as a child of " << name() << endl;
		}
	//if net master or doLocal is set
	else if (isNetMaster() || doLocal)
		{
		//if child has a parent have it remove the child
		if (child->parent())
			child->parent()->removeChild(child,doLocal);
		//set the parent of the child to this node
		child->p_->parent = this;
		//bubble up the pfChange flag if necessary
		if (child->p_->pfChange)
			p_->pfChange = true;
		//push the child to the back of children vector
		p_->children.push_back(child);
		//establish parent/child relationship for associated pfnodes
		if ((pfnode()) && (child->pfnode()))
			pfnode()->addChild(child->pfnode());
		if (isNetMaster())
			updateChildrenKey();
		}
	//else, generate an addChild message and send
	else
		{
		ygString netmsg("addChild(");
		netmsg += child->name();
		netmsg += ")";
		sendNetMessage(netmsg);
		}
	}


void ygNode::removeChild(ygNode *child,bool doLocal)
	{
	//if net master or doLocal is set
	if (isNetMaster() || doLocal)
		{
		vector<ygNode*>::iterator i = p_->children.begin();
		//while searching children
		while ((i != p_->children.end()) && (*i != child))
			++i;
		//if child is found
		if (i != p_->children.end())
			{
			//remove child
			p_->children.erase(i);
			child->p_->parent = NULL;
			//undo parent/child relationship for associated pfnodes
			if ((pfnode()) && (child->pfnode()))
				pfnode()->removeChild(child->pfnode());
			if (isNetMaster())
				updateChildrenKey();
			}
		}
	//else, generate a removeChild message and send
	else
		{
		ygString netmsg("removeChild(");
		netmsg += child->name();
		netmsg += ")";
		sendNetMessage(netmsg);
		}
	}


void ygNode::createDummyName(ygString& name)
	{
	//construct a unique for this node
	static int count=0;
	static ygString basename;
	char str[256];
	if (count == 0)
		{
		if (getenv("YG_DUMMYNAME_BASE"))
			basename = getenv("YG_DUMMYNAME_BASE");
		else
			{
			if (gethostname(str,sizeof(str)) == 0)
				basename = str;
			else
				basename = "X";
			sprintf(str,"%x",getpid()%1000);
			basename += str;
			}
		ygMakeValidNodeName(basename);
		}
	name = basename;
	sprintf(str,"%x",count);
	name += str;
	count++;
	}


void ygNode::setClassName(const ygString& s,bool net)
	{
	p_->className = s;
	p_->classDerivation += ":";
	p_->classDerivation += s;
	p_->classDerivation += ":";
	p_->debug.checkFlags(s);
	if (net)
		p_->netClassName = s;
	}


const ygString& ygNode::className(void) const
	{
	return p_->className;
	}


const ygString& ygNode::classDerivation(void) const
	{
	return p_->classDerivation;
	}


bool ygNode::isOfClass(const ygString& className) const
	{
	ygString colonName(":");
	colonName += className;
	colonName += ":";
	return p_->classDerivation.contains(colonName);
	}


const ygString& ygNode::netClassName(void) const
	{
	return p_->netClassName;
	}


const ygString& ygNode::name(void) const
	{
	return p_->name;
	}


void ygNode::addInitMessage(const ygString& str)
	{
	ygMessage * msg = new ygMessage;
	msg->parseString(str,this);
	p_->initMessages.push_back(msg);
	}


void ygNode::setPfNode(pfGroup * n)
	{
	p_->pfnode = n;
	if (n)
		n->setName(name().c_str());
	}


void ygNode::setPFChanged(void)
	{
	p_->pfChange = true;
	if (p_->parent)
		p_->parent->setPFChanged();
	}


bool ygNode::pfChanged(void)
	{
	if (p_->pfChange)
		{
		p_->pfChange = false;
		return true;
		}
	else		
		return false;
	}


pfGroup* ygNode::pfnode(void) const
	{
	return p_->pfnode;
	}


void ygNode::debugFlag(const ygString& flagname, bool * var)
	{
	p_->debug.addFlag(flagname,var);
	p_->debug.checkFlag(name(),flagname);
	p_->debug.checkFlag(className(),flagname);
	}


pfVec3 ygNode::origin(const ygNode *node) const
	{
	pfVec3 origin(0,0,0), pos, retval;
	pfMatrix mat;
	getTransform(mat);
	pos.xformPt(origin,mat);
	//if node is given get position relative to the node
	if (node)
		{
		pfMatrix mat,invmat;
		node->getTransform(mat);
		invmat.invertAff(mat);
		retval.xformPt(pos,invmat);
		return retval;
		}
	//else, get absolute position of this node
	else
		return pos;
	}


void ygNode::getTransform(pfMatrix& mat) const
	{
	getParentTransform(mat);
	if (pfnode()->isOfType(pfSCS::getClassType()))
		{
		pfMatrix scsmat;
		((pfSCS *)pfnode())->getMat(scsmat);
		scsmat *= mat;
		mat = scsmat;
		}
	}


void ygNode::getParentTransform(pfMatrix& mat) const
	{
	pfMatrix scsmat;
	ygNode *node = parent();
	mat.makeIdent();
	while (node)
		{
		if (node->pfnode()->isOfType(pfSCS::getClassType()))
			{
			((pfSCS *)(node->pfnode()))->getMat(scsmat);
			mat *= scsmat;
			}
		node = node->parent();
		}
	}



/***** Network functions *****/

void ygNode::sendNetMessage(const ygString& msgstr)
	{
	p_->netMessage = msgstr;
	netKeyChanged("message",true);
	if (p_->debugNetMessages)
		cout << name() << " sending net message '" << msgstr << "'\n";
	}


void ygNode::sendNetMessage(const ygMessage& msg)
	{
	msg.writeString(p_->netMessage);
	netKeyChanged("message",true);
	if (p_->debugNetMessages)
		cout << name() << " sending net message '" << msg << "'\n";
	}


void ygNode::receiveNetKey(const ygString& key,void *data)
	{
	if (p_->netKeys.receiveKey(key,data))
		acceptNetKey(key);
	}


void ygNode::acceptNetKey(const ygString& key)
	{
	if (key == "children")
		{
		if (isNetMaster())
			{
			cerr << "WARNING: ygNode::acceptNetKey: " << name()
				<< " master node received a 'children' key\n";
			return;
			}
		removeAllChildren();
		ygString *childName;
		int i=0;
		while (childName = p_->netChildren.nextToken(" ",&i))
			{
			ygNode * n = ygNodeDB::find(*childName);
			if (n)
				addChild(n,true);
			delete childName;
			}
		}
	else if (key == "message")
		{
		if (isNetMaster())
			{
			if (p_->debugNetMessages)
				cout << name() << " got net message '"
					 << p_->netMessage << "'\n";
			ygMessage msg;
			msg.parseString(p_->netMessage,this);
			message(msg);
			}
		}
	else if (p_->debugUnknownKeys)
		cerr << "ygNode::acceptNetKey: received data for unknown key '"
			<< key << "'\n";
	}


/* removeAllChildren() is intended solely for use when we receive
   a new list of children from the network */
void ygNode::removeAllChildren(void)
	{
	vector<ygNode*>::iterator i = p_->children.begin();
	for ( ; (i != p_->children.end()); ++i)
		{
		(*i)->p_->parent = NULL;
		if ((pfnode()) && ((*i)->pfnode()))
			pfnode()->removeChild((*i)->pfnode());
		}
	p_->children.clear();
	}


void ygNode::updateChildrenKey(void)
	{
	int i;
	p_->netChildren.clear();
	for (i=0; i < numChildren(); i++)
		{
		p_->netChildren += child(i)->name();
		p_->netChildren += " ";
		}
	netKeyChanged("children");
	}


void ygNode::addNetKey(const ygString& keyname, void* data, int datatype)
	{
	p_->netKeys.addKey(keyname,data,datatype);
	}


void ygNode::netKeyChanged(const ygString& keyname,bool force)
	{
	if (isNetMaster() || force)
		p_->netKeys.storeKey(keyname);
	}


void ygNode::dontBroadcastNetKey(const ygString& keyname)
	{
	p_->netKeys.dontBroadcast(keyname);
	}


void ygNode::requestNetKeys(void) const
	{
	p_->netKeys.requestAllKeys();
	}


bool ygNode::isNetMaster(void) const
	{
	return p_->isMaster;
	}


void ygNode::timingBegin(void)
	{
	p_->frameStart = ygGetTimeD();
	}


void ygNode::timingAppDone(void)
	{
	p_->appTimeTotal += ygGetTimeD() - p_->frameStart;
	}


void ygNode::timingEnd(void)
	{
	p_->frameTimeTotal += ygGetTimeD() - p_->frameStart;
	p_->frameCount++;
	}


void ygNode::printTimingStats(int indent)
	{
	int i, count;
	for (i=0; i < indent; i++)
		cout << "  ";
	count = p_->frameCount;
	if (count < 1)
		count = 1;
	cout << className() << " " << name() << " : "
		<< p_->frameCount << " updates, "
		<< p_->eventCount << " events, "
		<< p_->frameTimeTotal*1000000.0/count << " usec/appTraverse, "
		<< p_->appTimeTotal*1000000.0/count << " usec/app\n";
	for (i=0; i < numChildren(); i++)
		child(i)->printTimingStats(indent+1);
	p_->frameCount = 0;
	p_->eventCount = 0;
	p_->frameTimeTotal = 0;
	p_->appTimeTotal = 0;
	}
