// Revision: 11/01/01 Dave Pape

#include <Performer/pr/pfLinMath.h>
#include <CAVERN.hxx>
#include <map>
#include "ygNet.h"
#include "ygNetKeys.h"

using namespace std;


struct _ygNetKeysData
	{
	void * data;
	int datatype;
	bool reliable;
	bool broadcast;
	};

struct _ygNetKeysPrivateData
	{
	ygString nodeName;
	CAVERNnet_datapack_c packer;
	size_t bufferSize;
	void * cavernBuffer;
	map<ygString,struct _ygNetKeysData*> keys;
	};


ygNetKeys::ygNetKeys(void)
	{
	p_ = new struct _ygNetKeysPrivateData;
	p_->bufferSize = 64;
	p_->cavernBuffer = malloc(p_->bufferSize);
	}


ygNetKeys::~ygNetKeys(void)
	{
	free(p_->cavernBuffer);
	delete p_;
	}


void ygNetKeys::setNodeName(const ygString& name)
	{
	p_->nodeName = name;
	}


void ygNetKeys::addKey(const ygString& name,void * data,int datatype)
	{
	if (p_->keys[name])
		{
		cerr << "ERROR (ygNetKeys::addKey): trying to add multiple keys"
			" named '" << name << "'\n";
		return;
		}
	struct _ygNetKeysData *d = new struct _ygNetKeysData;
	d->data = data;
	d->datatype = datatype;
	d->reliable = true;
	if ((datatype == YG_NET_MATRIX) || (datatype == YG_NET_COORD)
	    || (getenv("YG_NET_DEFAULT_UNRELIABLE")))
		d->reliable = false;
	d->broadcast = true;
	p_->keys[name] = d;
	}


void ygNetKeys::dontBroadcast(const ygString& keyname)
	{
	struct _ygNetKeysData *d = p_->keys[keyname];	
	if (d)
	    d->broadcast = false;
	}


void ygNetKeys::unreliableKey(const ygString& keyname)
	{
	struct _ygNetKeysData *d = p_->keys[keyname];	
	if (d)
	    d->reliable = false;
	}


void ygNetKeys::reliableKey(const ygString& keyname)
	{
	struct _ygNetKeysData *d = p_->keys[keyname];	
	if (d)
	    d->reliable = true;
	}


void ygNetKeys::storeKey(const ygString& keyname)
	{
	struct _ygNetKeysData *d = p_->keys[keyname];	
	if (d)
		{
		if (!d->broadcast)
			return;
		if (d->datatype == YG_NET_STRING)
			storeStringKey(keyname,d);
		else if (d->datatype == YG_NET_INT)
			storeIntKey(keyname,d);
		else if (d->datatype == YG_NET_FLOAT)
			storeFloatKey(keyname,d);
		else if (d->datatype == YG_NET_BOOL)
			storeBoolKey(keyname,d);
		else if (d->datatype == YG_NET_VEC2)
			storeVec2Key(keyname,d);
		else if (d->datatype == YG_NET_VEC3)
			storeVec3Key(keyname,d);
		else if (d->datatype == YG_NET_VEC4)
			storeVec4Key(keyname,d);
		else if (d->datatype == YG_NET_MATRIX)
			storeMatrixKey(keyname,d);
		else if (d->datatype == YG_NET_COORD)
			storeCoordKey(keyname,d);
		else
			cerr << "ERROR (ygNetKeys::storeKey): " << p_->nodeName
				<< "'s key '" << keyname << "' is of an unsupported type\n";
		}
	else
		cerr << "ERROR (ygNetKeys::storeKey): " << p_->nodeName
				<< " tried to store unknown key '" << keyname << "'\n";
	}


void ygNetKeys::storeStringKey(const ygString& keyname,struct _ygNetKeysData *d)
	{
	ygString * str = (ygString*) (d->data);
	ygNet::storeKey(p_->nodeName, keyname.c_str(), *str, d->reliable);
	}


void ygNetKeys::storeIntKey(const ygString& keyname,struct _ygNetKeysData *d)
	{
	int * val = (int*) (d->data);
	size_t size = sizeof(int);
	p_->packer.initPack((char*)p_->cavernBuffer, size);
	p_->packer.packInt(*val);
	ygNet::storeKey(p_->nodeName, keyname.c_str(), p_->cavernBuffer, size, d->reliable);
	}


void ygNetKeys::storeFloatKey(const ygString& keyname,struct _ygNetKeysData *d)
	{
	float * val = (float*) (d->data);
	size_t size = sizeof(float);
	p_->packer.initPack((char*)p_->cavernBuffer, size);
	p_->packer.packFloat(*val);
	ygNet::storeKey(p_->nodeName, keyname.c_str(), p_->cavernBuffer, size, d->reliable);
	}


void ygNetKeys::storeBoolKey(const ygString& keyname,struct _ygNetKeysData *d)
	{
	bool * val = (bool*) (d->data);
	if (*val)
		ygNet::storeKey(p_->nodeName, keyname.c_str(), "1", 1, d->reliable);
	else
		ygNet::storeKey(p_->nodeName, keyname.c_str(), "0", 1, d->reliable);
	}


void ygNetKeys::storeVec2Key(const ygString& keyname,struct _ygNetKeysData *d)
	{
	pfVec2 * val = (pfVec2*) (d->data);
	size_t size = 2*sizeof(float);
	p_->packer.initPack((char*)p_->cavernBuffer, size);
	p_->packer.packFloat((*val)[0]);
	p_->packer.packFloat((*val)[1]);
	ygNet::storeKey(p_->nodeName, keyname.c_str(), p_->cavernBuffer, size, d->reliable);
	}


void ygNetKeys::storeVec3Key(const ygString& keyname,struct _ygNetKeysData *d)
	{
	pfVec3 * val = (pfVec3*) (d->data);
	size_t size = 3*sizeof(float);
	p_->packer.initPack((char*)p_->cavernBuffer, size);
	p_->packer.packFloat((*val)[0]);
	p_->packer.packFloat((*val)[1]);
	p_->packer.packFloat((*val)[2]);
	ygNet::storeKey(p_->nodeName, keyname.c_str(), p_->cavernBuffer, size, d->reliable);
	}


void ygNetKeys::storeVec4Key(const ygString& keyname,struct _ygNetKeysData *d)
	{
	pfVec4 * val = (pfVec4*) (d->data);
	size_t size = 4*sizeof(float);
	p_->packer.initPack((char*)p_->cavernBuffer, size);
	p_->packer.packFloat((*val)[0]);
	p_->packer.packFloat((*val)[1]);
	p_->packer.packFloat((*val)[2]);
	p_->packer.packFloat((*val)[3]);
	ygNet::storeKey(p_->nodeName, keyname.c_str(), p_->cavernBuffer, size, d->reliable);
	}


void ygNetKeys::storeMatrixKey(const ygString& keyname,struct _ygNetKeysData *d)
	{
	pfMatrix * val = (pfMatrix*) (d->data);
	size_t size = 16*sizeof(float);
	int i,j;
	p_->packer.initPack((char*)p_->cavernBuffer, size);
	for (i=0; i<4; i++)
		for (j=0; j<4; j++)
			p_->packer.packFloat((*val)[i][j]);
	ygNet::storeKey(p_->nodeName, keyname.c_str(), p_->cavernBuffer, size, d->reliable);
	}


void ygNetKeys::storeCoordKey(const ygString& keyname,struct _ygNetKeysData *d)
	{
	pfCoord * val = (pfCoord*) (d->data);
	size_t size = 6*sizeof(float);
	p_->packer.initPack((char*)p_->cavernBuffer, size);
	p_->packer.packFloatArray(val->xyz.vec,3);
	p_->packer.packFloatArray(val->hpr.vec,3);
	ygNet::storeKey(p_->nodeName, keyname.c_str(), p_->cavernBuffer, size, d->reliable);
	}


void ygNetKeys::requestAllKeys(void)
	{
	map<ygString,struct _ygNetKeysData*>::const_iterator iter;
	for (iter = p_->keys.begin(); iter != p_->keys.end(); ++iter)
		ygNet::requestKey(p_->nodeName,iter->first.c_str());
	}


bool ygNetKeys::receiveKey(const ygString& keyname,void *data)
	{
	struct _ygNetKeysData *d = p_->keys[keyname];	
	if (d)
		{
		if (d->datatype == YG_NET_STRING)
			receiveStringKey(keyname,d,data);
		else if (d->datatype == YG_NET_INT)
			receiveIntKey(keyname,d,data);
		else if (d->datatype == YG_NET_FLOAT)
			receiveFloatKey(keyname,d,data);
		else if (d->datatype == YG_NET_BOOL)
			receiveBoolKey(keyname,d,data);
		else if (d->datatype == YG_NET_VEC2)
			receiveVec2Key(keyname,d,data);
		else if (d->datatype == YG_NET_VEC3)
			receiveVec3Key(keyname,d,data);
		else if (d->datatype == YG_NET_VEC4)
			receiveVec4Key(keyname,d,data);
		else if (d->datatype == YG_NET_MATRIX)
			receiveMatrixKey(keyname,d,data);
		else if (d->datatype == YG_NET_COORD)
			receiveCoordKey(keyname,d,data);
		else
			{
			cerr << "ERROR (ygNetKeys::receiveData): " << p_->nodeName
				<< "'s key '" << keyname << "' is of an unsupported type\n";
			return false;
			}
		}
	else
		return false;
	return true;
	}


void ygNetKeys::receiveStringKey(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	ygString * str = (ygString*) (d->data);
	*str = (char*) newdata;
	}


void ygNetKeys::receiveIntKey(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	int * val = (int*) (d->data);
	size_t size = sizeof(int);
	p_->packer.initUnpack((char*) newdata, size);
	p_->packer.unpackInt(val);
	}


void ygNetKeys::receiveFloatKey(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	float * val = (float*) (d->data);
	size_t size = sizeof(float);
	p_->packer.initUnpack((char*) newdata, size);
	p_->packer.unpackFloat(val);
	}


void ygNetKeys::receiveBoolKey(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	bool * val = (bool*) (d->data);
	if (*((char*)newdata) == '1')
		*val = true;
	else
		*val = false;
	}


void ygNetKeys::receiveVec2Key(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	pfVec2 * val = (pfVec2*) (d->data);
	size_t size = 2*sizeof(float);
	p_->packer.initUnpack((char*) newdata, size);
	p_->packer.unpackFloat(&val->vec[0]);
	p_->packer.unpackFloat(&val->vec[1]);
	}


void ygNetKeys::receiveVec3Key(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	pfVec3 * val = (pfVec3*) (d->data);
	size_t size = 3*sizeof(float);
	p_->packer.initUnpack((char*) newdata, size);
	p_->packer.unpackFloat(&val->vec[0]);
	p_->packer.unpackFloat(&val->vec[1]);
	p_->packer.unpackFloat(&val->vec[2]);
	}


void ygNetKeys::receiveVec4Key(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	pfVec4 * val = (pfVec4*) (d->data);
	size_t size = 4*sizeof(float);
	p_->packer.initUnpack((char*) newdata, size);
	p_->packer.unpackFloat(&val->vec[0]);
	p_->packer.unpackFloat(&val->vec[1]);
	p_->packer.unpackFloat(&val->vec[2]);
	p_->packer.unpackFloat(&val->vec[3]);
	}


void ygNetKeys::receiveMatrixKey(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	pfMatrix * val = (pfMatrix*) (d->data);
	size_t size = 16*sizeof(float);
	int i,j;
	p_->packer.initUnpack((char*) newdata, size);
	for (i=0; i<4; i++)
		for (j=0; j<4; j++)
			p_->packer.unpackFloat(&val->mat[i][j]);
	}


void ygNetKeys::receiveCoordKey(const ygString& keyname,struct _ygNetKeysData *d,void *newdata)
	{
	pfCoord * val = (pfCoord*) (d->data);
	size_t size = 6*sizeof(float);
	p_->packer.initUnpack((char*) newdata, size);
	p_->packer.unpackFloatArray(&val->xyz[0],3);
	p_->packer.unpackFloatArray(&val->hpr[0],3);
	}
