// Revision: 11/01/01 Dave Pape

#include <stdlib.h>
#include <CAVERN.hxx>
#include <list>
#include "ygNetClient.h"
#include "ygMutex.h"
#include "ygUtil.h"
#include "ygNode.h"
#include "ygNodeDB.h"
#include "ygDebugFlags.h"
#include "ygFileServer.h"
#include "ygNet.h"

using namespace std;

#define cvrnPrintf printf

typedef enum {
	STORE_REQUEST,
	STORE_UNRELIABLE_REQUEST,
	FETCH_REQUEST
	} ygNetRequestType;


struct _ygNetRequest
	{
	ygNetRequestType reqType;
	ygString path;
	ygString key;
	void* data;
	size_t dataSize;
	};


struct _ygNewKeyData
	{
	ygNode* node;
	ygString key;
	void* data;
	};


struct _ygNetPrivateData
	{
	bool netActive;
	bool threaded;
	char * server;
	int port;
	ygNetClient netClient;
	ygMutex newKeyDataMutex;
	list<struct _ygNewKeyData*> newKeyData;
	ygMutex netRequestsMutex;
	list<struct _ygNetRequest*> netRequests;
	bool debugNodes, debugData;
	bool doReset, doFlush;
	double lastProcessTime, processInterval;
	};

struct _ygNetPrivateData* ygNet::p_ = NULL;



void ygNet::init(void)
	{
	CAVERNinit();
	if (!p_)
		p_ = new struct _ygNetPrivateData;
	p_->netActive = false;
	p_->debugNodes = false;
	p_->debugData = false;
#ifdef __sgi
	p_->threaded = false;
#else
	p_->threaded = true;
#endif
	p_->server = NULL;
	p_->port = 3500;
	p_->doReset = false;
	p_->doFlush = false;
	p_->lastProcessTime = 0;
	p_->processInterval = 0;
	if (getenv("YG_NET_PROCESS_INTERVAL"))
		p_->processInterval = atof(getenv("YG_NET_PROCESS_INTERVAL"));
	if (getenv("YG_NET_NOTHREAD"))
		p_->threaded = false;
	if (getenv("YG_NET_THREAD"))
		{
		if ((strlen(getenv("YG_NET_THREAD")) == 0) ||
		    (atoi(getenv("YG_NET_THREAD")) == 0))
			p_->threaded = true;
		else if (atoi(getenv("YG_NET_THREAD")) == 1)
			p_->threaded = true;
		else
			cerr << "WARNING: environment variable YG_NET_THREAD has"
				" the value '" << getenv("YG_NET_THREAD") <<
				"'; it should be either 0 or 1\n";
		}
	if (getenv("YG_NET_SERVER"))
		p_->server = strdup(getenv("YG_NET_SERVER"));
	if (getenv("YG_NET_PORT"))
		p_->port = atoi(getenv("YG_NET_PORT"));
	if (p_->server)
		{
		p_->debugNodes = ygDebugFlags::checkDebugEnv("net.nodes");
		p_->debugData = ygDebugFlags::checkDebugEnv("net.data");
		p_->netActive = true;
		p_->newKeyDataMutex.unlock();
		p_->netRequestsMutex.unlock();
		p_->netClient.init(p_->server, p_->port);
		if (p_->threaded)
			{
			if (!ygCreateThread(ygNet::netProcess, NULL))
				{
				cerr << "ERROR: ygNet::init: failed to start "
				     "network thread; will run non-threaded\n";
				p_->threaded = false;
				}
			}
		if (!p_->threaded)
			{
			p_->netClient.trigger(ygNet::receiveData, NULL);
			}
		}
	}


void ygNet::reset(void)
	{
	p_->doReset = true;
	}


void ygNet::showStats(void)
	{
	p_->netClient.showStats();
	}


void ygNet::halt(void)
	{
	ygFileServer::halt();
	if (p_->netActive)
		{
		p_->netActive = false;
		usleep(100000);	// Try to let netProcess() return
		}
	}


ygNode* ygNet::connectNewNode(const ygString& name)
	{
	if (p_->netActive)
		{
		ygNode * newNode = NULL;
		char * typeData;
		int size;
		if (p_->debugNodes)
			cvrnPrintf("ygNet::connectNewNode(%s)\n",name.c_str());
		typeData = getKey(name.c_str(), "type", &size);
		if (p_->debugNodes)
			cvrnPrintf("ygNet::connectNewNode(%s) got type='%s'\n",name.c_str(),typeData);
		if (typeData)
			{
			newNode = ygNodeDB::createProxy(typeData,name);
			delete typeData;
			}
		else
			cerr << "ygNet::connectNewNode: failed to find node "
				<< name << "'s 'type' key\n";
		return newNode;
		}
	else
		return NULL;
	}


char* ygNet::getKey(const char* path, const char* key, int *size, float timeout)
	{
	if (p_->netActive)
		{
		float t = ygGetTime();
		int iteration=0;
		void* val = p_->netClient.get(path, key, (size_t*)size);
		if (val)
			return (char*)val;
		requestKey(path,key);
		while ((val == NULL) && ((ygGetTime() - t) < timeout))
			{
			if (!p_->threaded)
				{
				p_->netClient.process();
				netProcessRequests();
				}
			val = p_->netClient.get(path, key, (size_t*)size);
			if (!val)
				{
				usleep(10000);
				if (iteration % 10 == 0)
					requestKey(path,key);
				iteration++;
				}
			}
		return (char*)val;
		}
	else
		return NULL;
	}


/*** Request data for a key - called by ygNetKeys::requestAllKeys.
    Originally, this would just queue up a Fetch request to be sent
    out to the network server.  With the 0.1.4 design, all (theoretically)
    of the database is downloaded in advance, so we can just look up
    the data and pass it immediately to receiveNetKey().  If the
    data doesn't exist locally, we still queue up a request, but
    since fetches are currently ignored by the ygNetClient, that
    doesn't really accomplish much - this should be revisited and cleaned
    up once things settle. ***/
void ygNet::requestKey(const ygString& name,const char* key)
	{
	size_t size;
	void *data = p_->netClient.get(name, key, &size);
	if (data)
		{
		ygNode *node = ygNodeDB::find(name,false);
		if (node)
			node->receiveNetKey(key, data);
		free(data);
		}
	else if (p_->netActive)
		{
		struct _ygNetRequest * req = new struct _ygNetRequest;
		req->reqType = FETCH_REQUEST;
		req->path = name;
		req->key = key;
		req->dataSize = 0;
		req->data = NULL;
		pushNetRequest(req);
		if (p_->debugNodes)
			cvrnPrintf("ygNet::requestKey(%s,%s)\n",name.c_str(),key);
		}
	}


void ygNet::storeNode(ygNode *node)
	{
	storeKey(node->name(), "type", node->netClassName(), true);
	if (p_->debugNodes)
		cvrnPrintf("ygNet::storeNode(%s), type=%s\n",node->name().c_str(),node->netClassName().c_str());
	}


void ygNet::storeKey(const ygString& name, const char* key,
			const ygString& val, bool reliable)
	{
	storeKey(name, key, val.c_str(), val.length()+1, reliable);
	}


void ygNet::storeKey(const ygString& name, const char* key,
			const void* val, size_t size, bool reliable)
	{
	if ((p_->netActive) && (p_->threaded))
		{
		struct _ygNetRequest * req = new struct _ygNetRequest;
		if (reliable)
			req->reqType = STORE_REQUEST;
		else
			req->reqType = STORE_UNRELIABLE_REQUEST;
		req->path = name;
		req->key = key;
		req->dataSize = size;
		req->data = malloc(size);
		memcpy(req->data, val, size);
		pushNetRequest(req);
		if (p_->debugData)
			cvrnPrintf("ygNet::storeKey(%s,%s,[%d bytes])\n",name.c_str(),key,size);
		}
	else if (p_->netActive)
		{
		if (reliable)
			{
			p_->netClient.put(name, key, (void*)val, size);
			if (p_->debugData)
				cvrnPrintf("ygNet::storeKey storing %s %s\n",name.c_str(),key);
			}
		else
			{
			p_->netClient.putUnreliably(name, key, (void*)val, size);
			if (p_->debugData)
				cvrnPrintf("ygNet::storeKey storing %s %s unreliably\n",name.c_str(),key);
			}
		}
	}


/* This function is to be called by ygNode::app(), to allow ygNet::process() to
  be called occasionally in the middle of a frame.  This was added for high-bandwidth
  apps running in non-threaded mode - if too much data arrives during the course of
  a frame's app traversal, the socket's buffer will overflow & lose data; this will
  hopefully get around that problem. */

void ygNet::midframeProcess(void)
	{
	if ((p_->netActive) && (p_->processInterval > 0))
		{
		double t = ygGetTimeD();
		if (t - p_->lastProcessTime > p_->processInterval)
			process();
		}
	}


void ygNet::process(void)
	{
	if (p_->netActive)
		{
		bool loop=true;
		while (loop)
			{
			if (!p_->threaded)
				{
				if (p_->doReset)
					{
					p_->netClient.reset();
					p_->doReset = false;
					}
				p_->netClient.process();
				netProcessRequests();
				}
			struct _ygNewKeyData * newData = NULL;
			p_->newKeyDataMutex.lock();
			loop = !p_->newKeyData.empty();
			if (loop)
				{
				newData = p_->newKeyData.front();
				p_->newKeyData.pop_front();
				}
			p_->newKeyDataMutex.unlock();
			if (newData)
				{
				newData->node->receiveNetKey(newData->key,
								newData->data);
				free(newData->data);
				delete newData;
				}
			}
		if (p_->threaded)
			p_->doFlush = true;
		else
			p_->netClient.flushWrites();
		}
	p_->lastProcessTime = ygGetTimeD();
	}

void * ygNet::netProcess(void*)
	{
	p_->netClient.trigger(ygNet::receiveData, NULL);
	while (p_->netActive)
		{
		if (p_->doReset)
			{
			p_->netClient.reset();
			p_->doReset = false;
			}
		p_->netClient.process();
		netProcessRequests();
		if (p_->doFlush)
			{
			p_->doFlush = false;
			p_->netClient.flushWrites();
			}
		}
	return NULL;
	}


void ygNet::receiveData(char* path, char* key, void*)
	{
	if (p_->debugData)
		cvrnPrintf("ygNet::receiveData(%s,%s)\n",path,key);
	/* Ignore "type" keys; connectNewNode will catch them via getKey() */
	if (strcmp(key,"type") == 0)
		return;
	ygNode *node = ygNodeDB::find(path,false);
	if (node)
		{
		size_t size;
		void *data = p_->netClient.get(path,key,&size);
		if (!data)
			return;
		struct _ygNewKeyData * newData = new struct _ygNewKeyData;
		newData->node = node;
		newData->key = key;
		newData->data = data;
		p_->newKeyDataMutex.lock();
		p_->newKeyData.push_back(newData);
		p_->newKeyDataMutex.unlock();
		}
	}


void ygNet::netProcessRequests(void)
	{
	bool loop=true;
	while (loop)
		{
		struct _ygNetRequest * req = popNetRequest();
		if (req)
			{
			if (req->reqType == STORE_REQUEST)
				{
				p_->netClient.put(req->path, req->key,
						req->data, req->dataSize);
				if (p_->debugData)
					cvrnPrintf("ygNet::netProcessRequests storing %s %s\n",req->path.c_str(),req->key.c_str());
				}
			else if (req->reqType == STORE_UNRELIABLE_REQUEST)
				{
				p_->netClient.putUnreliably(req->path, req->key,
						req->data, req->dataSize);
				if (p_->debugData)
					cvrnPrintf("ygNet::netProcessRequests storing %s %s unreliably\n",req->path.c_str(),req->key.c_str());
				}
			else if (req->reqType == FETCH_REQUEST)
				{
				p_->netClient.fetch(req->path,req->key);
				if (p_->debugData)
					cvrnPrintf("ygNet::netProcessRequests fetching %s %s\n",req->path.c_str(),req->key.c_str());
				}
			if (req->data)
				free(req->data);
			delete req;
			}
		else
			loop = false;
		}
	}


void ygNet::pushNetRequest(struct _ygNetRequest * req)
	{
	p_->netRequestsMutex.lock();
	p_->netRequests.push_back(req);
	p_->netRequestsMutex.unlock();
	}


struct _ygNetRequest * ygNet::popNetRequest(void)
	{
	struct _ygNetRequest * req = NULL;
	p_->netRequestsMutex.lock();
	if (!p_->netRequests.empty())
		{
		req = p_->netRequests.front();
		p_->netRequests.pop_front();
		}
	p_->netRequestsMutex.unlock();
	return req;
	}
