// Description: creates a light source with position, ambient, diffuse, and specular settings
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: light
//<li> this node creates a pfLightSource node
//<li> Lighting in a scene is based on the number of lights, lighting-model modes, and the material properties of the illuminated objects where the contribution of each light source is the sum of:
//<ol>
//<li> light source ambient color, scaled by the material ambient reflectance
//<li> light source diffuse color, scaled by the material diffuse reflectance and the dot product of the vertex normal and the vertex-to-light source vector
//<li> light source specular color, scaled by the material specular reflectance and a function of the angle between the vertex-to-viewpoint vector and the vertex-to-light source vector
//</ol>
//<li> The ambient component is nondirectional and distributed uniformly throughout the environment
//<li> The ambient component is reflected from the object independent of surface location and orientation with equal intensity in all directions
//<li> The diffuse component takes into account the direction of a light source. The intensity of light striking a surface varies with the angle between the orientation of the object and the direction of the light source -- the diffuse parameter of a material determines how much light is scattered evenly in all directions
//<li> The specular component represents reflection off shiny surfaces. Specular lighting operates based on the viewer's position. Maximum specular reflectance occurs when the angle between the viewer and the direction of the reflected light is zero -- the brightness of a specular reflection decays based upon the exponential value shininess
//</ul>
//
// Category: Attributes
// Author: Dave Pape
// Revision: 11/01/01 Geoffrey Baum - added ambient and specular messages
//
#include <Performer/pf/pfLightSource.h>
#include <Performer/pf/pfGroup.h>
#include "ygNetKeys.h"
#include "ygLight.h"

extern "C" ygNode* construct_ygLight(const char* name,bool master) { return new ygLight(name,master); }

struct _ygLightPrivateData
	{
	pfLightSource * light;
	struct
		{
		pfVec4 position;
		pfVec3 ambient;
		pfVec3 diffuse;
		pfVec3 specular;
		bool on;
		} net;
	bool prevOn;
	};


ygLight::ygLight(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("ygLight",true);
	p_ = new struct _ygLightPrivateData;
	//create pfLightSource
	p_->light = new pfLightSource;
	pfnode()->addChild(p_->light);
	addNetKey("pos",&p_->net.position,YG_NET_VEC4);
	addNetKey("amb",&p_->net.ambient,YG_NET_VEC3);
	addNetKey("dif",&p_->net.diffuse,YG_NET_VEC3);
	addNetKey("spec",&p_->net.specular,YG_NET_VEC3);
	addNetKey("on",&p_->net.on,YG_NET_BOOL);
	reset();
	}


ygLight::~ygLight(void)
	{
	pfnode()->removeChild(p_->light);
	pfDelete(p_->light);
	delete p_;
	}


void ygLight::reset(void)
	{
	//if net master
	if (isNetMaster())
		{
		//set position to 0,0,1,0
		setPosition(0,0,1,0);
		//set ambient to 0,0,0
		setAmbient(0,0,0);
		//set diffuse to 1,1,1
		setDiffuse(1,1,1);
		//set specular to 1,1,1
		setSpecular(1,1,1);
		//set to on
		on();
		p_->prevOn = true;
		}
	ygNode::reset();
	}


void ygLight::app(void)
	{
	bool ison = isOn();
	if ((ison) && (!p_->prevOn))
		eventOccurred("LightOn");
	else if ((!ison) && (p_->prevOn))
		eventOccurred("LightOff");
	p_->prevOn = ison;
	ygNode::app();
	}


/*****  LightSource functions *****/

bool ygLight::isOn(void) const
	{
	return (p_->light->isOn() != 0);
	}


void ygLight::on(void)
	{
	if (!isOn())
		{
		p_->light->on();
		p_->net.on = true;
		netKeyChanged("on");
		}
	}


void ygLight::off(void)
	{
	if (isOn())
		{
		p_->light->off();
		p_->net.on = false;
		netKeyChanged("on");
		}
	}


void ygLight::toggle(void)
	{
	if (isOn())
		off();
	else
		on();
	}


void ygLight::setPosition(float x,float y,float z,float w)
	{
	p_->light->setPos(x,y,z,w);
	p_->net.position.set(x,y,z,w);
	netKeyChanged("pos");
	}


void ygLight::setAmbient(float r,float g,float b)
	{
	p_->light->setColor(PFLT_AMBIENT, r, g, b);
	p_->net.ambient.set(r,g,b);
	netKeyChanged("amb");
	}

void ygLight::setDiffuse(float r,float g,float b)
	{
	p_->light->setColor(PFLT_DIFFUSE, r, g, b);
	p_->net.diffuse.set(r,g,b);
	netKeyChanged("dif");
	}

void ygLight::setSpecular(float r,float g,float b)
	{
	p_->light->setColor(PFLT_SPECULAR, r, g, b);
	p_->net.specular.set(r,g,b);
	netKeyChanged("spec");
	}


/***** Message functions *****/

void ygLight::message(const ygMessage& msg)
	{
	//set the light to on
	if (msg == "on")
		on();
	//set the light to off
	else if (msg == "off")
		off();
	//toggle the light state
	else if (msg == "toggle")
		toggle();
	//set the position from which the light comes
	else if (msg == "position")
		{
		pfVec4 v4;
		pfVec3 v3;
		if (msg.getVec4Args(v4))
			setPosition(v4);
		else if (msg.getVec3Args(v3))
			setPosition(v3);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the ambient color of the light
	else if (msg == "ambient")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setAmbient(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the diffuse color of the light
	else if (msg == "diffuse")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setDiffuse(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the specular color of the light
	else if (msg == "specular")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setSpecular(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygNode::message(msg);
	}

/***** Network functions *****/

void ygLight::acceptNetKey(const ygString& key)
	{
	if (key == "pos")
		setPosition(p_->net.position);
	else if (key == "amb")
		setAmbient(p_->net.ambient);
	else if (key == "dif")
		setDiffuse(p_->net.diffuse);
	else if (key == "spec")
		setSpecular(p_->net.specular);
	else if (key == "on")
		{
		if (p_->net.on)
			on();
		else
			off();
		}
	else
		ygNode::acceptNetKey(key);
	}
