// Revision: 11/01/01 Dave Pape

#include <queue>
#include <CAVERN.hxx>
#include "ygMutex.h"
#include "ygFileRequest.h"
#include "ygUtil.h"
#include "ygDebugFlags.h"
#include "ygFileServer.h"

using namespace std;


struct _ygFileServerPrivateData
	{
	bool netActive;
	ygString serverHost;
	int serverPort;
	ygString downloadDir;
	ygMutex queueMutex;
	queue<ygFileRequest *> requestQueue;
	bool debugTransfer;
	};


struct _ygFileServerPrivateData * ygFileServer::p_ = 0;


void ygFileServer::initialize(void)
	{
	p_ = new struct _ygFileServerPrivateData;
	if (getenv("YG_FILESERVER_HOST"))
		{
		p_->serverHost = getenv("YG_FILESERVER_HOST");
		if (getenv("YG_FILESERVER_PORT"))
			p_->serverPort = atoi(getenv("YG_FILESERVER_PORT"));
		else
			p_->serverPort = CAVERNnet_remoteFileIOServer32_c::DEFAULT_PORT;
		if (getenv("YG_FILESERVER_DIRECTORY"))
			p_->downloadDir = getenv("YG_FILESERVER_DIRECTORY");
		else
			p_->downloadDir = ".";
		p_->netActive = true;
		p_->debugTransfer = ygDebugFlags::checkDebugEnv("fileserver.transfer");
		ygCreateThread(serverThread, NULL);
		}
	else
		p_->netActive = false;
	}


void ygFileServer::queueRequest(ygFileRequest * request)
	{
	if (p_->netActive)
		{
		p_->queueMutex.lock();
		p_->requestQueue.push(request);
		p_->queueMutex.unlock();
		}
	else
		{
		handleLocally(request);
		request->setDone();
		}
	}


void ygFileServer::halt(void)
	{
	p_->netActive = false;
	}


void * ygFileServer::serverThread(void*)
	{
	while (p_->netActive)
		{
		ygFileServer::process();
		usleep(1000);
		}
	return 0;
	}


void ygFileServer::process(void)
	{
	while (!p_->requestQueue.empty())
		{
		p_->queueMutex.lock();
		ygFileRequest * request = p_->requestQueue.front();
		p_->requestQueue.pop();
		p_->queueMutex.unlock();
		if (request->direction() == ygFileRequest::DOWNLOAD)
			doDownload(request);
		else
			doUpload(request);
		request->setDone();
		}
	}


void ygFileServer::doDownload(ygFileRequest *request)
	{
	CAVERNnet_remoteFileIOClient32_c client;
	ygString finalLocalName;
	client.setRemotePort(p_->serverPort);
	if (localFileOkay(request->localFile(), request->remoteFile(), client,
			finalLocalName))
		request->setLocalFile(finalLocalName);
	else
		{
		prefixLocalFile(request->localFile(), finalLocalName);
		if (p_->debugTransfer)
			cout << "ygFileServer downloading '" <<
				p_->serverHost << ":" << p_->serverPort <<
				":" << request->remoteFile() <<
				"' as '" << finalLocalName << "'" << endl;
		int status = client.download(p_->serverHost.c_str(),
					finalLocalName.c_str(),
					request->remoteFile().c_str());
		if (status != CAVERNnet_remoteFileIOClient32_c::OK)
			handleLocally(request);
		else
			request->setLocalFile(finalLocalName);
		}
	}


struct ygCavernCouldUseATimestampStruct
	{
	int a,b,c,d,e,f;
	};

bool ygFileServer::localFileOkay(const ygString& localFile,
				const ygString& remoteFile,
				CAVERNnet_remoteFileIOClient32_c& client,
				ygString& returnName)
	{
	struct stat statbuf;
	prefixLocalFile(localFile, returnName);
	if (stat(returnName.c_str(), &statbuf) != 0)
		{
		if (!ygFindFileInYgPath(localFile, returnName))
			return false;
		}
	ygCavernCouldUseATimestampStruct loc,rmt;
	client.getLocalDateTime(returnName.c_str(),
				&loc.a, &loc.b, &loc.c, &loc.d, &loc.e, &loc.f);
	client.getRemoteDateTime(p_->serverHost.c_str(), remoteFile.c_str(),
				&rmt.a, &rmt.b, &rmt.c, &rmt.d, &rmt.e, &rmt.f);
	if (client.compareDateTime(loc.a, loc.b, loc.c, loc.d, loc.e, loc.f,
				   rmt.a, rmt.b, rmt.c, rmt.d, rmt.e, rmt.f)
			== CAVERNnet_remoteFileIOClient32_c::EARLIER)
		return false;
	return true;
	}


void ygFileServer::prefixLocalFile(const ygString& name, ygString& returnName)
	{
	if (name[0] == '/')
		returnName = name;
	else
		{
		returnName = p_->downloadDir;
		returnName += "/";
		returnName += name;
		}
	}


void ygFileServer::doUpload(ygFileRequest *request)
	{
	CAVERNnet_remoteFileIOClient32_c client;
	client.setRemotePort(p_->serverPort);
	if (p_->debugTransfer)
		cout << "ygFileServer uploading '" << request->localFile() <<
			"' to '" << p_->serverHost << ":" << p_->serverPort <<
			":" << request->remoteFile() << "'" << endl;
	int status = client.upload(p_->serverHost.c_str(),
					request->localFile().c_str(),
					request->remoteFile().c_str());
	if (status != CAVERNnet_remoteFileIOClient32_c::OK)
		{
		request->setFailed();
		cerr << "WARNING: ygFileServer::doUpload: failed to upload '"
			<< request->localFile() << "' to '"
			<< p_->serverHost << ":" << p_->serverPort
			<< ":" << request->remoteFile() << "';"
			<< "	CAVERN return status = " << status << endl;
		}
	}


void ygFileServer::handleLocally(ygFileRequest *request)
	{
	if (request->direction() == ygFileRequest::DOWNLOAD)
		{
		if (p_->debugTransfer)
			cout << "ygFileServer looking for '" <<
				request->localFile() << "' locally\n";
		ygString fullPath;
		if (ygFindFileInYgPath(request->localFile(),fullPath))
			request->setLocalFile(fullPath);
		else
			request->setFailed();
		}
	}

