// Description: establishes sky color, fog, and clipping plane parameters within a space 
// 
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: environment
//<li> volumes are defined by a box, cylinder, sphere, point or infinite volume 
//<li> for volume messages refer to the <a href="ygSpace.html">ygSpace</a> node
//<li> environment settings are applied when the user enters the space                        
//<li> the last environment defined will be applied when environments overlap
//<li> objects located outside of the field defined by the clipping planes will not be drawn
//<li> fog is the atmospheric effect of aerosol water particles that occludes vision over a distance
//<li> fog color is blended with the pixel color based on the range from the viewport and the fog function
//<li> the fog function is specified by the following "type" options:
//<ul>
//<li> linear (computes fog linearly at pixels),
//<li> exp (computes fog exponentially at pixels (ex)),
//<li> exp2 (computes fog exponentially at pixels (ex squared))
//</ul>
//<li> "onset" is the distance from the viewport at which the fog begins to take effect.
//<li> "opaque" is the distance at which the fog is 100% opaque, objects beyond this distance will not be visible
//</ul>
// 
// Category: Attributes
// Author: Dave Pape
//           11/01/01
// Revision: 06/01/03 Alex Hill - Added acceptNetKey function to update clients
// 	     09/01/04 Ben Chang - Added messages for gradient sky and ground

#include "ygUtil.h"
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "ygEnvironment.h"
#include <Performer/pf/pfEarthSky.h>
using namespace std;

extern "C" ygNode* construct_ygEnvironment(const char* name,bool net) { return new ygEnvironment(name,net); }

struct _ygEnvironmentPrivateData
	{
	pfVec2 clip;
	pfVec3 skyColor;
	pfVec3 skyTopColor;
	pfVec3 skyBottomColor;
	pfVec3 horizonColor;
	pfVec3 groundFarColor;
	pfVec3 groundNearColor;
	int backgroundMode;
	float horizonAngle;
	float groundHeight;
	bool fog;
	int fogType;
	pfVec3 fogColor;
	float fogOnset, fogOpaque;
	int lastApply;
	};


ygEnvironment * ygEnvironment::activeEnvironment_ = NULL;


ygEnvironment::ygEnvironment(const char* name,bool net) : ygSpace(name,net)
	{
	setClassName("ygEnvironment",true);
	p_ = new struct _ygEnvironmentPrivateData;
	p_->clip[0] = .1;
	p_->clip[1] = 1000;
	p_->skyColor.set(0,0,0);
	p_->skyTopColor.set(0,0,0);
	p_->skyBottomColor.set(0,0,0);
	p_->horizonColor.set(0,0,0);
	p_->groundFarColor.set(0,0,0);
	p_->groundNearColor.set(0,0,0);
	p_->backgroundMode = PFES_FAST;	
	p_->horizonAngle = 10.0;
	p_->groundHeight = 0.0; 
	p_->fog = false;
	p_->lastApply = -999;
	addNetKey("clip", &p_->clip, YG_NET_VEC2);
	addNetKey("sky", &p_->skyColor, YG_NET_VEC3);
	addNetKey("skytop", &p_->skyTopColor, YG_NET_VEC3);
	addNetKey("skybottom",&p_->skyBottomColor, YG_NET_VEC3);
	addNetKey("horizon",&p_->horizonColor, YG_NET_VEC3);
	addNetKey("groundfar",&p_->groundFarColor, YG_NET_VEC3);
	addNetKey("groundnear",&p_->groundNearColor, YG_NET_VEC3);
	addNetKey("groundheight",&p_->groundHeight,YG_NET_FLOAT);
	addNetKey("horizonangle",&p_->horizonAngle,YG_NET_FLOAT);
	addNetKey("mode",&p_->backgroundMode, YG_NET_INT);
	addNetKey("fog", &p_->fog, YG_NET_BOOL);
	addNetKey("fogType", &p_->fogType, YG_NET_INT);
	addNetKey("fogColor", &p_->fogColor, YG_NET_VEC3);
	addNetKey("fogOnset", &p_->fogOnset, YG_NET_FLOAT);
	addNetKey("fogOpaque", &p_->fogOpaque, YG_NET_FLOAT);
	}

ygEnvironment::~ygEnvironment(void)
	{
	delete p_;
	}


void ygEnvironment::reset(void)
	{
	//set near and far planes to 0.1 and 1000
	setClip(.1, 1000);
	//set sky color to black (0,0,0)
	setSkyColor(pfVec3(0,0,0));
	setBackgroundMode(PFES_FAST);
	ygSpace::reset();
	}


void ygEnvironment::message(const ygMessage& msg)
	{
	//set the near and far clipping planes
	if (msg == "clip")
		{
		if (msg.args.size() == 2)
			setClip(msg.floatArg(0),msg.floatArg(1));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the sky color
	else if (msg == "skyColor")
		{
		pfVec3 color;
		if (msg.getVec3Args(color))
			setSkyColor(color);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	// set the earthsky mode
	else if (msg == "mode")
		{
		if (msg.args[0] == "flat")
			setBackgroundMode (PFES_FAST);
		else if (msg.args[0] == "gradient")
			setBackgroundMode (PFES_SKY_GRND);
		else if (msg.args[0] == "sky")
			setBackgroundMode (PFES_SKY);
		else if (msg.args[0] == "sky_clear")
			setBackgroundMode (PFES_SKY_CLEAR);
		}
	// set the sky, horizon, and ground color
	else if (msg == "skyTopColor")
		{
		pfVec3 color;
		if (msg.getVec3Args(color)) {
			p_->skyTopColor = color;
			setBackgroundMode (PFES_SKY_GRND);
			p_->lastApply = -999;
			netKeyChanged("skytop");
		}
		else
			msg.error (name(),"(wrong number of arguments)");			
		}
	else if (msg == "skyBottomColor")
		{
		pfVec3 color;
		if (msg.getVec3Args(color)) {
			p_->skyBottomColor = color;
			p_->lastApply = -999;
			netKeyChanged("skybottom");
			setBackgroundMode (PFES_SKY_GRND);
		}
		else
			msg.error (name(),"(wrong number of arguments)");			
		}
	else if (msg == "horizonColor")
		{
		pfVec3 color;
		if (msg.getVec3Args(color)) {
			p_->horizonColor = color;
			p_->lastApply = -999;
			netKeyChanged("horizon");
			setBackgroundMode (PFES_SKY_GRND);
		}
		else
			msg.error (name(),"(wrong number of arguments)");			
		}
	else if (msg == "groundFarColor")
		{
		pfVec3 color;
		if (msg.getVec3Args(color)) {
			p_->groundFarColor = color;
			p_->lastApply = -999;
			netKeyChanged("groundfar");
			setBackgroundMode (PFES_SKY_GRND);
		}
		else
			msg.error (name(),"(wrong number of arguments)");			
		}
	else if (msg == "groundNearColor")
		{
		pfVec3 color;
		if (msg.getVec3Args(color)) {
			p_->groundNearColor = color;
			p_->lastApply = -999;
			netKeyChanged("groundnear");
			setBackgroundMode (PFES_SKY_GRND);
		}
		else
			msg.error (name(),"(wrong number of arguments)");			
		}
	else if (msg == "horizonAngle")
		{
		p_->horizonAngle = msg.floatArg(0);
		p_->lastApply = -999;
		netKeyChanged ("horizonangle");
		}
	else if (msg == "groundHeight")
		{
		p_->groundHeight = msg.floatArg(0);
		p_->lastApply = -999;
		netKeyChanged ("groundheight");
		}
	//set the fog properties (type, color, onset, opaque)
	else if (msg == "fog")
		{
		if (msg.args.size() < 1)
			msg.error(name(),"(wrong number of arguments)");
		else if (msg.args[0] == "off")
			fogOff();
		else if (msg.args.size() == 6)
			{
			int fogtype;
			pfVec3 color;
			float onset, opaque;
			if (msg.args[0] == "lin" || msg.args[0] == "linear")
				fogtype = PFFOG_PIX_LIN;
			else if (msg.args[0] == "exp")
				fogtype = PFFOG_PIX_EXP;
			else if (msg.args[0] == "exp2")
				fogtype = PFFOG_PIX_EXP2;
			msg.getVec3Args(color,1);
			onset = msg.floatArg(4);
			opaque = msg.floatArg(5);
			setFog(fogtype,color,onset,opaque);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygSpace::message(msg);
	}


void ygEnvironment::setClip(float near,float far)
	{
	p_->clip[0] = near;
	p_->clip[1] = far;
	p_->lastApply = -999;
	netKeyChanged("clip");
	}


float ygEnvironment::nearClip(void) const
	{
	return p_->clip[0];
	}


float ygEnvironment::farClip(void) const
	{
	return p_->clip[1];
	}


void ygEnvironment::setSkyColor(const pfVec3& color)
	{
	p_->skyColor = color;
	p_->lastApply = -999;
	netKeyChanged("sky");
	}


const pfVec3& ygEnvironment::skyColor(void) const
	{
	return p_->skyColor;
	}

void ygEnvironment::setBackgroundMode (int mode)
{
	p_->backgroundMode = mode;
	p_->lastApply = -999;
	netKeyChanged ("mode");
}
void ygEnvironment::fogOff(void)
	{
	p_->fog = false;
	p_->lastApply = -999;
	netKeyChanged("fog");
	}


void ygEnvironment::setFog(int fogtype,const pfVec3& color,float onset,float opaque)
	{
	p_->fog = true;
	p_->fogType = fogtype;
	p_->fogColor = color;
	p_->fogOnset = onset;
	p_->fogOpaque = opaque;
	p_->lastApply = -999;
	netKeyChanged("fog");
	netKeyChanged("fogType");
	netKeyChanged("fogColor");
	netKeyChanged("fogOnset");
	netKeyChanged("fogOpaque");
	}


void ygEnvironment::app(void)
	{
	ygNode * viewer = ygWorld::World->viewer();
	//if space contains user then apply settings
	if ((viewer) && (contains(viewer->origin(this))))
		apply();
	ygSpace::app();
	}


void ygEnvironment::apply(void)
	{
	//if properties have changed
	if (p_->lastApply < ygWorld::FrameNumber-1)
		{
		//set world clipping planes
		ygWorld::World->setClip(p_->clip[0], p_->clip[1]);
		//set world earthsky mode
		ygWorld::World->setBackgroundMode (p_->backgroundMode);
		//set world sky color
		if (p_->backgroundMode == PFES_SKY_GRND) {
			ygWorld::World->setEarthSkyGradient (p_->skyTopColor,p_->skyBottomColor,p_->horizonColor,p_->groundFarColor,p_->groundNearColor);			
			ygWorld::World->setHorizonAngle (p_->horizonAngle);
			ygWorld::World->setGroundHeight (p_->groundHeight);
		}
		else
			ygWorld::World->setSkyColor(p_->skyColor);
		//if fog then set world fog properties
		if (p_->fog)
			ygWorld::World->setFog(p_->fogType, p_->fogColor,
						p_->fogOnset, p_->fogOpaque);
		else
			ygWorld::World->fogOff();
		}
	/* The activeEnvironment_ class variable is intended to take care
	  of cases where environment nodes overlap.  It keeps track of which
	  node is 'active', i.e. the last one to apply itself.  Effectively,
	  when there is an overlap, the winning node changes the other node's
	  lastApply value to let the other node know that it has been un-applied,
	  allowing it to re-apply itself when the user gets out of the overlap
	  region. */
	if (activeEnvironment_)
		activeEnvironment_->p_->lastApply = -999;
	activeEnvironment_ = this;
	p_->lastApply = ygWorld::FrameNumber;
	}

void ygEnvironment::acceptNetKey(const ygString& key)
	{
	if (key == "clip" || 
		key == "sky" || key == "mode" || 
		key == "skytop" || key == "skybottom" || key == "horizon" || 
		key == "groundfar" || key == "groundnear" || 
		key == "horizonangle" ||
		key == "groundheight" || 
		key == "fog" || 
		key == "fogType" || key == "fogColor" || 
		key == "fogOnset" || key == "fogOpaque")
		p_->lastApply = -999;
	else
		ygSpace::acceptNetKey(key);
	}
