// Author: Dave Pape
// Revision: 08/01/03 Alex Hill - added ygSoundServer to constructor map

#include <stdio.h>
#include <iostream>
#include <dirent.h>
#include <strings.h>
#include <dlfcn.h>
#include <map>
#include "ygDebugFlags.h"
#include "ygNode.h"
#include "ygTransform.h"
#include "ygSimpleTransform.h"
#include "ygObject.h"
#include "ygTexture.h"
#include "ygLight.h"
#include "ygSwitch.h"
#include "ygSelector.h"
#include "ygUser.h"
#include "ygWand.h"
#include "ygSound.h"
#include "ygSpace.h"
#include "ygEnvironment.h"
#include "ygCAVEHead.h"
#include "ygCAVENavigator.h"
#include "ygCAVETracker.h"
#include "ygCAVEWand.h"
#include "ygDSOLoader.h"

using namespace std;


class ygDSOInfo
	{
	public:
	 ygString name;
	 ygString fullpath;
	 bool isLoaded;
	 bool isLoading;
	 bool loadFailed;	/* For some fucked up reason, Linux's dlopen will
				   succeed the second time you try a dso, even though
				   dependencies still aren't met; as a result, the
				   constructor quickly crashes.  This flag makes sure
				   we only try to load once */
	 ygNodeConstructorPtr constructPtr;
	 void *dlhandle;
	 ygDSOInfo(const ygString& n)
		{
		name = n;
		isLoaded = false;
		isLoading = false;
		loadFailed = false;
		constructPtr = 0;
		dlhandle = 0;
		}
	};


struct _ygDSOLoaderPrivateData
	{
	map<ygString,ygDSOInfo*> infomap;
	bool debugLoad;
	};


typedef void (*ygInitFuncPtr)(void);

static ygString searchFileForTag(const ygString &file, const char * tag);


struct _ygDSOLoaderPrivateData * ygDSOLoader::p_ = NULL;


void ygDSOLoader::init(void)
	{
	if (!p_)
		p_ = new struct _ygDSOLoaderPrivateData;
	p_->debugLoad = ygDebugFlags::checkDebugEnv("ygdso.load");
	addBuiltinConstructors();
	searchForDSOs();
	if (getenv("YG_PRELOAD_CLASSES"))
		preLoadClasses(getenv("YG_PRELOAD_CLASSES"));
	}


void ygDSOLoader::addBuiltinConstructors(void)
	{
	addConstructor("ygNode",construct_ygNode);
	addConstructor("Node",construct_ygNode);
	addConstructor("ygGroup",construct_ygNode);
	addConstructor("Group",construct_ygNode);
	addConstructor("ygTransform",construct_ygTransform);
	addConstructor("Transform",construct_ygTransform);
	addConstructor("ygSimpleTransform",construct_ygSimpleTransform);
	addConstructor("SimpleTransform",construct_ygSimpleTransform);
	addConstructor("ygLight",construct_ygLight);
	addConstructor("Light",construct_ygLight);
	addConstructor("ygObject",construct_ygObject);
	addConstructor("Object",construct_ygObject);
	addConstructor("ygTexture",construct_ygTexture);
	addConstructor("Texture",construct_ygTexture);
	addConstructor("ygSwitch",construct_ygSwitch);
	addConstructor("Switch",construct_ygSwitch);
	addConstructor("ygSelector",construct_ygSelector);
	addConstructor("Selector",construct_ygSelector);
	addConstructor("ygUser",construct_ygUser);
	addConstructor("User",construct_ygUser);
	addConstructor("ygCAVEHead",construct_ygCAVEHead);
	addConstructor("CAVEHead",construct_ygCAVEHead);
	addConstructor("ygWand",construct_ygWand);
	addConstructor("Wand",construct_ygWand);
	addConstructor("ygCAVENavigator",construct_ygCAVENavigator);
	addConstructor("CAVENavigator",construct_ygCAVENavigator);
	addConstructor("ygCAVETracker",construct_ygCAVETracker);
	addConstructor("CAVETracker",construct_ygCAVETracker);
	addConstructor("ygCAVEWand",construct_ygCAVEWand);
	addConstructor("CAVEWand",construct_ygCAVEWand);
	addConstructor("ygSoundServer",construct_ygSoundServer);
	addConstructor("ygSound",construct_ygSound);
	addConstructor("Sound",construct_ygSound);
	addConstructor("ygSpace",construct_ygSpace);
	addConstructor("Space",construct_ygSpace);
	addConstructor("ygEnvironment",construct_ygEnvironment);
	addConstructor("Environment",construct_ygEnvironment);
	}


void ygDSOLoader::addConstructor(const ygString& name,
					ygNodeConstructorPtr constructor)
	{
	ygDSOInfo * info = new ygDSOInfo(name);
	info->isLoaded = true;
	info->constructPtr = constructor;
	p_->infomap[name] = info;
	}


void ygDSOLoader::preLoadClasses(const ygString &classes)
	{
	int i=0;
	ygString * name;
	while (name = classes.nextToken(" \t",&i))
		{
		getNodeConstructor(*name);
		delete name;
		}
	}


void ygDSOLoader::searchForDSOs(void)
	{
	searchDirectory(".");
	if (getenv("YG_DSO_PATH"))
		{
		ygString searchPath(getenv("YG_DSO_PATH"));
		ygString *dir;
		int i=0;
		while (dir = searchPath.nextToken(":",&i))
			{
			searchDirectory(*dir);
			delete dir;
			}
		}
	}


void ygDSOLoader::searchDirectory(const ygString& dirname)
	{
	DIR *dir = opendir(dirname.c_str());
	if (dir)
		{
		struct dirent * ent;
		while (ent = readdir(dir))
			{
			int len = strlen(ent->d_name);
			if (strcmp(ent->d_name+len-3, ".so") == 0)
				{
				ygString name(ent->d_name);
				name.erase(name.length()-3);
				if (!p_->infomap[name])
					{
					ygDSOInfo *info = new ygDSOInfo(name);
					info->fullpath = dirname;
					info->fullpath += '/';
					info->fullpath += ent->d_name;
					p_->infomap[info->name] = info;
					}
				}
			}
		closedir(dir);
		}
	}


ygNodeConstructorPtr ygDSOLoader::getNodeConstructor(const ygString& nodeType)
	{
	ygDSOInfo *info = p_->infomap[nodeType];
	if (info)
		{
		if (!info->isLoaded)
			loadDSO(info);
		if ((info->isLoaded) && (info->constructPtr == NULL))
			cerr << "ERROR: ygConstructorDB::constructor(" << nodeType <<
				") - DSO loaded successfully, but could not find"
				" constructor\n";
		return info->constructPtr;
		}
	else
		{
		cerr << "ERROR: cannot find DSO for node type '"
			<< nodeType << "'\n";
		return NULL;
		}
	}


bool ygDSOLoader::loadDSO(ygDSOInfo *info)
	{
	if ((!info) || (info->loadFailed))
		return false;
	if (info->isLoaded)
		return true;
	if (info->isLoading)
		{
		cerr << "ERROR: ygDSOLoader::loadDSO(" << info->name << ") "
		     "- there appears to be an infinite loop of dependencies\n";
		return false;
		}
	info->isLoading = true;
	if (p_->debugLoad)
		cout << "ygDSOLoader loading DSO for " << info->name << endl;
	if (!loadDSODependencies(info))
		{
		cerr << "ERROR: failed to load dependencies for "
			<< info->fullpath << endl;
		return false;
		}
	info->dlhandle = dlopen(info->fullpath.c_str(), RTLD_LAZY|RTLD_GLOBAL);
	if (!info->dlhandle)
		{
		info->loadFailed = true;
		cerr << "ERROR: failed to open " << info->fullpath << endl;
		cerr << dlerror() << endl;
		return false;
		}
	ygString constructorName("construct_");
	constructorName += info->name;
	info->constructPtr = (ygNodeConstructorPtr)
				dlsym(info->dlhandle,constructorName.c_str());
	ygString initFuncName("initialize_");
	initFuncName += info->name;
	ygInitFuncPtr initFuncPtr = (ygInitFuncPtr)
				dlsym(info->dlhandle,initFuncName.c_str());
	if (initFuncPtr)
		(*initFuncPtr)();
	info->isLoaded = true;
	info->isLoading = false;
	return true;
	}


bool ygDSOLoader::loadDSODependencies(const ygDSOInfo *info)
	{
	if (!loadYgdepFileDependencies(info))
		return false;
	if (!loadStringTagDependencies(info))
		return false;
	return true;
	}


bool ygDSOLoader::loadYgdepFileDependencies(const ygDSOInfo *info)
	{
	FILE *fp;
	ygString filename(info->fullpath);
	filename.erase(filename.length()-2);	/* Remove the "so" */
	filename += "ygdep";
	fp = fopen(filename.c_str(),"r");
	if (fp)
		{
		char buf[256],name[256];
		while (fgets(buf,sizeof(buf),fp))
			{
			if (sscanf(buf,"%s",name) < 1)
				continue;
			ygDSOInfo* info = p_->infomap[ygString(name)];
			if (info)
				{
				if (!loadDSO(info))
					return false;
				}
			else
				{
				cerr << "ERROR: can't find DSO for "
					<< name << endl;
				return false;
				}
			}
		fclose(fp);
		}
	return true;
	}


bool ygDSOLoader::loadStringTagDependencies(const ygDSOInfo *info)
	{
	ygString dependencies = searchFileForTag(info->fullpath,
						"YG DSO Dependencies:");
	ygString *className;
	int i=0;
	while (className = dependencies.nextToken(" \t",&i))
		{
		ygDSOInfo* info = p_->infomap[*className];
		if (info)
			{
			if (!loadDSO(info))
				return false;
			}
		else
			{
			cerr << "ERROR: can't find DSO for " << *className <<
				 endl;
			return false;
			}
		delete className;
		}
	return true;
	}


/* searchFileForTag searches the given file for a string that begins with
   the given 'tag'.  If this is found, it returns the rest of the string,
   up to the closing \0 character or end-of-file.
   e.g., if the tag is "YG DSO Dependencies:", and the file contains the
   string "YG DSO Dependencies: foo.so", then " foo.so" is returned.
   Only the first matching string will be returned. */
static ygString searchFileForTag(const ygString &file, const char * tag)
	{
	ygString returnValue("");
	FILE *fp = fopen(file.c_str(),"r");
	if (fp)
		{
		int len = strlen(tag);
		char *inputBuffer = (char *) calloc(1,len);
		int i = 0;
		/* The search works by loading the file one byte at a time
		   into a ring buffer and then comparing the buffer to the
		   target string.  The comparison is done in 2 memcmps because
		   the data from the file will be split in the ring buffer. */
		int c = getc(fp);
		while (c != EOF)
			{
			inputBuffer[i] = c;
			i = (i+1) % len;
			if ((memcmp(tag,inputBuffer+i,len-i) == 0) &&
			    (memcmp(tag+(len-i),inputBuffer,i) == 0))
				{
				c = getc(fp);
				while ((c != '\0') && (c != EOF))
					{
					returnValue += c;
					c = getc(fp);
					}
				c = EOF;
				}
			else
				c = getc(fp);
			}
		fclose(fp);
		}
	return returnValue;
	}
