// Description: creates a navigator with CAVE navigation data for the user
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: CAVENavigator
//<li> this node should be a child of a <a href="ygUser.html">ygUser</a> node
//<li> sets the navigator of the user above it for use in navigation
//<li> this node is required for navigating the CAVE within the virtual space
//</ul>
//
// Category: User
// Author: Dave Pape
//           11/01/01
// Revision: 01/20/03 Alex Hill - compute head position from navData for checkGround()
//
#include <Performer/pr/pfLinMath.h>
#include <Performer/pf/pfScene.h>
#include <pfcave.h>
#include "ygWorld.h"
#include "ygUtil.h"
#include "ygSphereIsect.h"
#include "ygCAVENavigator.h"

using namespace std;

extern "C" ygNode* construct_ygCAVENavigator(const char* name,bool master) { return new ygCAVENavigator(name,master); }


ygCAVENavigator::ygCAVENavigator(const char* name,bool master) : ygNavigator(name,master)
	{
	rotSpeed_ = 45;
	transSpeed_ = 10;
	collide_ = true;
	collideRadius_ = 1;
	fly_ = true;
	}

void ygCAVENavigator::reset(void)
	{
	//set rotation speed to 45
	rotSpeed_ = 45;
	//set translation speed to 10
	transSpeed_ = 10;
	//set collide to true
	collide_ = true;
	//set collide radius to 1
	collideRadius_ = 1;
	//set fly mode to true
	fly_ = true;
	ygNavigator::reset();
	}

void ygCAVENavigator::message(const ygMessage& msg)
	{
	//move the user to the absolute position given
	if (msg == "teleport")
		{
		pfVec3 pos;
		if (msg.getVec3Args(pos))
			teleportTo(pos);
		}
	//orient the user to the absolute rotation given
	else if (msg == "heading")
		{
		if (msg.args.size() > 0)
			setHeading(msg.floatArg(0));
		}
	//set the forward and backward navigation speed
	else if (msg == "speed")
		{
		if (msg.args.size() > 0)
			setSpeed(msg.floatArg(0));
		}
	//set the turning rotational speed
	else if (msg == "rotspeed")
		{
		if (msg.args.size() > 0)
			setRotSpeed(msg.floatArg(0));
		}
	//turn on collision detection with objects
	else if (msg == "collide")
		{
		if (msg.args.size() > 0)
			setCollide(msg.boolArg(0));
		else
			setCollide();
		}
	//set the collision radius around the head
	else if (msg == "collideRadius")
		collideRadius_ = msg.floatArg(0);
	//toggle the collision detection mode
	else if (msg == "togglecollide")
		setCollide(!collide_);
	//set the fly mode between walking and flying
	else if (msg == "fly")
		{
		if (msg.args.size() > 0)
			setFly(msg.boolArg(0));
		else
			setFly();
		}
	//toggle the fly mode
	else if (msg == "togglefly")
		setFly(!fly_);
	//print out the current navigator position
	else if (msg == "printnav")
		{
		cout << "Navigator position = " << data().position()
			<< "  orientation = " << data().orientation() << endl;
		}
	else
		ygNavigator::message(msg);
	}


void ygCAVENavigator::teleportTo(const pfVec3& pos)
	{
	data().setPosition(pos);
	resetCollision();
	}
	
void ygCAVENavigator::setHeading(float deg)
	{
	data().setHeading(deg);
	resetCollision();
	}

void ygCAVENavigator::setSpeed(float s)
	{
	transSpeed_ = s;
	}


void ygCAVENavigator::setRotSpeed(float s)
	{
	rotSpeed_ = s;
	}


void ygCAVENavigator::setCollide(bool val)
	{
	if ((val) && (!collide_))
		resetCollision();
	collide_ = val;
	}


void ygCAVENavigator::setFly(bool val)
	{
	fly_ = val;
	}


void ygCAVENavigator::app(void)
	{
	//check for CAVE joystick input
	checkJoystick();
	//if fly mode is false then adjust height to any ground collision
	if (!fly_)
		checkGround();
	//if collision detection is on then test for object collisions
	if (collide_)
		checkCollision();
	ygNavigator::app();
	}

void ygCAVENavigator::checkJoystick(void)
	{
	//if the CAVE joystick X direction is above threshold
	if (fabs(CAVE_JOYSTICK_X) > 0.2f)
		{
		float h = data().heading();
		//rotate the navigation heading relative to the joystick value
		h -= CAVE_JOYSTICK_X * (fabs(CAVE_JOYSTICK_X)-0.2f) *
			rotSpeed_ * ygWorld::FrameDeltaTime;
		h = fmod(h,(float)360.0);
		data().setHeading(h);
		}
	//if the CAVE joystick Y direction is above threshold
	if (fabs(CAVE_JOYSTICK_Y) > 0.1f)
		{
		pfVec3 dir, pos = data().position();
		//get the CAVE wand direction
		CAVEGetVector(CAVE_WAND_FRONT_NAV, dir.vec);
		//if fly mode is false then zero the Z component
		if (!fly_)
			dir[PF_Z] = 0;
		//translate the navigation position relative to the joystick value
		pos += dir * CAVE_JOYSTICK_Y * transSpeed_
			* ygWorld::FrameDeltaTime;
		data().setPosition(pos);
		}
	}


void ygCAVENavigator::checkGround(void)
	{
	pfSegSet segset;
	pfHit **hits[32];
	segset.activeMask = 1;
	segset.isectMask = YG_ISECT_FLOORMASK;
	segset.discFunc = NULL;
	segset.bound = NULL;
	segset.mode = PFTRAV_IS_PRIM;
	//generate intersection segment down from user head
	segset.segs[0].dir.set(0.0f, 0.0f, -1.0f);
	segset.segs[0].length = 5000.0f;
	pfVec3 headTrackerPos, rotHeadTrackerPos;
	pfMatrix rotXform;
	CAVEGetPosition(CAVE_HEAD, headTrackerPos.vec);
	rotXform.makeRot(data().heading(),0,0,1);
	rotHeadTrackerPos.xformPt(headTrackerPos, rotXform);
	segset.segs[0].pos = rotHeadTrackerPos + data().position();
	//if intersection detected then adjust user height to collision position
	if (ygWorld::World->pfscene()->isect(&segset, hits))
		{
		pfVec3 pnt, xpnt, pos = data().position();
		pfMatrix xmat;
		(*hits[0])->query(PFQHIT_POINT, pnt.vec);
		(*hits[0])->query(PFQHIT_XFORM, (float*)xmat.mat);
		xpnt.xformPt(pnt, xmat);
		if (pos[PF_Z] > xpnt[PF_Z] + 2.0f)
			pos[PF_Z] -= 2.0f;
		else
			pos[PF_Z] = xpnt[PF_Z];
		data().setPosition(pos);
		}
	}


void ygCAVENavigator::checkCollision(void)
	{
	bool OK;
	pfVec3 headTrackerPos, rotHeadTrackerPos, curPos;
	pfMatrix rotXform;
	CAVEGetPosition(CAVE_HEAD, headTrackerPos.vec);
	rotXform.makeRot(data().heading(),0,0,1);
	rotHeadTrackerPos.xformPt(headTrackerPos, rotXform);
	curPos = rotHeadTrackerPos + data().position();
	if (curPos.almostEqual(prevCollidePos_,.001))
		return;
	OK = positionOK(curPos);
	if (OK)
		prevCollidePos_ = curPos;
	else
		data().setPosition(prevCollidePos_ - rotHeadTrackerPos);
	}


bool ygCAVENavigator::positionOK(pfVec3 pos)
	{
	pfSphere sphere;
	sphere.center = pos;
	sphere.radius = collideRadius_;
	return !ygSphereIsect(ygWorld::World->pfscene(),&sphere,YG_ISECT_WALLMASK);
	}


void ygCAVENavigator::resetCollision(void)
	{
	pfVec3 headTrackerPos, rotHeadTrackerPos;
	pfMatrix rotXform;
	CAVEGetPosition(CAVE_HEAD, headTrackerPos.vec);
	rotXform.makeRot(data().heading(),0,0,1);
	rotHeadTrackerPos.xformPt(headTrackerPos, rotXform);
	prevCollidePos_ = rotHeadTrackerPos + data().position();
	}
