#include <stdlib.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <CAVERN.hxx>
#include "ygNetClient.h"
#include "ygKeyDatabase.h"
#include "ygrepeaterQueue.h"

using namespace std;


#define MAX_LINKS 256

typedef struct
	{
	char * linkHost[MAX_LINKS];
	int numLinks;
	int portNumber;
	int queueSize;
	int cachePortNumber;
	int statsInterval, dumpInterval;
	} RepeaterOpts_t;


void parseCommandLine(int argc,char **argv,RepeaterOpts_t *opts);
void dualLoop(ygrepeaterQueue *,RepeaterOpts_t *opts);
void * databaseCache(void *);


RepeaterOpts_t repOpts;


main(int argc, char **argv)
	{
	cout << "ygrepeater for Ygdrasil version 0.1.11\n";
	repOpts.numLinks = 0;
	repOpts.portNumber = 3500;
	repOpts.queueSize = 2000000;
	repOpts.cachePortNumber = 0;
	repOpts.statsInterval = 0;
	repOpts.dumpInterval = 0;
	parseCommandLine(argc,argv,&repOpts);
	if (!repOpts.cachePortNumber)
		repOpts.cachePortNumber = repOpts.portNumber+1;
	CAVERNinit();
	ygrepeaterQueue * repqueue = new ygrepeaterQueue(repOpts.queueSize);
	CAVERNts_thread_c dbThread;
	dbThread.create(databaseCache,repqueue);
	dualLoop(repqueue,&repOpts);
	}


void parseCommandLine(int argc,char **argv,RepeaterOpts_t *opts)
	{
	int i;
	for (i=1; i < argc; i++)
		{
		if (!strncmp(argv[i],"--",2))
			{
			if (!strcasecmp(argv[i],"--link"))
				{
				i++;
				if (opts->numLinks < MAX_LINKS)
					{
					opts->linkHost[opts->numLinks] = strdup(argv[i]);
					opts->numLinks++;
					}
				else
					fprintf(stderr, "ERROR: cannot link to more than"
						" %d hosts; link to '%s' ignored\n",
						MAX_LINKS, argv[i]);
				}
			else if (!strcasecmp(argv[i],"--port"))
				{
				i++;
				opts->portNumber = atoi(argv[i]);
				}
			else if (!strcasecmp(argv[i],"--queuesize"))
				{
				i++;
				opts->queueSize = atoi(argv[i]);
				}
			else if (!strcasecmp(argv[i],"--cacheport"))
				{
				i++;
				opts->cachePortNumber = atoi(argv[i]);
				}
			else if (!strcasecmp(argv[i],"--stats"))
				{
				i++;
				opts->statsInterval = atoi(argv[i]);
				}
			else if (!strcasecmp(argv[i],"--dump"))
				{
				i++;
				opts->dumpInterval = atoi(argv[i]);
				}
			else
				fprintf(stderr,"WARNING: unknown command line "
					"option '%s'\n",argv[i]);
			}
		}
	}


/**************************************************************************/

int udpIntercept(CAVERNnet_udp_c *,char**,int *,void *);
int tcpIntercept(CAVERNnet_tcpReflectorClient_c *client, char** buffer,
		int *bufferSize, void *userData);

void dualLoop(ygrepeaterQueue *repqueue,RepeaterOpts_t *opts)
	{
	int i, sprevTime=time(NULL), nowTime;

	CAVERNnet_udpReflector_c *udpReflector;
	udpReflector = new CAVERNnet_udpReflector_c;
	udpReflector->setIncomingPort(opts->portNumber);
	for (i=0; i < opts->numLinks; i++)
		udpReflector->setForcedDestination(opts->linkHost[i], opts->portNumber);
	udpReflector->intercept(udpIntercept,repqueue);
	udpReflector->init();

	CAVERNnet_tcpReflector_c *tcpReflector;
	tcpReflector = new CAVERNnet_tcpReflector_c;
	tcpReflector->init(opts->portNumber);
	for (i=0; i < opts->numLinks; i++)
		tcpReflector->setForcedDestination(opts->linkHost[i], opts->portNumber);
	tcpReflector->intercept(tcpIntercept,repqueue);

	signal(SIGSEGV, SIG_DFL);
	signal(SIGBUS, SIG_DFL);
	while (true)
		{
/* Being able to call select() & block on these 2 sockets would really really help */
		udpReflector->process();
		tcpReflector->process();
		if (opts->statsInterval > 0)
			{
			nowTime = time(NULL);
			if (nowTime - sprevTime >= opts->statsInterval)
				{
				udpReflector->showStats("UDPReflector","ygrepeater017 UDPReflector");
				tcpReflector->showStats("TCPReflector","ygrepeater017 TCPReflector");
				sprevTime = nowTime;
				}
			}
		}
	}


int udpIntercept(CAVERNnet_udp_c *client, char** buffer, int *bufferSize,
		void *userData)
	{
	ygrepeaterQueue *repqueue = (ygrepeaterQueue *) userData;
	repqueue->add(*buffer,*bufferSize);
// Previously, there were 8 garbage bytes on the front of UDP packets;
// now, they seem to be okay.
// ???
//	repqueue->add((*buffer)+8,(*bufferSize)-8);
	return CAVERNnet_udpReflector_c::OK;
	}


int tcpIntercept(CAVERNnet_tcpReflectorClient_c *client, char** buffer,
		int *bufferSize, void *userData)
	{
	ygrepeaterQueue *repqueue = (ygrepeaterQueue *) userData;
	repqueue->add(*buffer,*bufferSize);
	return CAVERNnet_udpReflector_c::OK;
	}


/**************************************************************************/
/**************************************************************************/


/** Note:  I would have just used a CAVERNnet_tcpServer for the cache's
   TCP connection, since we don't need to reflect data among clients.
   But, because CAVERNnet_tcpReflector handles the packetizing of data
   automatically, and because the other, existing code currently expects
   a CAVERNnet_tcpReflectorClient, it was simpler to use the reflector,
   and just prevent it from doing any actual reflection.
   Clients will be expected to disconnect once the download is finished.
**/

int cacheTCPClient(CAVERNnet_tcpReflectorClient_c *client, char** buffer,
		int *bufferSize, void *userData);
void checkRepeaterQueue(ygrepeaterQueue *repqueue,ygKeyDatabase *ygdb);
void saveKeyData(ygKeyDatabase *db,CAVERNnet_datapack_c& packer);


void * databaseCache(void *userData)
	{
	FILE *dumpFP;
	int prevTime=time(NULL), nowTime;
	struct timespec sleeptime;
	ygrepeaterQueue *repqueue = (ygrepeaterQueue *) userData;
	ygKeyDatabase *ygdb = new ygKeyDatabase;
	CAVERNnet_tcpReflector_c *server = new CAVERNnet_tcpReflector_c;
	server->init(repOpts.cachePortNumber);
	server->intercept(cacheTCPClient,ygdb);
	if (repOpts.dumpInterval > 0)
		{
		char filename[256];
		sprintf(filename, "ygdbdump.%d", getpid());
		dumpFP = fopen(filename, "w");
		if (!dumpFP)
			dumpFP = stdout;
		}
	while (true)
		{
		server->process();
		checkRepeaterQueue(repqueue,ygdb);
		if (repOpts.dumpInterval > 0)
			{
			nowTime = time(NULL);
			if (nowTime - prevTime >= repOpts.dumpInterval)
				{
				ygdb->printContents(dumpFP);
				prevTime = nowTime;
				}
			}
/* Sleep since we can't block on a select(); this thread doesn't need to be as
   ultra-responsive as the reflector thread */
		sleeptime.tv_sec = 0;
		sleeptime.tv_nsec = 1000000;
		nanosleep(&sleeptime,NULL);
		}
	return NULL;
	}


int cacheTCPClient(CAVERNnet_tcpReflectorClient_c *client, char** buffer,
		int *bufferSize, void *userData)
	{
	ygKeyDatabase *ygdb = (ygKeyDatabase *) userData;
	CAVERNnet_datapack_c packer;
	int packetType;
	packer.initUnpack(*buffer,*bufferSize);
	packer.unpackInt(&packetType);
	if (packetType == YG_NET_REQUEST_ALL)
		{
		char clientName[256];
		client->getRemoteIP(clientName);
		printf("sending database to new client %s\n",clientName);
		ygdb->sendContents(client);
		printf("done sending database\n");
		fflush(stdout);
		}
	return CAVERNnet_udpReflector_c::SKIP_DISTRIBUTION;
	}


void checkRepeaterQueue(ygrepeaterQueue *repqueue,ygKeyDatabase *ygdb)
	{
	char buffer[YG_NET_MAX_BUFFER_SIZE];
	size_t packetSize;
	while (repqueue->get(buffer,&packetSize,sizeof(buffer)) > 0)
		{
		CAVERNnet_datapack_c packer;
		int packetType;
		packer.initUnpack(buffer,packetSize);
		while (true)
			{
			int packetType = -1;
			if (packer.unpackInt(&packetType) ==
			    CAVERNnet_datapack_c::FAILED)
				break;
			if (packetType == YG_NET_PUT_KEY)
				saveKeyData(ygdb, packer);
			else
				break;
			}
		}
	}


void saveKeyData(ygKeyDatabase *db,CAVERNnet_datapack_c& packer)
	{
	static char buffer[YG_NET_MAX_BUFFER_SIZE];
	ygString path, key;
	int length;
/* Get path string length & data */
	packer.unpackInt(&length);
	packer.unpack(buffer, length);
	path.append(buffer);
/* Get key string length & data */
	packer.unpackInt(&length);
	packer.unpack(buffer, length);
	key.append(buffer);
/* Get data length & data */
	packer.unpackInt(&length);
	packer.unpack(buffer, length);
	db->store(path, key, buffer, length);
	}


/**************************************************************************/
