// Description: creates a positional whiteNoise source within a space
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the volume of a sound is the region in which the sound will play at 100% amplitude(loudness)
//<li> falloffDistance is a region outside the volume in which the amplitude is diminished as the user moves further away from the source(volume)
//<li> the sound server, snerd, must be run on the a local or remote machine
//<li> sounds under a switch will turn off but must be restarted when the switch is toggled back on
//</ul>
//
// Category: Sound
// Author: Alex Hill
// Revision: 08/01/03
//
#include <pfcave.h>
#include <bergenWhiteNoise.h>
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "whiteNoise.h"

using namespace std;

extern "C" ygNode* construct_whiteNoise(const char* name,bool master) { return new whiteNoise(name,master); }

struct _whiteNoisePrivateData
	{
	bergenWhiteNoise * sample;
	bool isPlaying;
	float curAmplitude;
	pfVec3 currPosition;
	float updateInterval, lastUpdate;
	bool netPlay;
	};


whiteNoise::whiteNoise(const char* name,bool master) : ygSoundServer(name,master)
	{
	setClassName("whiteNoise",true);
	p_ = new struct _whiteNoisePrivateData;
	p_->sample = NULL;
	p_->curAmplitude = 0;
	p_->updateInterval = 0.1;
	p_->lastUpdate = 0;
	p_->isPlaying = false;
	addNetKey("play", &p_->netPlay, YG_NET_BOOL);
	}

whiteNoise::~whiteNoise(void)
	{
	if (p_->sample)
		delete p_->sample;
	delete p_;
	}


void whiteNoise::reset(void)
	{
	//reset playing flag to false
	p_->isPlaying = false;
	//set current amplitude to 0.0
	p_->curAmplitude = 0;
	//set update interval to 0.1
	p_->updateInterval = 0.1;
	//if a sample exists then delete it
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	ygSpace::reset();
	}


void whiteNoise::message(const ygMessage& msg)
	{
	//set the amplitude within the space
	if (msg == "amplitude")
		{
		if (msg.args.size() > 0)
			setAmplitude(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the zero amplitude distance from the space perimeter
	else if (msg == "falloffDistance")
		{
		if (msg.args.size() > 0)
			setFalloffDistance(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//start the sound playing with optional amplitude
	else if (msg == "play")
		{
		if (msg.args.size() > 0)
			{
			setAmplitude(msg.floatArg(1));
			play();
			}
		else
			play();
		}
	//stop the playing sound
	else if (msg == "stop")
		stop();
	//set the update interval for sound server
	else if (msg == "updateInterval")
		{
		if (msg.args.size() > 0)
			p_->updateInterval = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygSoundServer::message(msg);
	}


void whiteNoise::setCurAmplitude(float val)
	{
	p_->curAmplitude = val;
	if (p_->sample)
		p_->sample->setAmplitude(val);
	}

bool whiteNoise::isPlaying(void)
	{
	return p_->isPlaying;
	}

void whiteNoise::play(void)
	{
	if (p_->isPlaying)
		return;
	//if no sample then create a new bergenSample
	if (!p_->sample)
		p_->sample = new bergenWhiteNoise(server());
	setCurAmplitude(computeAmplitude());
	//play the sample
	p_->sample->play();
	//generate a heartbeat for the server
	heartbeat();
	p_->isPlaying = true;
	eventOccurred("start");
	p_->netPlay = true;
	netKeyChanged("play");
	}


void whiteNoise::stop(bool sendStopCommand)
	{
	//if a sample is playing
	if (p_->isPlaying)
		{
		//send a stop command to the server
		if ((sendStopCommand) && (p_->sample))
			{
			p_->sample->stop();
			setCurAmplitude(0.0);
			}
		p_->isPlaying = false;
		eventOccurred("stop");
		p_->netPlay = false;
		netKeyChanged("play");
		}
	}


void whiteNoise::app(void)
	{
	//if a sample exists
	if (p_->sample)
		{
		//generate a heartbeat for the server
		heartbeat();
		//if the sample is playing
		if (p_->isPlaying)
			{
			//if update interval has passed then update sound server
			if ((ygWorld::FrameTime - p_->lastUpdate) > p_->updateInterval)
				{
				float newAmpl = computeAmplitude();
				if (fabs(newAmpl - p_->curAmplitude) > 0.005)
					setCurAmplitude(newAmpl);
				//determine position of this sound
				pfVec3 newPos;
				newPos = origin();
				pfVec3 cavePos;
				//get position relative to CAVE
				CAVENavConvertWorldToCAVE(newPos.vec,cavePos.vec);
				if (cavePos != p_->currPosition)
					{
					if (p_->sample)
						p_->sample->setPosition(cavePos.vec[0],cavePos.vec[2],-cavePos.vec[1]);
					p_->currPosition = cavePos;
					}
				p_->lastUpdate = ygWorld::FrameTime;
				}
			}
		}
	ygSoundServer::app();
	}

void whiteNoise::acceptNetKey(const ygString& key)
	{
	if (key == "play")
		{
		if (p_->netPlay)
			play();
		else
			stop();
		}
	else
		ygSoundServer::acceptNetKey(key);
	}
