// Description: a tri-strip surface with a sinusoidal wave motion
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> for best results scale parent transforms equally in all directions
//</ul>
//
// Category: Geometry
// Author: Kyoung Park
//           11/01/01
//           01/15/03 Alex Hill - derived from surface instead of ygNode
// Revision: 04/01/04 Alex Hill - changed "size" to "width" msg, add "height" message
//
#include <Performer/pf/pfGroup.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pr/pfMaterial.h> 
#include <Performer/pr/pfTexture.h>
#include <Performer/pr.h>
#include <ygNetKeys.h>
#include <ygWorld.h>
#include "waves.h"

using namespace std;

extern "C" ygNode* construct_waves(const char* name,bool master) { return new waves(name,master); }

waves::waves(const char* name,bool master) : surface(name,master)
	{
	setClassName("waves",true);
	speed_ = 90.0;
	width_ = 50.0;
	height_ = 1.0;
	active_ = true;
	currTime_ = 0.0;
	addNetKey("speed",&speed_,YG_NET_FLOAT);
	addNetKey("width",&width_,YG_NET_FLOAT);
	addNetKey("height",&height_,YG_NET_FLOAT);
	addNetKey("active",&active_,YG_NET_FLOAT);
	}

waves::~waves(void)
	{
	}

void waves::reset(void)
	{	
	//set speed to 100.0
	speed_ = 100.0;
	netKeyChanged("speed");
	//set size to 180.0
	width_ = 180.0;
	netKeyChanged("width");
	//set height to 1.0
	height_ = 1.0;
	netKeyChanged("height");
	//set current time to 0.0
	currTime_ = 0.0;
	//start motion
	active_ = true;
	netKeyChanged("active");
	surface::reset();
	}


void waves::message(const ygMessage& msg)
	{
	//set the wave speed
	if (msg == "speed")
		{
		speed_ = msg.floatArg(0);
		netKeyChanged("speed");
		}
	//set the wave width
	else if (msg == "width")
		{
		width_ = msg.floatArg(0);
		netKeyChanged("width");
		}
	//set the wave height
	else if (msg == "height")
		{
		height_ = msg.floatArg(0);
		netKeyChanged("height");
		}
	//turn the motion on
	else if (msg == "on")
		{
		if (!active_)
			{
			active_ = true;
			netKeyChanged("active");
			}
		}
	//turn the motion off
	else if (msg == "off")
		{
		if (active_)
			{
			active_ = false;
			netKeyChanged("active");
			}
		}
	else
		surface::message(msg);
	}

void waves::app(void)
	{
	if (active_ && geode)
		{
		int numVerts;
		float sinX, cosX, sinY, cosY;
		currTime_ += ygWorld::FrameDeltaTime;
		float phase = currTime_ * speed_;
	
		numVerts = (int)vertices_[0] * (int)vertices_[1];
	
		for (int i = 0; i < numVerts; i++)
			{
			pfSinCos(verts[i][0] * width_ + phase, &sinX, &cosX);
			pfSinCos(verts[i][1] * width_ + phase, &sinY, &cosY);
			verts[i][2] = height_ * sinX * sinY;
	
			norms[i][0] = cosX;
			norms[i][1] = cosY;
			norms[i][2] = 1;
	
			float zu = cosX * sinY;
			float zv = sinX * cosY;
	
			pfVec3 u, v;
			float scale;
			scale = sqrt(1 + zu * zu);
			u[0] = 1.0 / scale;
			u[1] = 0.0;
			u[2] = zu / scale;
	
			scale = sqrt(1 + zv * zv);
			v[0] = 0.0;
			v[1] = 1.0 / scale;
			v[2] = zv / scale;
	
			norms[i].cross(u, v);
			norms[i].normalize();
			}
		}
	surface::app();
	}
