// Description: a tri-strip surface with a sinusoidal wave motion
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> wall and floor modes can be set for this geometry
//</ul>
//
// Category: Deprecated
// Author: Kyoung Park
// Revision: 11/01/01
//
#include <Performer/pf/pfGroup.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pr/pfMaterial.h> 
#include "ygWorld.h"
#include "water.h"

extern "C" ygNode* construct_water(const char* name,bool master) { return new water(name,master); }

water::water(const char* name,bool master) : ygTransform(name,master)
{
	setClassName("water");

	isWall_ = false;
	isFloor_ = false;
	state_ = INITIAL;
	startTime_ = 0.;
	duration_ = 0.;
	delay_ = 0.;
	phaseSpeed_ = 90.;
	rippleSize_ = 50.;
	rippleHeight_ = 1.;
	specularOn_ = false;
	color_.set(0., 0., 0., .6);
	len_.set(12., 12.);

	firstFrame_ = true;
}

water::~water(void)
{
}

void water::reset(void)
{
	duration_ = 0.0;
	state_ = INITIAL;

	ygTransform::reset();
}


void water::message(const ygMessage& msg)
{
	//set the surface color
	if (msg == "color") {
		msg.getVec4Args(color_);
	}
	//set the wave delay
	else if (msg == "delay") {
		delay_ = msg.floatArg(0);
	}
	//set the surface specuarity mode
	else if (msg == "specularity") {
		specularOn_ = msg.boolArg(0);
	}
	//set the wave speed
	else if (msg == "phaseSpeed") {
		phaseSpeed_ = msg.floatArg(0);
	}
	//set the wave width
	else if (msg == "rippleSize") {
		rippleSize_ = msg.floatArg(0);
	}
	//set the wave height
	else if (msg == "rippleHeight") {
		rippleHeight_ = msg.floatArg(0);
	}
	//set the size of the surface
	else if (msg == "size") {
		msg.getVec2Args(len_);
	}
	//set the wall intersection mode
	else if (msg == "wall") {
		isWall_ = msg.boolArg(0);
	}
	//set the floor intersection mode
	else if (msg == "floor") {
		isFloor_ = msg.boolArg(0);
	}
	//start the water motion
	else if (msg == "start") {
		state_ = START;
	}
	//reset the surface parameters
	else if (msg == "reset") {
		reset();
	}
	else
		ygTransform::message(msg);
}

pfGeode * water::createGeode(void)
{
	pfGeode *geode;
	pfVec3 *verts, *norms;
	pfVec4 *colors;
	pfMaterial *mat;
	int *lengths;
	int x, y, index;
	int numMesh = 40;

	verts = (pfVec3 *) pfMalloc(numMesh * numMesh * sizeof(pfVec3), pfGetSharedArena());
	norms = (pfVec3 *) pfMalloc(numMesh * numMesh * sizeof(pfVec3), pfGetSharedArena());
	colors = (pfVec4 *) pfMalloc(numMesh * numMesh * sizeof(pfVec4), pfGetSharedArena());

	//initialize surface points
	for (y = 0, index = 0; y < numMesh; y++) {
		for (x = 0; x < numMesh; x++, index++) {
			verts[index].set(len_[0] * x / numMesh - len_[0] / 2. + .5,
					 len_[1] * y / numMesh - len_[1] / 2. + .5, -.5);
			norms[index].set(0, 0, 1);
			colors[index].set(color_[0], color_[1], color_[2], color_[3]);
		}
	}

	// create geoset
	gset_ = new pfGeoSet;
	gset_->setPrimType(PFGS_TRISTRIPS);
	gset_->setNumPrims(numMesh - 1);

	lengths = (int *) pfMalloc((numMesh - 1) * sizeof(int), pfGetSharedArena());

	for (y = 0; y < numMesh - 1; y++)
		lengths[y] = numMesh * 2;

	gset_->setPrimLengths(lengths);

	ushort *ilist = (ushort *) pfMalloc(2 * numMesh * (numMesh - 1) * sizeof(ushort), pfGetSharedArena());

	//set up tristrip indexing
	index = 0;
	for (y = 0; y < numMesh - 1; y++) {
		for (x = 0; x < numMesh; x++) {
			ilist[index++] = x + y * numMesh;
			ilist[index++] = x + y * numMesh + numMesh;
		}
	}

	gset_->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, verts, ilist);
	gset_->setAttr(PFGS_NORMAL3, PFGS_PER_VERTEX, norms, ilist);
	gset_->setAttr(PFGS_COLOR4, PFGS_PER_VERTEX, colors, ilist);

	mat = new pfMaterial();
	if (specularOn_) {
		mat->setColor(PFMTL_SPECULAR, 1, 1, 1);
		mat->setShininess(16);
	}

	// create geostate
	pfGeoState *gstate = new pfGeoState;
	gstate->setMode(PFSTATE_TRANSPARENCY, PF_ON);
	gstate->setAttr(PFSTATE_FRONTMTL, mat);
	gstate->setMode(PFSTATE_ENLIGHTING, PF_ON);
	gset_->setGState(gstate);

	// create geode
	geode = new pfGeode;
	geode->addGSet(gset_);
	geode->setTravMask(PFTRAV_ISECT, 0,PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);

	return geode;
}

void water::app(void)
{
	//if first frame then create geometry
	if (firstFrame_) {
		pfnode()->addChild(createGeode());
		firstFrame_ = false;
	}

	//check if delay has passed
	if (state_ == START) {	
		startTime_ = pfGetTime();
		state_ = WAITING;
	}
	else if ((state_ == WAITING) && (duration_ <= delay_)) {
		duration_ = pfGetTime() - startTime_;
	}
	else if ((state_ == WAITING) && (duration_ > delay_)) {
		state_ = MOVING;
	}

	pfVec3 *verts, *norms;
	ushort *ilist;
	int numVerts;
	float sinX, cosX, sinY, cosY;
	float phase = pfGetTime() * phaseSpeed_;

	// change the geometry of the waves
	gset_->getAttrLists(PFGS_COORD3, (void **) &verts, &ilist);
	gset_->getAttrLists(PFGS_NORMAL3, (void **) &norms, &ilist);
	numVerts = pfGetSize(verts) / sizeof(pfVec3);

	for (int i = 0; i < numVerts; i++) {
		pfSinCos(verts[i][0] * rippleSize_ + phase, &sinX, &cosX);
		pfSinCos(verts[i][1] * rippleSize_ + phase, &sinY, &cosY);
		verts[i][2] = rippleHeight_ * sinX * sinY;

		norms[i][0] = cosX;
		norms[i][1] = cosY;
		norms[i][2] = 1;

		float zu = cosX * sinY;
		float zv = sinX * cosY;

		pfVec3 u, v;
		float scale;
		scale = sqrt(1 + zu * zu);
		u[0] = 1.0 / scale;
		u[1] = 0.0;
		u[2] = zu / scale;

		scale = sqrt(1 + zv * zv);
		v[0] = 0.0;
		v[1] = 1.0 / scale;
		v[2] = zv / scale;

		norms[i].cross(u, v);
		norms[i].normalize();
	}

	ygTransform::app();
}
