// Description: a switch that changes state if a user is in one or more spaces
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the default logic will turn the switch on when the viewer is inside any of the given spaces.
//<li> the switch logic can be overriden with the mode message
//</ul>
//
// Category: Selection
// Author: Dave Pape
// Revision: 11/01/01
//
#include <Performer/pf/pfSwitch.h>
#include <vector>
#include "ygNodeDB.h"
#include "ygSpace.h"
#include "ygVolume.h"
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "visibility.h"

using namespace std;

extern "C" ygNode* construct_visibility(const char* name,bool master) { return new visibility(name,master); }

enum {
	ALWAYS_ON,
	ALWAYS_OFF,
	NORMAL
	};

struct _visibilityPrivateData
	{
	pfSwitch * pfswitch;
	vector<ygSpace*> space;
	bool inside;
	int mode;
	ygString netString;
	};


visibility::visibility(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("visibility",true);
	p_ = new struct _visibilityPrivateData;
	p_->inside = true;
	p_->mode = NORMAL;
	p_->pfswitch = new pfSwitch;
	setPfNode(p_->pfswitch);
	addNetKey("space", &p_->netString, YG_NET_STRING);
	addNetKey("in", &p_->inside, YG_NET_BOOL);
	addNetKey("mode", &p_->mode, YG_NET_INT);
	}

visibility::~visibility(void)
	{
	delete p_;
	}


void visibility::reset(void)
	{
	//clear the local space array
	p_->space.clear();
	setInside(true);
	//if net master then clear net space array
	if (isNetMaster())
		{
		p_->netString.clear();
		netKeyChanged("space");
		}
	ygNode::reset();
	}


void visibility::message(const ygMessage& msg)
	{
	//make visible when inside space
	if (msg == "inside")
		setInside(true);
	//make visible when outside space
	else if (msg == "outside")
		setInside(false);
	//add a space node
	else if (msg == "space")
		addSpace(msg.args[0]);
	//set mode to normal, on or off
	else if (msg == "mode")
		{
		if (msg.args[0] == "on")
			p_->mode = ALWAYS_ON;
		else if (msg.args[0] == "off")
			p_->mode = ALWAYS_OFF;
		else
			p_->mode = NORMAL;
		netKeyChanged("mode");
		}
	else
		ygNode::message(msg);
	}


void visibility::setInside(bool val)
	{
	p_->inside = val;
	netKeyChanged("in");
	}


void visibility::addSpace(const ygString& name)
	{
	//find space node
	ygNode * spaceNode = ygNodeDB::find(name);
	//if node is found
	if ((spaceNode) && (spaceNode->isOfClass("ygSpace")))
		{
		//add to space array
		p_->space.push_back((ygSpace*)spaceNode);
		if (isNetMaster())
			{
			p_->netString += " ";
			p_->netString += name;
			netKeyChanged("space");
			}
		}
	}


void visibility::app(void)
	{
	ygNode * viewer = ygWorld::World->viewer();
	//if mode is off then switch off
	if (p_->mode == ALWAYS_OFF)
		p_->pfswitch->setVal(PFSWITCH_OFF);
	//else, if mode is on switch on
	else if ((!viewer) || (p_->space.size() == 0) || (p_->mode == ALWAYS_ON))
		p_->pfswitch->setVal(PFSWITCH_ON);
	//else determine if any space contains user
	else
		{
		bool viewerInside = false;
		int i;
		//for all spaces in array
		for (i=0; i < p_->space.size(); i++)
			{
			//if space contains user then set inside flag true
			if (p_->space[i]->contains(viewer->origin(p_->space[i])))
				{
				viewerInside = true;
				break;
				}
			}
		//if inside flag is true then switch on, else switch off
		if ((viewerInside && p_->inside) || ((!viewerInside) && (!p_->inside)))
			p_->pfswitch->setVal(PFSWITCH_ON);
		else
			p_->pfswitch->setVal(PFSWITCH_OFF);
		}
	ygNode::app();
	}


void visibility::acceptNetKey(const ygString& key)
	{
	if (key == "space")
		{
		ygString * spacename;
		int i=0;
		p_->space.clear();
		while (spacename = p_->netString.nextToken(" ",&i))
			{
			addSpace(*spacename);
			delete spacename;
			}
		}
	else if (key == "in")
		;
	else
		ygNode::acceptNetKey(key);
	}
