// Description: replaces the texture on objects below it with that of a rendered view
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the view is rendered from the position of the node
//<li> you must add the following to your RUN file in order to use this node:
//<ul>
//<li> setenv YG_PRELOAD_CLASSES viewTexture
//</ul>
//</ul>
//
// Category: Attributes
// Author: Alex Hill
//           05/10/02
// Revision: 01/15/03 Alex Hill - added save option
//
#include <Performer/pf/pfPipe.h>
#include <Performer/pf/pfPipeWindow.h>
#include <Performer/pf/pfEarthSky.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pf/pfGroup.h>
#include <ygWorld.h>
#include <ygNodeDB.h>
#include <ygNetKeys.h>
#include "viewTexture.h"

extern "C" ygNode* construct_viewTexture(const char* name,bool master) { return new viewTexture(name,master); }

typedef struct _channelData
	{
	pfTexture* texture;
	int xSize;
	int ySize;
	bool save;
	char file[20];
	} channelData_t;

int viewTexture::caveChannelNum = 0;

viewTexture::viewTexture(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("viewTexture",true);
	scenenode = NULL;
	gstate = NULL;
	//allocated channel data to share texture file
	channelData = (channelData_t*)pfMalloc(sizeof(channelData_t),pfGetSharedArena());
	channelData->texture = NULL;
	channelData->xSize = 0;
	channelData->ySize = 0;
	channelData->save = false;
	//create a performer scene node
	scene = new pfScene;
	//for each pipe do
	for (int i=0;i<pfGetMultipipe();i++)
		{
		//get a handle to the pipe
		pfPipe* pipe = pfGetPipe(i);
		//get a handle to the associated pipe window
		pfPipeWindow* pipeWindow = pipe->getPWin(0);
		//get a handle to the associated cave channel
		pfChannel* caveChannel = pipeWindow->getChan(caveChannelNum);
		//create a new channel for this view
		pfChannel* viewChannel = new pfChannel(pipe);
		//set the channel data for the view
		viewChannel->setChanData(channelData,sizeof(channelData_t));
		//pass the channel data for the view
		viewChannel->passChanData();
		//set the scene for the view
		viewChannel->setScene(scene);
		//set the earth sky model to that of the cave channel
		pfEarthSky* earthSky = caveChannel->getESky();
		viewChannel->setESky(earthSky);
		viewChannel->setTravMode(PFTRAV_CULL, PFCULL_VIEW|PFCULL_GSET);
		viewChannel->setTravFunc(PFTRAV_CULL, viewCullChannel);
		viewChannel->setTravFunc(PFTRAV_DRAW, viewDrawChannel);
		//insert view channel as the first channel rendered
		pipeWindow->insertChan(0,viewChannel);
		//add view channel to the list of view channels
		viewChannels.push_back(viewChannel);
		}
	caveChannelNum++;
	//distribute the name of the scene node
	addNetKey("scene",&scenename,YG_NET_STRING);
	//distribute the camera field of view
	addNetKey("fov",&fov,YG_NET_VEC2);
	//distribute the name of the node affected
    addNetKey("file", &filename, YG_NET_STRING);
	//distribute the name of file to save
    addNetKey("save", &savename, YG_NET_STRING);
	}

viewTexture::~viewTexture(void)
	{
	}

void viewTexture::reset(void)
	{
	//clear the scenename string
	scenename.clear();
	//clear the filename string
	filename.clear();
	//clear the savename string
	savename.clear();
	//set the scene to be the entire visible scene
    setScene("root");
	//set the field of view to 45.0 horizontal and 45.0 vertical
    setFOV(45.0,45.0);
	ygNode::reset();
	}


void viewTexture::message(const ygMessage& msg)
	{
	//load file to establish texture size and format
    if (msg == "file")
		{
		if (msg.args.size() == 1)
			setTexture(msg.stringArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set node name of the source scene if not the entire scene
    else if (msg == "scene")
		{
		if (msg.args.size() > 0)
			setScene(msg.stringArg(0));
		else
			setScene("root");
		}
	//set field of view in degrees (horizontal, vertical)
    else if (msg == "view")
		{
		if (msg.args.size() == 2)
			setFOV(msg.floatArg(0),msg.floatArg(1));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//save current view into an RGB image file
    else if (msg == "save")
		{
		if (msg.args.size() == 1)
			saveTexture(msg.stringArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygNode::message(msg);
	}


void viewTexture::viewCullChannel(pfChannel*, void*)
	{
    //invoke Performer cull-processing for this frame 
	pfCull();
	}


void viewTexture::viewDrawChannel(pfChannel* channel, void* data)
	{
	//cast channel data
	channelData_t* channelData = (channelData_t*)data;
	
	//make sure the back left buffer is used
	glDrawBuffer(GL_BACK_LEFT);

    //erase framebuffer and draw Earth-Sky model 
    channel->clear();
    
    //invoke Performer draw-processing for this frame 
    pfDraw();
	
    //subload frame buffer into the texture
    if (channelData->texture)
		{
		channelData->texture->subload(PFTEX_SOURCE_FRAMEBUFFER,NULL,0,0,0,0,0,channelData->xSize,channelData->ySize);		
    	if (channelData->save)
			{
			uint* image;
			int comp;
			int nr;
			glReadBuffer(GL_BACK_LEFT);
			channelData->texture->getImage(&image,&comp,NULL,NULL,&nr);
			glReadPixels(0,0,channelData->xSize,channelData->ySize,GL_RGB,GL_UNSIGNED_BYTE,image);
			channelData->texture->setImage(image,comp,channelData->xSize,channelData->ySize,nr);
			printf("savename %s\n",channelData->file);
			channelData->texture->saveFile(channelData->file);
			channelData->save = false;
			}
		}
	}
	
void viewTexture::setTexture(const ygString& file)
	{
    //create a new pfTexture
	channelData->texture = new pfTexture;
	//if the texture file can be loaded then
    if (channelData->texture->loadFile(file.c_str()))
		{
		//set the filename net key string
		filename = file;
		uint* image;
		int comp;
		int nr;
		//get image size
		channelData->texture->getImage(&image,&comp,&(channelData->xSize),&(channelData->ySize),&nr);
		//set the texture to allow subloading
		channelData->texture->setFormat(PFTEX_SUBLOAD_FORMAT, PF_ON);
		//initialize the origin of both the source and the destination texture
		channelData->texture->setLoadOrigin(PFTEX_ORIGIN_SOURCE,0,0);
		channelData->texture->setLoadOrigin(PFTEX_ORIGIN_DEST,0,0);
		//set the size of the texture to be loaded
		channelData->texture->setLoadSize(channelData->xSize,channelData->ySize);
		//set the load mode to extract from the frame buffer
		channelData->texture->setLoadMode(PFTEX_LOAD_SOURCE,PFTEX_SOURCE_FRAMEBUFFER);
		//set up texture filtering
		channelData->texture->setFilter(PFTEX_MINFILTER,PFTEX_BILINEAR);
		channelData->texture->setFilter(PFTEX_MAGFILTER,PFTEX_BILINEAR);
		//for each view channel, update the passed data
		for (int i=0;i<viewChannels.size();i++)		
			viewChannels[i]->passChanData();
		//indicate a change in the filename net key
		netKeyChanged("file");
		}
	}

void viewTexture::saveTexture(const ygString& file)
	{
	if (channelData->texture);
		{
		channelData->save = true;
		strcpy(channelData->file,file.c_str());
		//for each view channel, update the passed data
		for (int i=0;i<viewChannels.size();i++)		
			viewChannels[i]->passChanData();
		savename = file;
		netKeyChanged("save");
		}
	}

void viewTexture::setScene(const ygString& file)
	{
	//remove any previous node from under the scene node
	if (scenenode)
		scene->removeChild(scenenode->pfnode());
	//find the desired node for the scene
	scenenode = ygNodeDB::find(file);
	//if the node is found then
	if (scenenode)
		{
		//update the scenename net key string
		scenename = file;
		//add the associated performer node as a child to the scene
		scene->addChild(scenenode->pfnode());
		//indicate a change in the scenename net key
		netKeyChanged("scene");
		}
	else
		printf("%s: ERROR - scene node %s not found\n",name().c_str(),file.c_str());
	}

void viewTexture::setFOV(float horiz, float vert)
	{
	//update the field of view net key vector
	fov[0] = horiz;
	fov[1] = vert;
	//for each view channel, set the field of view
	for (int i=0;i<viewChannels.size();i++)		
    	viewChannels[i]->setFOV(horiz,vert);
	//indicate a change in the field of view net key
	netKeyChanged("fov");
	}

void viewTexture::setChildrenTextures(pfNode *node)
    {
    int i;
	//if node type is pfGeode then set geoset textures
    if (node->isOfType(pfGeode::getClassType()))
		{
		pfGeode *geode = (pfGeode *)node;
		//for each geode, call setGeosetTexture
		for (i=0; i < geode->getNumGSets(); i++)
			setGeosetTexture(geode->getGSet(i));
		}
	//else, if node type is pfGroup
    else if (node->isOfType(pfGroup::getClassType()))
		{
		pfGroup *group = (pfGroup *)node;
		//call setChildrenTextures recursively on all children
		for (i=0; i < group->getNumChildren(); i++)
			setChildrenTextures(group->getChild(i));
		}
    }

void viewTexture::setGeosetTexture(pfGeoSet *gset)
    {
	//find and set geostate texture to texture with view
	gstate = gset->getGState();
	gstate->setAttr(PFSTATE_TEXTURE,channelData->texture);
	}

void viewTexture::updateChannels(void)
	{
	//get node transformation matrix
	pfMatrix matrix;
	getTransform(matrix);
	int xs, ys;
	float l, r, b, t;
	float n, f;
	float right, top;
	//for each view channel
	for (int i=0;i<viewChannels.size();i++)
		{
		//get a handle to the pipe
		pfPipe* pipe = pfGetPipe(i);
		//get a handle to the pipe window
		pfPipeWindow* pipeWindow = pipe->getPWin(0);
		//get a handle to associated cave channel
		pfChannel* caveChannel = pipeWindow->getChan(caveChannelNum);
		//set the view channel view matrix to the node transformation matrix
   		viewChannels[i]->setViewMat(matrix);
		//get the cave channel near and far
		caveChannel->getNearFar(&n,&f);
		//set the view channel near and far
		viewChannels[i]->setNearFar(n,f);
		//get the cave channel size
		caveChannel->getSize(&xs,&ys);
		//get the cave channel viewport
		caveChannel->getViewport(&l,&r,&b,&t);
		//calculate the required viewport for given view texture size
		right = (((float)channelData->xSize)/((float)xs))*(r - l);
		top = (((float)channelData->ySize)/((float)ys))*(t - b);
		//set the view channel viewport
		viewChannels[i]->setViewport(0.0,right,0.0,top);
		}
	}

void viewTexture::app(void)
	{
	//if texture has been loaded then 
	if (channelData->texture)
		{
		//if geometry changed then find textures and replace with view texture
	    if (pfChanged())
			setChildrenTextures(pfnode());
		//update channel viewports
		updateChannels();
		}
    ygNode::app();
	}
	
void viewTexture::acceptNetKey(const ygString& key)
	{
	if (key == "file")
		setTexture(filename);
	else if (key == "scene")
		setScene(scenename);
	else if (key == "fov")
		setFOV(fov[0],fov[1]);
	else if (key == "save")
		saveTexture(savename);
	else
		ygNode::acceptNetKey(key);
	}
