// Description: a navigator that takes a direction and rotation vector
//
//<b>notes:</b>
//<ul>
//<li> all parent node messages are inherited
//<li> see reset method for default settings
//<li> a direction of (0,1,0) will allways move in the direction of the front wall
//</ul>
//
// Category: User
// Author: Alex Hill
// Revision: 12/05/03
//
#include <ygWorld.h>
#include <ygNodeDB.h>
#include <iostream.h>
#include "vectorNavigator.h"

extern "C" ygNode* construct_vectorNavigator(const char* name,bool master) { return new vectorNavigator(name,master); }

vectorNavigator::vectorNavigator(const char* name,bool master) : ygCAVENavigator(name,master)
	{
	setClassName("vectorNavigator");
	direction.set(0,0,0);
	rotation.set(0,0,0);
	}

vectorNavigator::~vectorNavigator(void)
	{
	}

void vectorNavigator::reset(void)
	{
	//set directon vector to 0,0,0
	direction.set(0,0,0);
	//set rotation vector to 0,0,0
	rotation.set(0,0,0);
	ygCAVENavigator::reset();
	}

void vectorNavigator::message(const ygMessage& msg)
	{
	//
	if (msg == "direction")
		{
		if (msg.args.size() == 3)
			{
			pfVec3 dir;
			msg.getVec3Args(dir);
			setDirection(dir);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//
	else if (msg == "rotation")
		{
		if (msg.args.size() == 3)
			{
			pfVec3 rot;
			msg.getVec3Args(rot);
			setRotation(rot);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else if (msg == "orientation")
		{
		if (msg.args.size() == 3)
			{
			pfVec3 ori;
			msg.getVec3Args(ori);
			data().setOrientation(ori);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygCAVENavigator::message(msg);
	}

void vectorNavigator::setDirection(const pfVec3& val)
	{
	direction = val;
	}

void vectorNavigator::setRotation(const pfVec3& val)
	{
	rotation = val;
	}

void vectorNavigator::app(void)
	{
	//process direction and orientation input
	checkInput();
	//if fly mode is false then adjust height to any ground collision
	if (!fly_)
		checkGround();
	//if collision detection is on then test for object collisions
	if (collide_)
		checkCollision();
	ygNavigator::app();
	}


void vectorNavigator::checkInput(void)
	{
	//extract X and Z vector components
	pfVec3 rot,ori;
	ori = data().orientation();
	rot[0] = rotation[2];
	rot[1] = 0.0;
	rot[2] = -rotation[0];
	pfMatrix mat;
	mat.makeEuler(ori[2],ori[0],ori[1]);
	pfVec3 currDir(0,1,0),newDir(0,1,0);
	currDir.xformVec(currDir,mat);
	//add X and Z components to forward vector (0,1,0)
	newDir += rot * (rotSpeed_/90.0) * ygWorld::FrameDeltaTime;
	newDir.normalize();
	//transform direction vector by current navigation matrix and normalize
	newDir.xformVec(newDir,mat);
	pfMatrix headingPitch;
	headingPitch.makeVecRotVec(currDir,newDir);
	pfMatrix newMat;
	//multiply existing navigation matrix by new heading and pitch matrix
	newMat = mat * headingPitch;
	float newRoll;
	newRoll = rotation[1] * rotSpeed_ * ygWorld::FrameDeltaTime;
	pfMatrix roll;
	//use Y vector component to create rotation matrix
	roll.makeRot(newRoll,0,1,0);
	//multiply existing navigation matrix by new roll matrix
	newMat *= roll;
	//calculate translation component
	pfVec3 dir, pos;
	pos = data().position();
	dir = direction;
	//if fly mode is false then zero the Z component
	if (!fly_)
		dir[PF_Z] = 0;
	//transform the direction vector by the user matrix
	dir.xformPt(dir,mat);
	//multiply existing navigation matrix by new translation matrix
	pos += dir * transSpeed_ * ygWorld::FrameDeltaTime;
	newMat.postTrans(newMat,pos[0],pos[1],pos[2]);
	//update navigation coordinates
	pfCoord coord;
	newMat.getOrthoCoord(&coord);
	data().setOrientation(coord.hpr[1],coord.hpr[2],coord.hpr[0]);
	data().setPosition(coord.xyz[0],coord.xyz[1],coord.xyz[2]);
	}






















