// Description: a variable size array of values
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node generates an event when any one of the array values changes
//<li> any other event generated by this node will include value arguments
//<li> the reference message is used to reference other value nodes
//</ul>
//
// Category: Math
// Author: Alex Hill
//           11/01/01
// Revision: 01/15/03 Alex Hill - added event message
//
#include "ygNodeDB.h"
#include "value.h"

extern "C" ygNode* construct_value(const char* name,bool master) { return new value(name,master); }

value::value(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("value");
	values = new float;
	lastValues = new float;
	values[0] = 0.0;
	lastValues[0] = 0.0;
	size = 1;
	deltas = new float;
	deltas[0] = 1.0;
	numDeltas = 1;
	minimums = NULL;
	maximums = NULL;
	numMins = 0;
	numMaxs = 0;
	reference = NULL;
	}
	
value::~value(void)
	{
	}

void value::reset(void)
	{
	//set size to 1
	setValueSize(1);
	//initialize value to 0.0
	values[0] = 0.0;
	lastValues[0] = 0.0;
	//set delta size to 1
	setSize(deltas,1,numDeltas);
	//initialize delta value to 1.0
	deltas[0] = 1.0;
	numDeltas = 1;
	//set minimums size to 0
	setSize(minimums,0,numMins);
	numMins = 0;
	//set maximums size to 0
	setSize(minimums,0,numMaxs);
	numMaxs = 0;
	reference = NULL;
	//set integer flag to false
	integer = false;
	ygNode::reset();
	}

void value::message(const ygMessage& msg)
	{
	//make this node an instance of another node
	if (msg == "reference")
		{
		if (msg.args.size() > 0 && className() == "value")
			{
			ygNode* node = ygNodeDB::find(msg.args[0]);
			if (node && node->isOfClass("value"))
				reference = (value*)node;
			else
				reference = NULL;
			}
		else
			reference = NULL;
		}
	//else, if referenced, send the message to the reference node
	else if (reference)
		{
		reference->message(msg);
		}
	//set the value of a single array element
	else if (msg == "set" || msg == "value")
		{
		if (msg.args.size() > 0)
			{
			float newValue = msg.floatArg(0);
			if (msg.args.size() > 1) 
				{
				int element = msg.intArg(1);
				if (element >= 0)
					{	
					if (element >= size)
						setValueSize(element+1);
					values[element] = newValue;
					}
				else
					msg.error(name(),"(index out of range)");
				}
			else
				{
				if (size > 0)
					values[0] = newValue;
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the values of all array elements
	else if (msg == "values")
		{
		if (msg.args.size() > 0)
			{
			int newSize = msg.args.size();
			if (newSize >= 0)
				{
				setValueSize(newSize);
				for (int i=0;i<size;i++) 
					values[i] = msg.floatArg(i);
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the size of the array
	else if (msg == "size")
		{
		if (msg.args.size() == 1)
			{
			int newSize = msg.intArg(0);
			if (newSize >= 0)
				setValueSize(newSize);
			else
				msg.error(name(),"(index out of range)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the value of a single delta element
	else if (msg == "delta")
		{
		if (msg.args.size() > 0)
			{
			float newValue = msg.floatArg(0);
			if (msg.args.size() > 1) 
				{
				int element = msg.intArg(1);
				if (element >= 0)
					{	
					if (element >= numDeltas)
						{
						setSize(deltas,element+1,numDeltas);
						numDeltas = element+1;
						}
					deltas[element] = newValue;
					}
				else
					msg.error(name(),"(index out of range)");
				}
			else
				deltas[0] = newValue;
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the values of all delta array elements
	else if (msg == "deltas")
		{
		if (msg.args.size() > 0)
			{
			int newSize = msg.args.size();
			if (newSize >= 0)
				{
				setSize(deltas,newSize,numDeltas);
				numDeltas = newSize;
				for (int i=0;i<numDeltas;i++) 
					deltas[i] = msg.floatArg(i);
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the value of single minimum clamp
	else if (msg == "minimum")
		{
		if (msg.args.size() > 0)
			{
			float newValue = msg.floatArg(0);
			if (msg.args.size() > 1) 
				{
				int element = msg.intArg(1);
				if (element >= 0)
					{	
					if (element >= numMins)
						{
						setSize(minimums,element+1,numMins);
						numMins = element+1;
						}
					minimums[element] = newValue;
					}
				else
					msg.error(name(),"(index out of range)");
				}
			else
				{
				if (numMins == 0)
					{
					setSize(minimums,1,numMins);
					numMins = 1;
					}
				minimums[0] = newValue;
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the values of all minimum array elements
	else if (msg == "minimums")
		{
		if (msg.args.size() > 0)
			{
			int newSize = msg.args.size();
			if (newSize >= 0)
				{
				setSize(minimums,newSize,numMins);
				numMins = newSize;
				for (int i=0;i<numMins;i++) 
					minimums[i] = msg.floatArg(i);
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the value of single maximum clamp
	else if (msg == "maximum")
		{
		if (msg.args.size() > 0)
			{
			float newValue = msg.floatArg(0);
			if (msg.args.size() > 1) 
				{
				int element = msg.intArg(1);
				if (element >= 0)
					{	
					if (element >= numMaxs)
						{
						setSize(maximums,element+1,numMaxs);
						numMaxs = element+1;
						}
					maximums[element] = newValue;
					}
				else
					msg.error(name(),"(index out of range)");
				}
			else
				{
				if (numMaxs == 0)
					{
					setSize(maximums,1,numMaxs);
					numMaxs = 1;
					}
				maximums[0] = newValue;
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the values of all maximum array elements
	else if (msg == "maximums")
		{
		if (msg.args.size() > 0)
			{
			int newSize = msg.args.size();
			if (newSize >= 0)
				{
				setSize(maximums,newSize,numMaxs);
				numMaxs = newSize;
				for (int i=0;i<numMaxs;i++) 
					maximums[i] = msg.floatArg(i);
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//decrement all or a single element by delta
	else if (msg == "decrement")
		{
		if (msg.args.size() > 0)
			{
			int element = msg.intArg(0);
			if (element >= 0 && element < size)
				{
				if (element >= numDeltas)
					values[element] -= deltas[numDeltas];
				else
					values[element] -= deltas[element];
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			{
			for (int i=0;i<size;i++)
				{
				if (i >= numDeltas)
					values[i] -= deltas[numDeltas];
				else
					values[i] -= deltas[i];
				}
			}
		}
	//increment all or a single element by delta
	else if (msg == "increment")
		{
		if (msg.args.size() > 0)
			{
			int element = msg.intArg(0);
			if (element >= 0 && element < size)
				{
				if (element >= numDeltas)
					values[element] += deltas[numDeltas-1];
				else
					values[element] += deltas[element];
				}
			else
				msg.error(name(),"(index out of range)");
			}
		else
			{
			for (int i=0;i<size;i++)
				{
				if (i >= numDeltas)
					values[i] += deltas[numDeltas-1];
				else
					values[i] += deltas[i];
				}
			}
		}
	//set if the value should be rounded to the nearest integer
	else if (msg == "integer")
		{
		if (msg.args.size() > 0)
			integer = msg.boolArg(0);
		else
			integer = true;
		}
	//generate an event with associated arguments
	else if (msg == "event")
		{
		if (msg.args.size() > 0)
			{
			ygString eventName = msg.args[0];
			ygString args;
			char index[32];
			//for each value in the array
			for (int i=0;i<size;i++)
				{
				if (i == 0)
					{
					args += "value=";
					if (integer)
						sprintf(index,"%d",(int)values[i]);
					else
						sprintf(index,"%f",values[i]);
					args += index;
					}
				args += " ";
				args += "value";
				sprintf(index,"%d",i);
				args += index;
				args += "=";
				if (integer)
					sprintf(index,"%d",(int)values[i]);
				else
					sprintf(index,"%f",values[i]);
				args += index;
				}
			if (msg.args.size() == 1)
				{
				//add $value and $valueX arguments to each event
				eventOccurred(eventName,args);
				}
			else
				{
				for (int i=1;i<msg.args.size();i++)
					{
					args += " ";
					args += msg.args[i];
					}
				//add all $value arguments to each event plus any arguments
				eventOccurred(eventName,args);
				}
			}
		}
	else
		ygNode::message(msg);
	}

void value::setValueSize(int newSize)
	{
	setSize(values,newSize,size);
	setSize(lastValues,newSize,size);
	size = newSize;
	}
	
void value::setSize(float*& array, int newSize, int oldSize, float initValue)
	{
	//if size has changed
	if (newSize != oldSize)
		{
		float* oldArray = array;
		if (newSize > 0)
			{
			//allocate new memory
			array = new float[newSize];
			//if old size is 0 then fill with initial values
			if (oldSize == 0)
				{
				for (int i=0;i<newSize;i++) 
					array[i] = initValue;
				}
			//else, if new size is larger then pad with initial values
			else if (newSize > oldSize)
				{
				for (int i=0;i<oldSize;i++) 
					array[i] = oldArray[i];
				for (int i=oldSize;i<newSize;i++) 
					array[i] = initValue;
				delete oldArray;
				}
			//else, update with existing values
			else
				{
				for (int i=0;i<newSize;i++) 
					array[i] = oldArray[i];
				delete oldArray;
				}
			}
		else
			{
			array = NULL;
			delete oldArray;
			}
		}
	}

bool value::isChanged(void) 
	{ 
	bool changed = false; 
	for (int i=0;i<size;i++)
		{
		if (values[i] != lastValues[i])
			changed = true;
		}
	return changed;
	}
	
void value::clampValues(void)
	{
	//for each value in the array
	for (int i=0;i<size;i++)
		{
		//if exceeding a limit then clamp
		if (numMins > 0)
			{
			int minIndex = i;
			if (minIndex >= numMins)
				minIndex = numMins-1;
			if (values[i] < minimums[minIndex])
				values[i] = minimums[minIndex];
			}
		if (numMaxs > 0)
			{
			int maxIndex = i;
			if (maxIndex >= numMaxs)
				maxIndex = numMaxs-1;
			if (values[i] > maximums[maxIndex])
				values[i] = maximums[maxIndex];
			}
		if (integer)
			values[i] = rint(values[i]);
		}
	}

void value::app(void)
	{
	//clamp values
	clampValues();
	//if changed then generate event
	if (isChanged())
		{
		ygString args;
		char index[32];
		//for each value in the array
		for (int i=0;i<size;i++)
			{
			if (i == 0)
				{
				args += "value=";
				if (integer)
					sprintf(index,"%d",(int)values[i]);
				else
					sprintf(index,"%f",values[i]);
				args += index;
				}
			args += " ";
			args += "value";
			sprintf(index,"%d",i);
			args += index;
			args += "=";
			if (integer)
				sprintf(index,"%d",(int)values[i]);
			else
				sprintf(index,"%f",values[i]);
			args += index;
			}
		//generate changed event with $value, $valueX where X=index
		eventOccurred("changed",args);
		//for each value in the array
		for (int i=0;i<size;i++)
			lastValues[i] = values[i];
		}
	ygNode::app();
	}


void value::appTraverse(void)
	{
	//start timing
	timingBegin();
	//call appTraverse recursively on all children
	for (int i=0; i < numChildren(); i++)
		child(i)->appTraverse();
	//execute app method
	app();
	//end timing
	timingEnd();
	}

float value::getValue(int element) const
	{
	if (!reference)
		{
		if (element < size)
			return values[element];
		else
			return 0.0;
		}
	else
		return reference->getValue(element);
	}

float value::getLastValue(int element) const
	{
	if (!reference)
		{
		if (element < size)
			return lastValues[element];
		else
			return 0.0;
		}
	else
		return reference->getLastValue(element);
	}

void value::setValue(const float& iValue, int element)
	{
	if (!reference)
		{ 
		if (element >= size)
			setValueSize(element+1);
		values[element] = iValue;
		}
	else
		reference->setValue(iValue,element);
	}

int value::getSize(void) const
	{
	if (!reference)
		return size;
	else
		return reference->getSize();
	}


