// Description: an interpolate node with transformed position and orientation values
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the array size is 6 with the following values
//<ul>
//<li> value0 is the X translation
//<li> value1 is the Y translation
//<li> value2 is the Z translation
//<li> value3 is the X orientation
//<li> value4 is the Y orientation
//<li> value5 is the Z orientation
//</ul>
//<li> this node is useful for constraining and reporting transform position and orientation
//<li> this node is useful for grabbing and moving when used in conjunction with a <a href="wandTrigger.html">wandTrigger</a> node
//</ul>
//
// Category: Math
// Author: Alex Hill
// Revision: 06/01/02
//
#include <Performer/pf/pfDCS.h>
#include <ygNodeDB.h>
#include <ygNetKeys.h>
#include <ygWorld.h>
#include "valuator.h"

extern "C" ygNode* construct_valuator(const char* name,bool master) { return new valuator(name,master); }

#define COORD_NETKEY "x"

valuator::valuator(const char* name,bool master) : interpolate(name,master)
	{
	setClassName("valuator",true);
	dcs = new pfDCS;
	setPfNode(dcs);
	node = NULL;
	//create a simple coordinate network key
	addNetKey(COORD_NETKEY, &netCoord, YG_NET_COORD);
	//create an unreliable key for coordinate values
	unreliableKey(COORD_NETKEY);
	}

valuator::~valuator(void)
	{
	}


void valuator::reset(void)
	{
	//if net master then 
	if (isNetMaster())
		{
		//reset matrix to identity
		deltaMatrix.makeIdent();
		//reset node pointer
		node = NULL;
		//reset grab flag
		grab = false;
		//set position to (0,0,0)
		netCoord.xyz.set(0,0,0);
		//set orientation to (0,0,0)
		netCoord.hpr.set(0,0,0);
		//set update interval to 0.0
		updateInterval = 0;
		//set last update time to 0.0
		lastUpdate = 0;
		//set matrix to identity
		pfMatrix mat;
		mat.makeIdent();
		setMatrix(mat);
		}
	interpolate::reset();
	setValueSize(6);
	}


void valuator::message(const ygMessage& msg)
	{
	//toggle the grab/drop state with respect to a node
	if (msg == "toggle")
		{
		if (msg.args.size() == 1)
			{
			if (nodeName == msg.args[0] && node)
				node = NULL;
			else
				{
				nodeName = msg.args[0];
				node = ygNodeDB::find(nodeName);
				if (node)
					grab = true;
				else
					msg.error(name(),"(node not found)");
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
    	}
	//grab a node and move with repect to that node
	else if (msg == "grab")
		{
		if (msg.args.size() < 2)
			{
			if (msg.args.size() == 1)
				{
				nodeName = msg.args[0];
				node = ygNodeDB::find(nodeName);
				if (node)
					grab = true;
				else
					msg.error(name(),"(node not found)");
				}
			else
				{
				if (node)
					grab = true;
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
    	}
	//drop connection with a node or any node grabbed
	else if (msg == "drop")
		{
		if (msg.args.size() < 2)
			{
			if (msg.args.size() == 1)
				{
				ygString dropName = msg.args[0];
				if (dropName == nodeName)
					node = NULL;
				}
			else
				node = NULL;
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the translational component
	else if (msg == "position")
		{
		pfVec3 v;
		if (msg.getVec3Args(v) && size >= 3)
			{
			for (int i=0; i<3; i++)
				values[i] = startValues[i]+v[i]*(endValues[i]-startValues[i]);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the orientation component
	else if (msg == "orientation")
		{
		pfVec3 v;
		if (msg.getVec3Args(v) && size >= 6)
			{
			for (int i=3; i<6; i++)
				values[i] = startValues[i]+v[i]*(endValues[i]-startValues[i]);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the key update interval for remote users
	else if (msg == "updateInterval")
		{
		if (msg.args.size() > 0)
			updateInterval = msg.floatArg(0);
		}
	else
		interpolate::message(msg);
	}

void valuator::app(void)
	{
	//if net master
	if (isNetMaster())
		{
		calculateTransform();
		value::app();
		updateTransform();
		}
	}

void valuator::calculateTransform(void)
	{
	//if the node has been found
	if (node)
		{
		if (grab)
			{
			/*calculate the relative matrix between the node and this object
			  using the following formula:
									  -1
			  delatMatrix = M      * M
							 self     node
			*/
			pfMatrix node_matrix;
			node->getTransform(node_matrix);
			
			pfMatrix inverse_node_matrix;
			inverse_node_matrix.invertOrtho(node_matrix);
	
			pfMatrix self_matrix;
			getTransform(self_matrix);
			
			// Set this instance as "grabbed".
			deltaMatrix = self_matrix * inverse_node_matrix;
			grab = false;
			}
		/*move this matrix with respect to the node
									  -1
		   M = deltaMatrix * M     * M
							  node    parent
		*/

		//get the node's transformation matrix
		pfMatrix node_matrix;
		node->getTransform(node_matrix);

		//get our object's transformation matrix relative to the world
		pfMatrix parent_matrix;
		getParentTransform(parent_matrix);
		parent_matrix.invertOrtho(parent_matrix);

		//set the new transformation and return
		pfMatrix self_matrix = deltaMatrix * node_matrix * parent_matrix;
		pfCoord coord;
		self_matrix.getOrthoCoord(&coord);
		//calculate interpolated array values
		if (size >= 3)
			{
			values[0] = startValues[0] + coord.xyz[0]*(endValues[0]-startValues[0]);
			values[1] = startValues[1] + coord.xyz[1]*(endValues[1]-startValues[1]);
			values[2] = startValues[2] + coord.xyz[2]*(endValues[2]-startValues[2]);
			}
		if (size >= 6)
			{
			values[3] = startValues[3] + coord.hpr[1]*(endValues[3]-startValues[3]);
			values[4] = startValues[4] + coord.hpr[2]*(endValues[4]-startValues[4]);
			values[5] = startValues[5] + coord.hpr[0]*(endValues[5]-startValues[5]);
			}
		}
	}
	
void valuator::updateTransform(void)
	{
	float x, y, z;
	float x_ang, y_ang, z_ang;
	if (size >= 3)
		{
		x = (values[0] - startValues[0])/(endValues[0]-startValues[0]);
		y = (values[1] - startValues[1])/(endValues[1]-startValues[1]);
		z = (values[2] - startValues[2])/(endValues[2]-startValues[2]);
		setPosition(x,y,z);
		}
	if (size >= 6)
		{
		x_ang = (values[3] - startValues[3])/(endValues[3]-startValues[3]);
		y_ang = (values[4] - startValues[4])/(endValues[4]-startValues[4]);
		z_ang = (values[5] - startValues[5])/(endValues[5]-startValues[5]);
		setOrientation(x_ang,y_ang,z_ang);
		}
	//if update interval has passed then update key
	if ((ygWorld::FrameTime - lastUpdate) > updateInterval)
		{
		pfMatrix matrix;
		getMatrix(matrix);
		pfCoord coord;
		matrix.getOrthoCoord(&coord);
		if (coord.xyz != netCoord.xyz || coord.hpr != netCoord.hpr)
			{
			netCoord = coord;
			netKeyChanged(COORD_NETKEY);
			lastUpdate = ygWorld::FrameTime;
			}
		}
	}

/***** Net functions *****/

void valuator::acceptNetKey(const ygString& key)
	{
	if (key == COORD_NETKEY)
		{
		setPosition(netCoord.xyz);
		setOrientation(netCoord.hpr[1],netCoord.hpr[2],netCoord.hpr[0]);
		}
	else
		ygNode::acceptNetKey(key);
	}

/*****  DCS functions *****/

void valuator::getMatrix(pfMatrix& mat) const
	{
	dcs->getMat(mat);
	}


void valuator::setMatrix(pfMatrix& mat)
	{
	dcs->setMat(mat);
	}

void valuator::setPosition(float x,float y,float z)
	{
	dcs->setTrans(x,y,z);
	}
	
void valuator::setOrientation(float x,float y,float z)
	{
	dcs->setRot(z,x,y);
	}
