// Description: sets up an independent Performer switch state for each user
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> creates a switch using a Performer pfSwitch
//<li> this node does not stop YG application processing from occuring
//<li> this node is frequently used to toggle geometry on and off
//</ul>
//
// Category: Selection
// Author: Alex Hill
// Revision: 10/01/02
//
#include <ygWorld.h>
#include <ygNodeDB.h>
#include <ygNetKeys.h>
#include <ygCAVEViewer.h>
#include "userSwitch.h"

extern "C" ygNode* construct_userSwitch(const char* name,bool master) { return new userSwitch(name,master); }

userSwitch::userSwitch(const char* name,bool master) : ygNode(name,master,false)
	{
	setClassName("userSwitch",true);
	//create pfSwitch for the pfnode
	sw = new pfSwitch;
	sw->setVal(PFSWITCH_ON);
	setPfNode(sw);
	addNetKey("users",&users,YG_NET_STRING);
	}

userSwitch::~userSwitch(void)
	{
	}

void userSwitch::reset(void)
	{
	users.clear();
	sw->setVal(PFSWITCH_ON);
	netKeyChanged("users");
	ygNode::reset();
	}

void userSwitch::message(const ygMessage& msg)
	{
	//set the switch to on for all or a single user
	if (msg == "on")
		{
		if (msg.args.size() < 2)
			{
			if (msg.args.size() == 1)
				{
				ygString user = msg.args[0];
				if (on(user))
					{
					setState(users);
					netKeyChanged("users");
					}
				}
			else
				{
				users.clear();
				sw->setVal(PFSWITCH_ON);
				netKeyChanged("users");
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the switch to off for all or a single user
	else if (msg == "off")
		{
		if (msg.args.size() < 2)
			{
			if (msg.args.size() == 1)
				{
				ygString user = msg.args[0];
				if (off(user))
					{
					setState(users);
					netKeyChanged("users");
					}
				}
			else
				{
				users.clear();
				//for each user in the world
				for (int i=0; i < ygWorld::World->numUsers(); i++)
					{
					ygNode* node = (ygNode*)ygWorld::World->user(i);
					users += node->name();
					users += " ";
					}
				sw->setVal(PFSWITCH_OFF);
				netKeyChanged("users");
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//toggle the state of the switch for all or a single user
	else if (msg == "toggle")
		{
		if (msg.args.size() < 2)
			{
			if (msg.args.size() == 0)
				toggle(msg.args[0]);
			else
				{
				//for each user in the world
				for (int i=0; i < ygWorld::World->numUsers(); i++)
					{
					ygNode* node = (ygNode*)ygWorld::World->user(i);
					toggle(node->name());
					}
				setState(users);
				netKeyChanged("users");
				}
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygNode::message(msg);
	}

bool userSwitch::on(const ygString& user)
	{
	//remove this user from the users list
	ygString *curUser;
	int i=0;
	bool removed = false;
	ygString newUsers;
	while (curUser = users.nextToken(" ",&i))
			{
			if (*curUser == user)
				removed = true;
			else
				{
				newUsers += *curUser;
				newUsers += " ";
				}
			delete curUser;
			}
	if (removed)
		users = newUsers;
	return removed;
	}

bool userSwitch::off(const ygString& user)
	{
	//add this user to the users list
	ygString *curUser;
	int i=0;
	bool added = true;
	ygString newUsers;
	while (curUser = users.nextToken(" ",&i))
		{
		if (*curUser == user)
			added = false;
		delete curUser;
		}
	if (added)
		{
		users += user;
		users += " ";
		}
	return added;
	}

void userSwitch::toggle(const ygString& user)
	{
	if (!off(user))
		on(user);
	}

void userSwitch::setState(const ygString& users)
	{
	ygString *user;
	ygNode* node;
	ygCAVEViewer* viewer;
	viewer = (ygCAVEViewer*)ygWorld::World->viewer();
	int i=0;
	float state = PFSWITCH_ON;
	while (user = users.nextToken(" ",&i))
			{
			node = ygNodeDB::find(*user);
			if (node && (node->isNetMaster() || viewer->isCloned(node)))
				state = PFSWITCH_OFF;
			delete user;
			}
	sw->setVal(state);
	}

void userSwitch::acceptNetKey(const ygString& key)
	{
	if (key == "users")
		setState(users);
	else
		ygNode::acceptNetKey(key);
	}
