// Description: a space that generates position events in the presence of a user head
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> for volume messages refer to the <a href="ygSpace.html">ygSpace</a> node
//<li> this node requires the user have a <a href="ygCAVEHead.html">ygCAVEHead</a> node
//<li> the local mode determines if the user position is local to the object or relative to world coordinates
//</ul>
//
// Category: Trigger
// Author: Alex Hill
// Revision: 05/10/02
//
#include <ygUser.h>
#include <ygWorld.h>
#include <ygUtil.h>
#include "userPosition.h"

extern "C" ygNode* construct_userPosition(const char* name,bool master) { return new userPosition(name,master); }

userPosition::userPosition(const char* name,bool master) : ygSpace(name,master)
	{
	setClassName("userPosition");
	debugFlag("inside", &debugInside);
	}

userPosition::~userPosition(void)
	{
	}

void userPosition::reset(void)
	{
	//clear set of users previously inside
	prevInside.clear();
	//set reporting mode to local
	local = true;
	lastFrame = ygWorld::FrameNumber;
	ygSpace::reset();
	}

void userPosition::message(const ygMessage& msg)
	{
	//set position reporting mode
	if (msg == "local")
		{
		if (msg.args.size() > 0)
			local = msg.boolArg(0);
		else
			local = true;
		}
	else
		ygSpace::message(msg);
	}

void userPosition::app(void)
	{
	set<ygUser*> inside;
	int i;
	if (lastFrame < (ygWorld::FrameNumber-1))
		prevInside.clear();
	lastFrame = ygWorld::FrameNumber;
	//for each user in the world
	for (i=0; i < ygWorld::World->numUsers(); i++)
		{
		ygUser * user = ygWorld::World->user(i);
		//if user has a head then
		if (user->head())
			{
			//if head is within the space then
			pfVec3 position = user->head()->origin(this);
			if (contains(position))
				{
				if (debugInside)
					cout << name() << ": " << user->name() << " is inside\n";
				//add to set of users inside
				inside.insert(inside.begin(),user);
				ygString args("user=");
				args += user->name();
				if (!local)
					position = user->head()->origin();
				char value[32];
				args += " xpos=";
				sprintf(value,"%f",position[0]);
				args += value;
				args += " ypos=";
				sprintf(value,"%f",position[1]);
				args += value;
				args += " zpos=";
				sprintf(value,"%f",position[2]);
				args += value;
				//a user is inside the space
				eventOccurred("inside",args);
				//if inside but not previously then generate enter event
				if (prevInside.find(user) == prevInside.end())
					{
					//a user has entered the space
					eventOccurred("enter",args);
					}
				}
			}
		}
	set<ygUser*>::const_iterator iter;
	//for each user previously inside
	for (iter = prevInside.begin(); iter != prevInside.end();++iter)
		{
		//if not currently inside then generate exit event
		if (inside.find(*iter) == inside.end())
			{
			ygString args("user=");
			args += (*iter)->name();
			//a user has exited the space
			eventOccurred("exit",args);
			}
		}
	//if not previously empty and now empty generate empty event
	if ((prevInside.size() > 0) && (inside.size() == 0))
		{
		//all users have exited the space
		eventOccurred("empty");
		}
	//if previously empty and now not empty generate firstenter event
	if ((prevInside.size() == 0) && (inside.size() > 0))
		{
		ygString args("user=");
		args += (*(inside.begin()))->name();
		//a user has entered a previously empty space
		eventOccurred("firstenter",args);
		}
	prevInside = inside;
	ygSpace::app();
	}
