// Description: creates a positional tone at a specified frequency within a space
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the volume of a sound is the region in which the sound will play at 100% amplitude(loudness)
//<li> falloffDistance is a region outside the volume in which the amplitude is diminished as the user moves further away from the source(volume)
//<li> the sound server, snerd, must be run on the a local or remote machine
//<li> sounds under a switch will turn off but must be restarted when the switch is toggled back on
//</ul>
//
// Category: Sound
// Author: Alex Hill
// Revision: 08/01/03
//
#include <pfcave.h>
#include <bergenTone.h>
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "tone.h"

using namespace std;

extern "C" ygNode* construct_tone(const char* name,bool master) { return new tone(name,master); }

struct _tonePrivateData
	{
	float frequency;
	bergenTone * sample;
	bool isPlaying;
	float curAmplitude;
	pfVec3 currPosition;
	float netFrequency;
	float updateInterval, lastUpdate;
	bool netPlay;
	};


tone::tone(const char* name,bool master) : ygSoundServer(name,master)
	{
	setClassName("tone",true);
	p_ = new struct _tonePrivateData;
	p_->sample = NULL;
	p_->curAmplitude = 0;
	p_->updateInterval = 0.1;
	p_->lastUpdate = 0;
	p_->isPlaying = false;
	addNetKey("play", &p_->netPlay, YG_NET_BOOL);
	addNetKey("freq", &p_->netFrequency, YG_NET_FLOAT);
	}

tone::~tone(void)
	{
	if (p_->sample)
		delete p_->sample;
	delete p_;
	}


void tone::reset(void)
	{
	//reset playing flag to false
	p_->isPlaying = false;
	//set current amplitude to 0.0
	p_->curAmplitude = 0;
	//set update interval to 0.1
	p_->updateInterval = 0.1;
	//if a sample exists then delete it
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	ygSpace::reset();
	}


void tone::message(const ygMessage& msg)
	{
	//set the frequency of the tone
	if (msg == "frequency")
		{
		if (msg.args.size() > 0)
			setFrequency(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the amplitude within the space
	else if (msg == "amplitude")
		{
		if (msg.args.size() > 0)
			setAmplitude(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the zero amplitude distance from the space perimeter
	else if (msg == "falloffDistance")
		{
		if (msg.args.size() > 0)
			setFalloffDistance(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//start the sound playing with frequency and amplitude
	else if (msg == "play")
		{
		if (msg.args.size() > 0)
			{
			setFrequency(msg.floatArg(0));
			if (msg.args.size() > 1)
				{
				setAmplitude(msg.floatArg(1));
				play();
				}
			else
				play();
			}
		else
			play();
		}
	//stop the playing sound
	else if (msg == "stop")
		stop();
	//set the update interval for sound server
	else if (msg == "updateInterval")
		{
		if (msg.args.size() > 0)
			p_->updateInterval = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygSoundServer::message(msg);
	}


void tone::setFrequency(float val)
	{
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	p_->frequency = val;
	p_->netFrequency = p_->frequency;
	netKeyChanged("freq");
	}


void tone::setCurAmplitude(float val)
	{
	p_->curAmplitude = val;
	if (p_->sample)
		p_->sample->setAmplitude(val);
	}

bool tone::isPlaying(void)
	{
	return p_->isPlaying;
	}

void tone::play(void)
	{
	if (p_->isPlaying)
		return;
	//if no sample then create a new bergenSample
	if (!p_->sample)
		p_->sample = new bergenTone(server());
	p_->sample->setFrequency(p_->frequency);
	setCurAmplitude(computeAmplitude());
	//play the sample
	p_->sample->play();
	//generate a heartbeat for the server
	heartbeat();
	p_->isPlaying = true;
	eventOccurred("start");
	p_->netPlay = true;
	netKeyChanged("play");
	}


void tone::stop(bool sendStopCommand)
	{
	//if a sample is playing
	if (p_->isPlaying)
		{
		//send a stop command to the server
		if ((sendStopCommand) && (p_->sample))
			{
			p_->sample->stop();
			setCurAmplitude(0.0);
			}
		p_->isPlaying = false;
		eventOccurred("stop");
		p_->netPlay = false;
		netKeyChanged("play");
		}
	}


void tone::app(void)
	{
	//if a sample exists
	if (p_->sample)
		{
		//generate a heartbeat for the server
		heartbeat();
		//if the sample is playing
		if (p_->isPlaying)
			{
			//if update interval has passed then update sound server
			if ((ygWorld::FrameTime - p_->lastUpdate) > p_->updateInterval)
				{
				float newAmpl = computeAmplitude();
				if (fabs(newAmpl - p_->curAmplitude) > 0.005)
					setCurAmplitude(newAmpl);
				//determine position of this sound
				pfVec3 newPos;
				newPos = origin();
				pfVec3 cavePos;
				//get position relative to CAVE
				CAVENavConvertWorldToCAVE(newPos.vec,cavePos.vec);
				if (cavePos != p_->currPosition)
					{
					if (p_->sample)
						p_->sample->setPosition(cavePos.vec[0],cavePos.vec[2],-cavePos.vec[1]);
					p_->currPosition = cavePos;
					}
				p_->lastUpdate = ygWorld::FrameTime;
				}
			}
		}
	ygSoundServer::app();
	}


void tone::acceptNetKey(const ygString& key)
	{
	if (key == "play")
		{
		if (p_->netPlay)
			play();
		else
			stop();
		}
	else if (key == "freq")
		setFrequency(p_->netFrequency);
	else
		ygSoundServer::acceptNetKey(key);
	}
