// Description: a timer that generates changing values over a duration
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the timer can move backwards and forwards
//<li> see the <a href="interpolate">interpolate</a> node for more information
//<li> timers can be used in conjunction with other nodes to add dynamics to the scene
//</ul>
//
// Category: Math
// Author: Alex Hill
// Revision: 11/01/01
//
#include <ygWorld.h>
#include "timer.h"

extern "C" ygNode* construct_timer(const char* name,bool master) { return new timer(name,master); }

timer::timer(const char* name,bool master) : interpolate(name,master)
	{
	setClassName("timer");
	position = 0.0;
	duration = 1.0;
	active = false;
	reverse = false;
	}

timer::~timer(void)
	{
	}

void timer::reset(void)
	{
	//reset position
	position = 0.0;
	//set duration to 1
	duration = 1.0;
	//set inactive
	active = false;
	//set direction to forward
	reverse = false;
	interpolate::reset();
	}

void timer::message(const ygMessage& msg)
	{
	//set timer duration
	if (msg == "duration")
		{
		duration = msg.floatArg(0);
		}
	//start timer moving forward from zero
	else if (msg == "start")
		{
		position = 0.0;
		active = true;
		reverse = false;
		}
	//stop timer
	else if (msg == "stop")
		{
		active = false;
		}
	//move timer forward
	else if (msg == "forward")
		{
		active = true;
		reverse = false;
		}
	//move timer backward
	else if (msg == "reverse")
		{
		active = true;
		reverse = true;
		}
	//move timer to the front
	else if (msg == "front")
		{
		active = true;
		reverse = true;
		position = 0.0;
		}
	//move timer to the back
	else if (msg == "back")
		{
		active = true;
		reverse = false;
		position = 1.0;
		}
	else
		interpolate::message(msg);
	}
	
void timer::app(void)
	{
	//if active
	if (active)
		{
		//if reverse subtract frame time, else add
		if (reverse)
			position -= ygWorld::FrameDeltaTime/duration;
		else
			position += ygWorld::FrameDeltaTime/duration;
		//if timer is at an extreme
		if (position <= 0.0 || position >= 1.0)
			{
			active = false;
			//if direction is reverse
			if (reverse)
				{
				position = 0.0;
				//the timer has reached the front
				eventOccurred("front");
				}
			else
				{
				position = 1.0;
				//the timer has reached the back
				eventOccurred("back");
				}
			//the timer has ended at one extreme
			eventOccurred("end");
			}
		//calculate interpolated array values
		for (int i=0;i<size;i++)
			setInterpolate(position,i);
		}
	value::app();
	}
