// Description: adjust the texture matrix of all nodes below this node
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> rotations are applied before translation
//<li> use the startPosition message to do a pre-translation if necessary
//<li> the texture matrix of all instanced object nodes will be changed
//<li> use the cache option on child objects to avoid unwanted side effects
//</ul>
//
// Category: Attributes
// Author: Alex Hill
// Revision: 11/01/01
//
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pf/pfGroup.h>
#include <Performer/pfdu.h>
#include "ygNetKeys.h"
#include "textureTransform.h"

extern "C" ygNode* construct_textureTransform(const char* name,bool master) { return new textureTransform(name,master); }


textureTransform::textureTransform(const char* name,bool master) : ygNode(name,master)
    {
    setClassName("textureTransform",true);
    gstate = NULL;
    startPosition.set(0,0);
    position.set(0,0);
    orientation.set(0,0);
    size.set(1,1);
    addNetKey("matrix",&matrix,YG_NET_MATRIX);
    matrix = (pfMatrix*) pfMalloc(sizeof(pfMatrix), pfGetSharedArena());
    matrix->makeIdent();
    }

textureTransform::~textureTransform(void)
    {
    }

void textureTransform::reset(void)
    {
	//reset texture matrix to identity
    matrix->makeIdent();
    startPosition.set(0,0);
    position.set(0,0);
    orientation.set(0,0);
    size.set(1,1);
    setMatrix();
    } 

void textureTransform::setTexMat(pfNode *node)
    {
    int i;
	//if node type is pfGeode then set geoset material
    if (node->isOfType(pfGeode::getClassType()))
		{
		pfGeode *geode = (pfGeode *)node;
		for (i=0; i < geode->getNumGSets(); i++)
			setGeosetTexMat(geode->getGSet(i));
		}
	//else, if node type is pfGroup
    else if (node->isOfType(pfGroup::getClassType()))
		{
		pfGroup *group = (pfGroup *)node;
		//call setMaterial recursively on all children
		for (i=0; i < group->getNumChildren(); i++)
			setTexMat(group->getChild(i));
		}
    }

void textureTransform::setGeosetTexMat(pfGeoSet *gset)
    {
	//find and setup geostate texture matrix
	gstate = gset->getGState();
	gstate->setMode(PFSTATE_ENTEXMAT, PF_ON);
	gstate->setAttr(PFSTATE_TEXMAT, matrix);
	}


void textureTransform::message(const ygMessage& msg)
    {
	//set pre-rotation translational component
    if (msg == "startPosition")
		{
		if (msg.args.size() == 2)
			{
			msg.getVec2Args(startPosition);
			setMatrix();
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set rotational component
    else if (msg == "orientation")
		{
		if (msg.args.size() == 2)
			{
			msg.getVec2Args(orientation);
			setMatrix();
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set translational component
    else if (msg == "position")
		{
		if (msg.args.size() == 2)
			{
			msg.getVec2Args(position);
			setMatrix();
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set scaling component
    else if (msg == "size")
		{
		if (msg.args.size() == 1)
			{
			float newValue = msg.floatArg(0);
			size.set(newValue,newValue);
			setMatrix();
			}
		else if (msg.args.size() == 2)
			{
			msg.getVec2Args(size);
			setMatrix();
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
    else
		ygNode::message(msg);
    }

void textureTransform::setMatrix(void)
    {
	//create scaling matrix
    matrix->makeScale(size[0], size[1], 1);
    pfMatrix tmpMatrix;
	//multiply by pre-rotational translation
    tmpMatrix.makeTrans(startPosition[0], startPosition[1], 0);
    matrix->postMult(tmpMatrix);
	//multiply by rotation
    tmpMatrix.makeEuler(orientation[0], orientation[1], 0);
    matrix->postMult(tmpMatrix);
	//multiply by translation
    tmpMatrix.makeTrans(position[0], position[1], 0);
	//set texture matrix
    matrix->postMult(tmpMatrix);
    netKeyChanged("matrix");
    }

void textureTransform::app(void)
    {
    //if any child has changed then setMatrix for all child nodes
    if (pfChanged())
    	setTexMat(pfnode());
    ygNode::app();
    }
