// Description: a tri-strip surface of variable size
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the surface is created with the vertices message
//</ul>
//
// Category: Geometry
// Author: Alex Hill
//           11/01/02
// Revision: 04/01/04 Alex Hill - corrected material properties to work with materialProperty node
//
#include <Performer/pf/pfChannel.h>
#include <Performer/pf/pfGeode.h>
#include <ygPFTextureCache.h>
#include <ygWorld.h>
#include <ygNetKeys.h>
#include "surface.h"

using namespace std;

extern "C" ygNode* construct_surface(const char* name,bool master) { return new surface(name,master); }

surface::surface(const char* name,bool master) : geometry(name,master)
	{
	setClassName("surface",true);
	color_.set(1,1,1,1);
	vertices_.set(0,0);
	netVertices_.set(0,0);
	colors = NULL;
	texture = NULL;
	gstate = NULL;
	geode = NULL;
	addNetKey("verts",&netVertices_,YG_NET_VEC2);
	addNetKey("color",&color_,YG_NET_VEC4);
	addNetKey("texfile",&textureFile,YG_NET_STRING);
	}

surface::~surface(void)
	{
	}

void surface::reset(void)
	{
	//reset color to opaque white
	color_.set(1,1,1,1);
	netKeyChanged("color");
	//reset vertices to zero
	vertices_.set(0,0);
	netVertices_.set(0,0);
	netKeyChanged("verts");
	//reset texture file name
	textureFile.clear();
	netKeyChanged("texfile");
	colors = NULL;
	texture = NULL;
	gstate = NULL;
	//remove any existing child geode
	if (geode)
		pfnode()->removeChild(geode);
	geode = NULL;
	geometry::reset();	
	}


void surface::message(const ygMessage& msg)
	{
	//determine the number of vertices
	if (msg == "vertices") 
		{
		if (msg.args.size() == 2)
			{
			int xSize = msg.intArg(0);
			int ySize = msg.intArg(1);
			createGeometry(xSize,ySize);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the surface color
	else if (msg == "color") 
		{
		if (msg.args.size() == 4)
			{
			pfVec4 c;
			msg.getVec4Args(c);
			setColor(c);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the surface texture
	else if (msg == "texture") 
		{
		if (msg.args.size() > 0)
			{
			textureFile = msg.args[0];
			setTexture(textureFile);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		geometry::message(msg);
	}

void surface::createGeometry(int xSize, int ySize)
	{
	if (geode)
		pfnode()->removeChild(geode);
	vertices_.set(xSize,ySize);
	
	//allocate vertex, normal, color, and texture coordinate arrays
	verts  = (pfVec3*) pfMalloc(xSize * ySize * sizeof(pfVec3), pfGetSharedArena());
	norms  = (pfVec3*) pfMalloc(xSize * ySize * sizeof(pfVec3), pfGetSharedArena());
	colors = (pfVec4*) pfMalloc(xSize * ySize * sizeof(pfVec4), pfGetSharedArena());
	texCoords  = (pfVec2*) pfMalloc(xSize * ySize * sizeof(pfVec2), pfGetSharedArena());

	//set up vertex, normal, color, and texture corrdinate values
	int index;
	for (int y = 0, index = 0; y < ySize; y++)
		{
		for (int x = 0; x < xSize; x++, index++) 
			{
			verts[index].set(x/(float)(xSize-1) - 0.5,y/(float)(ySize-1) - 0.5,0.0);
			norms[index].set(0, 0, 1);
			colors[index].set(color_[0], color_[1], color_[2], color_[3]);
			texCoords[index].set(x/(float)(xSize-1), y/(float)(ySize-1));
			}
		}

	//create geoset
	gset = new pfGeoSet;
	gset->setPrimType(PFGS_TRISTRIPS);
	gset->setNumPrims(ySize-1);

	int* lengths = (int*)pfMalloc((ySize-1)*sizeof(int), pfGetSharedArena());
	for (int y = 0; y < ySize-1; y++)
		lengths[y] = xSize * 2;
	gset->setPrimLengths(lengths);

	//set up indexing
	ilist = (ushort*)pfMalloc(2*xSize*(ySize-1)*sizeof(ushort), pfGetSharedArena());
	index = 0;
	for (int y=0; y<ySize-1; y++)
		{
		for (int x=0; x<xSize; x++)
			{
			ilist[index++] = x + y*xSize + xSize;
			ilist[index++] = x + y*xSize;
			}
		}

	//add array attributes to geoset
	gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, verts, ilist);
	gset->setAttr(PFGS_NORMAL3, PFGS_PER_VERTEX, norms, ilist);
	gset->setAttr(PFGS_COLOR4, PFGS_PER_VERTEX, colors, ilist);
	gset->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, texCoords, ilist);               

	//create geostate
	gstate = new pfGeoState;
	mat = new pfMaterial();
	mat->setColorMode(PFMTL_BOTH,PFMTL_CMODE_DIFFUSE);
	gstate->setAttr(PFSTATE_FRONTMTL, mat);
	gstate->setAttr(PFSTATE_BACKMTL, mat);
	gstate->setMode(PFSTATE_ENLIGHTING, PF_ON);
	gset->setGState(gstate);
	if (texture)
		{
		int comp;
		//get number of image components
		texture->getImage(NULL,&comp,NULL,NULL,NULL);
		//if alpha channel then set transparency
		if (comp == 4)
			{
			gstate->setMode(PFSTATE_TRANSPARENCY,PF_ON|PFTR_NO_OCCLUDE|PFTR_BLEND_ALPHA);
			gset->setDrawBin(PFSORT_TRANSP_BIN);
			}
		else
			{
			gstate->setMode(PFSTATE_TRANSPARENCY,PF_OFF);
			gset->setDrawBin(PFSORT_OPAQUE_BIN);
			}
		gstate->setAttr(PFSTATE_TEXTURE,texture);
		gstate->setMode(PFSTATE_ENTEXTURE,PF_ON);
		}

	//create geode
	geode = new pfGeode;
	geode->addGSet(gset);
	pfnode()->addChild(geode);
	setPFChanged();
	netVertices_ = vertices_;
	netKeyChanged("verts");
	}

void surface::app(void)
	{
	geometry::app();
	}

void surface::setColor(pfVec4 c)
	{
	if (colors)
		{
		color_ = c;
		int xSize = (int)vertices_[0];
		int ySize = (int)vertices_[1];
		for (int y = 0, index = 0; y < ySize; y++) 
			for (int x = 0; x < xSize; x++, index++) 
				colors[index].set(color_[0], color_[1], color_[2], color_[3]);
		}
	netKeyChanged("color");
	}

void surface::setTexture(const ygString& file)
	{
	textureFile = file;
	if (textureFile.length() > 0)
		texture = ygPFTextureCache::textureReference(file.c_str());
	else
		texture = NULL;
	if (texture && gstate)
		{
		int comp;
		//get number of image components
		texture->getImage(NULL,&comp,NULL,NULL,NULL);
		//if alpha channel then set transparency
		if (comp == 4)
			{
			gstate->setMode(PFSTATE_TRANSPARENCY,PF_ON|PFTR_NO_OCCLUDE|PFTR_BLEND_ALPHA);
			gset->setDrawBin(PFSORT_TRANSP_BIN);
			}
		else
			{
			gstate->setMode(PFSTATE_TRANSPARENCY,PF_OFF);
			gset->setDrawBin(PFSORT_OPAQUE_BIN);
			}
		gstate->setAttr(PFSTATE_TEXTURE, texture);
		gstate->setMode(PFSTATE_ENTEXTURE, PF_ON);
		setPFChanged();
		}
	netKeyChanged("texfile");
	}

void surface::acceptNetKey(const ygString& key)
	{
	if (key == "verts")
		createGeometry((int)netVertices_[0],(int)netVertices_[1]);
	else if (key == "color")
		setColor(color_);
	else if (key == "texfile")
		setTexture(textureFile);
	else
		geometry::acceptNetKey(key);
	}
