// Description: a node that implements stencil buffering
//
//<b>notes:</b>
//<ul>
//<li> all parent node messages are inherited
//<li> see reset method for default settings
//<li> the children of this node are drawn into the stencil buffer but not the color buffer
//<li> the scene message defines the node and its children that are drawn after the stencil buffer has been setup
//<li> the scene node should neither be a transform nor have a parent transform
//<li> this node should be the first node in your scene followed by the scene node that it will use
//<li> you must add the following to your RUN file in order to use this node:<br>
//<ul>
//<li> setenv YG_PRELOAD_CLASSES stencilBuffer
//</ul>
//</ul>
//
// Category: Attributes
// Author: Alex Hill
// Revision: 10/02/04
//
#include <Performer/pf/pfGroup.h>
#include <Performer/pf/pfTraverser.h>
#include <ygNodeDB.h>
#include <ygNetKeys.h>
#include "stencilBuffer.h"

using namespace std;

extern "C" ygNode* construct_stencilBuffer(const char* name,bool master) { return new stencilBuffer(name,master); }

typedef struct _stencilData
	{
	bool active;
	bool depth;
	bool reverse;
	} stencilData_t;

stencilBuffer::stencilBuffer(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("stencilBuffer",true);
	//allocate shared memorystencilData structure for pre and post callbacks
	data_ = (stencilData_t*) pfMalloc(sizeof(stencilData_t),pfGetSharedArena());
	stencilNode = NULL;
	data_->active = true;
	data_->depth = true;
	data_->reverse = false;
	//set the pre and post draw traversal functions for stencil writing
	pfnode()->setTravFuncs(PFTRAV_DRAW,writePreDraw,writePostDraw);
	pfnode()->setTravData(PFTRAV_DRAW,data_);
	clone = new pfGroup();
	clone->setTravFuncs(PFTRAV_DRAW,clonePreDraw,clonePostDraw);
	clone->setTravData(PFTRAV_DRAW,data_);
	addNetKey("stencilName",&nodeName,YG_NET_STRING);
	//distribute the off state of the stencil buffer
	addNetKey("active",&data_->active,YG_NET_BOOL);
	//distribute the depth test mode
	addNetKey("depth",&data_->depth,YG_NET_BOOL);
	//distribute the on/off state of the stencil buffer
	addNetKey("reverse",&data_->reverse,YG_NET_BOOL);
	}

stencilBuffer::~stencilBuffer(void)
	{
	}


void stencilBuffer::reset(void)
	{
	//clear the affected node name
	nodeName.clear();
	stencilNode = NULL;
	//set the plane to active
	data_->active = true;
	//set the depth mode to true
	data_->depth = true;
	//set the reverse mode to false
	data_->reverse = false;
	ygNode::reset();
	}


void stencilBuffer::message(const ygMessage& msg)
	{
	//set the name of the node to be affected
	if (msg == "scene")
		{
		if (msg.args.size() > 0)
			{
			nodeName = msg.args[0];
			setStencilNode(nodeName);
			netKeyChanged("stencilName");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//turn the stencil buffer on
	else if (msg == "on")
		{
		if (!data_->active)
			{
			data_->active = true;
			netKeyChanged("active");
			}
		}
	//turn the stencil buffer off
	else if (msg == "off")
		{
		if (data_->active)
			{
			data_->active = false;
			netKeyChanged("active");
			}
		}
	//use the depth buffer true/false
	else if (msg == "depth")
		{
		if (msg.args.size() > 0)
			{
			data_->depth = msg.boolArg(0);
			netKeyChanged("depth");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//reverse the stencil buffer test
	else if (msg == "reverse")
		{
		if (msg.args.size() > 0)
			{
			data_->reverse = msg.boolArg(0);
			netKeyChanged("reverse");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygNode::message(msg);
	}

void stencilBuffer::setStencilNode(const ygString& name)
	{
	//remove clone children
	while (clone->getNumChildren())
		clone->removeChild(clone->getChild(0));
	//add clone children
	for (int i=0;i<pfnode()->getNumChildren();i++)
		clone->addChild(pfnode()->getChild(i));
	nodeName = name;
	//if an affected node is being stenciled then
	if (stencilNode)
		{
		//reset the traversal functions to NULL
		stencilNode->pfnode()->removeChild(clone);
		stencilNode->pfnode()->setTravFuncs(PFTRAV_DRAW,NULL,NULL);
		stencilNode->pfnode()->setTravData(PFTRAV_DRAW,NULL);
		}	
	stencilNode = NULL;
	if (name.length() > 0)
		{
		//find the desired affected node
		stencilNode = ygNodeDB::find(name);
		//if node is found then
		if (stencilNode)
			{
			//set the pre and post draw traversal functions for affected pfNode
			stencilNode->pfnode()->setTravFuncs(PFTRAV_DRAW,stencilPreDraw,stencilPostDraw);
			stencilNode->pfnode()->setTravData(PFTRAV_DRAW,data_);
			stencilNode->pfnode()->addChild(clone);
			}
		}
	}
	
int stencilBuffer::writePreDraw(pfTraverser*,void *data)
	{
	//cast the traversal data as the stencilData structure
	stencilData_t* stencilData = (stencilData_t*)data;
	//if the plane is active then
	if (stencilData->active)
		{
		//disable writing to the color buffer
		glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE); 
		//disable writing to the depth buffer
		glDisable(GL_DEPTH_TEST);
		glClearStencil(0);
		glClear(GL_STENCIL_BUFFER_BIT);		
		//set stencil function to always fail
		glStencilFunc(GL_ALWAYS,1,1);
		glStencilOp(GL_REPLACE,GL_REPLACE,GL_REPLACE);
		//enable stencil testing
		glEnable(GL_STENCIL_TEST);
		}
	return PFTRAV_CONT;
	}

int stencilBuffer::writePostDraw(pfTraverser*,void *data)
	{
	//cast the traversal data as the stencilData structure
	stencilData_t* stencilData = (stencilData_t*)data;
	//disable writing to the stencil buffer
	glDisable(GL_STENCIL_TEST);
	//enable writing to the color buffer
	glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
	//enable writing to the depth buffer
	glEnable(GL_DEPTH_TEST);
	return PFTRAV_CONT;
	}

int stencilBuffer::stencilPreDraw(pfTraverser*,void *data)
	{
	//cast the traversal data as the stencilData structure
	stencilData_t* stencilData = (stencilData_t*)data;
	//if the plane is active then
	if (stencilData->active)
		{
		//set stencil function to write when stencil equals 1
		if (stencilData->reverse)
			glStencilFunc(GL_NOTEQUAL,1,1);
		else
			glStencilFunc(GL_EQUAL,1,1);
		glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
		//enable stencil testing
		glEnable(GL_STENCIL_TEST);
		}
	return PFTRAV_CONT;
	}

int stencilBuffer::stencilPostDraw(pfTraverser*,void *data)
	{
	//cast the traversal data as the stencilData structure
	stencilData_t* stencilData = (stencilData_t*)data;
	//disable the stencil testing
	glDisable(GL_STENCIL_TEST);
	return PFTRAV_CONT;
	}

int stencilBuffer::clonePreDraw(pfTraverser*,void *data)
	{
	//cast the traversal data as the stencilData structure
	stencilData_t* stencilData = (stencilData_t*)data;
	//disable depth testing if necessary
	if (!stencilData->depth)
		glDisable(GL_DEPTH_TEST);
	//disable writing to the color buffer
	glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE);
	return PFTRAV_CONT;
	}

int stencilBuffer::clonePostDraw(pfTraverser*,void *data)
	{
	//enable writing to the color buffer
	glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
	glEnable(GL_DEPTH_TEST);
	return PFTRAV_CONT;
	}

void stencilBuffer::acceptNetKey(const ygString& key)
	{
	if (key == "stencilName")
		setStencilNode(nodeName);
	else
		ygNode::acceptNetKey(key);
	}
