// Description: loads models from any one of the accepted Performer model formats
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> if an included transform is required use the <a href="ygObject.html">ygObject</a> node
//<li> objects are an instance from a Performer object cache unless the cache mode is set false before the file message
//<li> Common file formats supported by Performer:
//<b> .pfb</b> - IRIS Performer fast binary format
//<b> .flt</b> - MultiGen OpenFlight format
//<b> .iv</b> - SGI Open Inventor format
//<b> .wrl</b> - VRML version 2.0 format
//<b> .obj</b> - Wavefront Technologies data format
//<b> .3ds</b> - AutoDesk 3DStudio binary data
//<b> .dxf</b> - AutoDesk AutoCAD ASCII format
//<b> .lsa</b> - Lightscape Technologies ASCII radiosity data
//<b> .lsb</b> - Lightscape Technologies binary radiosity data
//<li> The pfconv utility can be used to convert any format that Performer will read into the native Performer fast binary format(.pfb)
//<ol> the pfconv utility is located in /usr/sbin/
//<li> type pvconv then the name of the file you want to convert, followed by the name of the converted file
//<li> <b>ex:</b> pfconv oldfile.flt newfile.pfb
//<li> see the man pages on pfconv for more information
//</ol>
//<li> the wall mode is used to turn collision detection on (default is false)
//<li> the floor mode is used to specify whether the model is treated as terrain (default is false)
//<li> the draw mode is set to false, the model is not drawn but collision detection can still be enabled (default is true)
//</ul>
//
// Category: Geometry
// Author: Dave Pape
//           11/01/01
// Revision: 10/15/02 Dave Pape - add optional cache flag to "file" msg, "reload" message, and included cache-state in netFilename
//
#include <Performer/pf/pfGroup.h>
#include <ygWorld.h>
#include <ygNetKeys.h>
#include <ygPFObjectCache.h>
#include <Performer/pfdu.h>
#include "staticObject.h"

extern "C" ygNode* construct_staticObject(const char* name,bool master) { return new staticObject(name,master); }

struct _staticObjectPrivateData
	{
	ygString filename;
	pfNode *obj;
	uint isectMask;
	bool drawn, cache, showLocal;
	ygString netFilename;
	};


staticObject::staticObject(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("staticObject",true);
	p_ = new struct _staticObjectPrivateData;
	p_->obj = NULL;
	p_->isectMask = YG_ISECT_ALLMASK;
	p_->drawn = true;
	p_->cache = true;
	p_->showLocal = true;
	addNetKey("file",&p_->netFilename,YG_NET_STRING);
	addNetKey("isect",&p_->isectMask,YG_NET_INT);
	addNetKey("drawn",&p_->drawn,YG_NET_BOOL);
	}

staticObject::~staticObject(void)
	{
	delete p_;
	}


void staticObject::reset(void)
	{
	//reset intersection mask
	p_->isectMask = YG_ISECT_ALLMASK;
	netKeyChanged("isect");
	//set drawn to true
	p_->drawn = true;
	netKeyChanged("drawn");
	//set cache to true
	p_->cache = true;
	//set show local to true
	p_->showLocal = true;
	//if an object has been loaded
	if (p_->obj)
		{
		//reset intersection traversal mask
		p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask,
					PFTRAV_SELF|PFTRAV_DESCEND,
					PF_SET);
		//if drawn and showLocal are true then allow drawing
		if ((p_->drawn) && (p_->showLocal))
			p_->obj->setTravMask(PFTRAV_DRAW, 0xffffffff,
					PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		else
			p_->obj->setTravMask(PFTRAV_DRAW, 0,
					PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	ygNode::reset();
	}


void staticObject::message(const ygMessage& msg)
	{
	//load the given filename with optional caching
	if (msg == "file")
		{
		if (msg.args.size() == 1 || msg.args.size() == 2)
			{
			ygString file = msg.args[0];
			if (msg.args.size() == 2)
				{
				bool saveCache = p_->cache;
				p_->cache = msg.boolArg(1);
				loadFile(file);
				p_->cache = saveCache;
				}
			else
				loadFile(file);
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set collision detection status
	else if (msg == "wall")
		{
		if (msg.args.size() > 0)
			setWall(msg.boolArg(0));
		else
			setWall();
		}
	//set walk navigation status
	else if (msg == "floor")
		{
		if (msg.args.size() > 0)
			setFloor(msg.boolArg(0));
		else
			setFloor();
		}
	//set if the object is drawn
	else if (msg == "draw")
		{
		if (msg.args.size() > 0)
			setDraw(msg.boolArg(0));
		else
			setDraw();
		}
	//set whether the object is drawn locally
	else if (msg == "showLocal")
		{
		if (msg.args.size() > 0)
			showLocal(msg.boolArg(0));
		else
			showLocal();
		}
	//set the default object caching mode
	else if (msg == "cache")
		{
		if (msg.args.size() > 0)
			setCache(msg.boolArg(0));
		else
			setCache();
		}
	//reload the currently loaded object
	else if (msg == "reload")
		{
		/* This is slightly contorted just to avoid changing the class interface */
		bool saveCache = p_->cache;
		ygString file = p_->filename;
		p_->cache = false;
		p_->filename.clear();
		loadFile(file);
		p_->cache = saveCache;
		}
	else
		ygNode::message(msg);
	}


void staticObject::loadFile(const ygString& file)
	{
	if (p_->obj)
		pfnode()->removeChild(p_->obj);
	if (p_->cache)
		p_->obj = ygPFObjectCache::objectReference(file.c_str());
	else
		p_->obj = pfdLoadFile(file.c_str());
	if (p_->obj)
		{
		p_->filename = file;
		pfnode()->addChild(p_->obj);
		if ((p_->drawn) && (p_->showLocal))
			p_->obj->setTravMask(PFTRAV_DRAW, 0xffffffff,
					PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		else
			p_->obj->setTravMask(PFTRAV_DRAW, 0,
					PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask,
					PFTRAV_SELF|PFTRAV_DESCEND|PFTRAV_IS_CACHE, PF_SET);
		p_->netFilename = file;
		char cacheStr[32];
		sprintf(cacheStr," %d",p_->cache);
		p_->netFilename += cacheStr;
		netKeyChanged("file");
		setPFChanged();
		}
	else
		p_->filename.clear();
	}


void staticObject::setWall(bool val)
	{
	if (val)
		p_->isectMask |= YG_ISECT_WALLMASK;
	else
		p_->isectMask &= ~YG_ISECT_WALLMASK;
	netKeyChanged("isect");
	if (p_->obj)
		p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask,
				PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	}


void staticObject::setFloor(bool val)
	{
	if (val)
		p_->isectMask |= YG_ISECT_FLOORMASK;
	else
		p_->isectMask &= ~YG_ISECT_FLOORMASK;
	netKeyChanged("isect");
	if (p_->obj)
		p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask,
				PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
	}


void staticObject::setDraw(bool val)
	{
	p_->drawn = val;
	netKeyChanged("drawn");
	if (p_->obj)
		{
		if ((p_->drawn) && (p_->showLocal))
			p_->obj->setTravMask(PFTRAV_DRAW, 0xffffffff,
					PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		else
			p_->obj->setTravMask(PFTRAV_DRAW, 0,
					PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	}


void staticObject::showLocal(bool show)
	{
	p_->showLocal = show;
	if (p_->obj)
		{
		if ((p_->drawn) && (p_->showLocal))
			p_->obj->setTravMask(PFTRAV_DRAW, 0xffffffff, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		else
			p_->obj->setTravMask(PFTRAV_DRAW, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	}


void staticObject::setCache(bool v)
	{
	p_->cache = v;
	}

const ygString& staticObject::filename(void) const
	{
	return p_->filename;
	}

void staticObject::acceptNetKey(const ygString& key)
	{
	if (key == "file")
		{
		ygString *file,*cacheStr;
		int index=0;
		file = p_->netFilename.nextToken(" ",&index);
		cacheStr = p_->netFilename.nextToken(" ",&index);
		if (cacheStr)
			p_->cache = atoi(cacheStr->c_str());
		loadFile(*file);
		}
	else if (key == "drawn")
		setDraw(p_->drawn);
	else if (key == "isect")
		{
		if (p_->obj)
			p_->obj->setTravMask(PFTRAV_ISECT, p_->isectMask,
					PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
		}
	else
		ygNode::acceptNetKey(key);
	}
