// Description: creates a spotlight with transformed position and direction
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node creates a pfLightSource node
//<li> Lighting in a scene is based on the number of lights, lighting-model modes, and the material properties of the illuminated objects where the contribution of each light source is the sum of:
//<ol>
//<li> light source ambient color, scaled by the material ambient reflectance
//<li> light source diffuse color, scaled by the material diffuse reflectance and the dot product of the vertex normal and the vertex-to-light source vector
//<li> light source specular color, scaled by the material specular reflectance and a function of the angle between the vertex-to-viewpoint vector and the vertex-to-light source vector
//</ol>
//<li> The ambient component is nondirectional and distributed uniformly throughout the environment
//<li> The ambient component is reflected from the object independent of surface location and orientation with equal intensity in all directions
//<li> The diffuse component takes into account the direction of a light source. The intensity of light striking a surface varies with the angle between the orientation of the object and the direction of the light source -- the diffuse parameter of a material determines how much light is scattered evenly in all directions
//<li> The specular component represents reflection off shiny surfaces. Specular lighting operates based on the viewer's position. Maximum specular reflectance occurs when the angle between the viewer and the direction of the reflected light is zero -- the brightness of a specular reflection decays based upon the exponential value shininess
//<li> The position and direction of the spotlight is derived from position, orientation, and any parent transformations
//<li> The default direction of the spotlight is down (0,0,-1)
//<li> The divergence adjusts the distance from which light comes from infinity, 0.0, to the actual position, 1.0
//<li> The falloff reduces the inensity as the angle from the lighting direction increases
//<li> The spread sets the cuttoff angle of the intensity from the lighting direction
//<li> The attenuation values determine the intensity of light on a surface as: 
//		1.0 / (constant + linear * distance + quadratic * distance^2)
//		where distance is the distance from the light source to the surface
//</ul>
//
// Category: Attributes
// Author: Alex Hill
// Revision: 11/01/02
//
#include <Performer/pf/pfLightSource.h>
#include <Performer/pf/pfGroup.h>
#include "ygNetKeys.h"
#include "spotLight.h"

extern "C" ygNode* construct_spotLight(const char* name,bool master) { return new spotLight(name,master); }

struct _spotLightPrivateData
	{
	pfLightSource* light;
	struct
		{
		pfVec3 attenuation;
		pfVec3 cone;
		pfVec3 ambient;
		pfVec3 diffuse;
		pfVec3 specular;
		bool on;
		} net;
	bool prevOn;
	};


spotLight::spotLight(const char* name,bool master) : ygSimpleTransform(name,master)
	{
	setClassName("spotLight",true);
	p_ = new struct _spotLightPrivateData;
	//create pfLightSource
	p_->light = new pfLightSource;
	pfnode()->addChild(p_->light);
	addNetKey("atten",&p_->net.attenuation,YG_NET_VEC3);
	addNetKey("cone",&p_->net.cone,YG_NET_VEC3);
	addNetKey("amb",&p_->net.ambient,YG_NET_VEC3);
	addNetKey("dif",&p_->net.diffuse,YG_NET_VEC3);
	addNetKey("spec",&p_->net.specular,YG_NET_VEC3);
	addNetKey("on",&p_->net.on,YG_NET_BOOL);
	setAttenuation(1,0,0);
	setCone(1.0,0.0,180.0);
	p_->light->setPos(0,0,0,p_->net.cone[0]);
	setAmbient(0,0,0);
	setDiffuse(1,1,1);
	setSpecular(1,1,1);
	on();
	p_->prevOn = true;
	}


spotLight::~spotLight(void)
	{
	pfnode()->removeChild(p_->light);
	pfDelete(p_->light);
	delete p_;
	}


void spotLight::reset(void)
	{
	//if net master
	if (isNetMaster())
		{
		//set attenuation to 1,0,0
		setAttenuation(1,0,0);
		//set divergence to 1.0, falloff to 0.0, and spread to 180.0
		setCone(1.0,0.0,180.0);
		p_->light->setPos(0,0,0,p_->net.cone[0]);
		//set ambient to 0,0,0
		setAmbient(0,0,0);
		//set diffuse to 1,1,1
		setDiffuse(1,1,1);
		//set specular to 1,1,1
		setSpecular(1,1,1);
		//set to on
		on();
		p_->prevOn = true;
		}
	ygSimpleTransform::reset();
	}


/*****  LightSource functions *****/

bool spotLight::isOn(void) const
	{
	return (p_->light->isOn() != 0);
	}


void spotLight::on(void)
	{
	if (!isOn())
		{
		p_->light->on();
		p_->net.on = true;
		netKeyChanged("on");
		}
	}


void spotLight::off(void)
	{
	if (isOn())
		{
		p_->light->off();
		p_->net.on = false;
		netKeyChanged("on");
		}
	}


void spotLight::toggle(void)
	{
	if (isOn())
		off();
	else
		on();
	}


void spotLight::setAttenuation(float c,float l,float q)
	{
	p_->light->setAtten(c,l,q);
	p_->net.attenuation.set(c,l,q);
	netKeyChanged("atten");
	}


void spotLight::setCone(float divergence,float intensity,float spread)
	{
	p_->light->setSpotCone(intensity,spread);
	p_->net.cone.set(divergence,intensity,spread);
	netKeyChanged("cone");
	}


void spotLight::setAmbient(float r,float g,float b)
	{
	p_->light->setColor(PFLT_AMBIENT, r, g, b);
	p_->net.ambient.set(r,g,b);
	netKeyChanged("amb");
	}

void spotLight::setDiffuse(float r,float g,float b)
	{
	p_->light->setColor(PFLT_DIFFUSE, r, g, b);
	p_->net.diffuse.set(r,g,b);
	netKeyChanged("dif");
	}

void spotLight::setSpecular(float r,float g,float b)
	{
	p_->light->setColor(PFLT_SPECULAR, r, g, b);
	p_->net.specular.set(r,g,b);
	netKeyChanged("spec");
	}


/***** Message functions *****/

void spotLight::message(const ygMessage& msg)
	{
	//set the light to on
	if (msg == "on")
		on();
	//set the light to off
	else if (msg == "off")
		off();
	//toggle the light state
	else if (msg == "toggle")
		toggle();
	//set the light attenuation (constant,linear,quadratic)
	else if (msg == "attenuation")
		{
		pfVec3 v3;
		if (msg.getVec3Args(v3))
			setAttenuation(v3);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the spotlight divergence [0:1]
	else if (msg == "divergence")
		{
		pfVec3 v3;
		if (msg.args.size() == 1)
			setCone(msg.floatArg(0),p_->net.cone[1],p_->net.cone[2]);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the spotlight intensity falloff [0:128]
	else if (msg == "falloff")
		{
		pfVec3 v3;
		if (msg.args.size() == 1)
			setCone(p_->net.cone[0],msg.floatArg(0),p_->net.cone[2]);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the spotlight spread angle [0:90|180]
	else if (msg == "spread")
		{
		pfVec3 v3;
		if (msg.args.size() == 1)
			setCone(p_->net.cone[0],p_->net.cone[1],msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the ambient color of the light
	else if (msg == "ambient")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setAmbient(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the diffuse color of the light
	else if (msg == "diffuse")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setDiffuse(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the specular color of the light
	else if (msg == "specular")
		{
		pfVec3 v;
		if (msg.getVec3Args(v))
			setSpecular(v);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygSimpleTransform::message(msg);
	}

/***** Network functions *****/

void spotLight::acceptNetKey(const ygString& key)
	{
	if (key == "atten")
		setAttenuation(p_->net.attenuation);
	else if (key == "cone")
		setCone(p_->net.cone);
	else if (key == "amb")
		setAmbient(p_->net.ambient);
	else if (key == "dif")
		setDiffuse(p_->net.diffuse);
	else if (key == "spec")
		setSpecular(p_->net.specular);
	else if (key == "on")
		{
		if (p_->net.on)
			on();
		else
			off();
		}
	else
		ygSimpleTransform::acceptNetKey(key);
	}
