// Description: creates a positional sound within a space using one of several file formats
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> this node can also be created with the alias: sound
//<li> the volume of a sound is the region in which the sound will play at 100% amplitude(loudness)
//<li> falloffDistance is a region outside the volume in which the amplitude is diminished as the user moves further away from the source(volume)
//<li> Common file formats supported by snerd:
//<b> .aiff</b> - Audio Interchange File Format
//<b> .wav</b> - MS RIFF WAVE Format
//<li> sound files should be saved using the <b>.aiff</b> format for best performance
//<li> the sound server, snerd, must be run on the a local or remote machine
//<li> the location of sound files is determined by the YG_PATH environmental variable
//<li> sounds under a switch will turn off but must be restarted when the switch is toggled back on
//</ul>
//
// Category: Sound
// Author: Dave Pape
// Revision: 02/10/04 Alex Hill
//
#include <pfcave.h>
#include <bergenSample.h>
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "soundSource.h"

using namespace std;

extern "C" ygNode* construct_soundSource(const char* name,bool master) { return new soundSource(name,master); }

struct _soundSourcePrivateData
	{
	ygString file;
	bergenSample * sample;
	bool isPlaying;
	float duration;
	bool loop;
	int attenuation;
	float startTime, endTime;
	float curAmplitude;
	pfVec3 currPosition;
	ygString netFile;
	float updateInterval, lastUpdate;
	bool netPlay;
	};


soundSource::soundSource(const char* name,bool master) : ygSoundServer(name,master)
	{
	setClassName("soundSource",true);
	p_ = new struct _soundSourcePrivateData;
	p_->sample = NULL;
	p_->attenuation = 0;
	p_->loop = false;
	p_->startTime = 0;
	p_->endTime = 0;
	p_->curAmplitude = 0;
	p_->updateInterval = 0.1;
	p_->lastUpdate = 0;
	p_->isPlaying = false;
	addNetKey("play", &p_->netPlay, YG_NET_BOOL);
	addNetKey("file", &p_->netFile, YG_NET_STRING);
	addNetKey("loop", &p_->loop, YG_NET_BOOL);
	addNetKey("atten", &p_->attenuation, YG_NET_INT);
	}

soundSource::~soundSource(void)
	{
	if (p_->sample)
		delete p_->sample;
	delete p_;
	}


void soundSource::reset(void)
	{
	//reset playing flag to false
	p_->isPlaying = false;
	//reset loop flag to false
	p_->loop = false;
	//set attenuation to inverse square law
	p_->attenuation = 3;
	//set start time to 0.0
	p_->startTime = 0;
	//set end time to 0.0
	p_->endTime = 0;
	//set current amplitude to 0.0
	p_->curAmplitude = 0;
	//if a sample exists then delete it
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	ygSpace::reset();
	}


void soundSource::message(const ygMessage& msg)
	{
	//set the name of the sound file
	if (msg == "file")
		{
		if (msg.args.size() > 0)
			setFile(msg.args[0]);
		}
	//set the amplitude within the space
	else if (msg == "amplitude")
		setAmplitude(msg.floatArg(0));
	//set the zero amplitude distance from the space perimeter
	else if (msg == "falloffDistance")
		setFalloffDistance(msg.floatArg(0));
	//set the sound attenuation mode
	else if (msg == "attenuation")
		setAttenuation(msg.intArg(0));
	//set the sound looping mode
	else if (msg == "loop")
		setLoop(msg.boolArg(0));
	//start the sound playing with file and amplitude
	else if (msg == "play")
		{
		if (msg.args.size() > 0)
			{
			setFile(msg.args[0]);
			if (msg.args.size() > 1)
				{
				setAmplitude(msg.floatArg(1));
				play();
				}
			else
				play();
			}
		else
			play();
		}
	//stop the playing sound
	else if (msg == "stop")
		stop();
	//set the update interval for sound server
	else if (msg == "updateInterval")
		{
		if (msg.args.size() > 0)
			p_->updateInterval = msg.floatArg(0);
		}
	else
		ygSoundServer::message(msg);
	}


void soundSource::setFile(const ygString& val)
	{
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	p_->file = val;
	p_->netFile = p_->file;
	netKeyChanged("file");
	}


void soundSource::setAttenuation(int val)
	{
	p_->attenuation = val;
	if (p_->sample)
		p_->sample->setAttenuationType((AttenuationType)val);
	netKeyChanged("atten");
	}

void soundSource::setLoop(bool val)
	{
	p_->loop = val;
	if (p_->sample)
		p_->sample->setLoop(val);
	netKeyChanged("loop");
	}


void soundSource::setCurAmplitude(float val)
	{
	p_->curAmplitude = val;
	if (p_->sample)
		p_->sample->setAmplitude(val);
	}

bool soundSource::isPlaying(void)
	{
	return p_->isPlaying;
	}

void soundSource::play(void)
	{
	if (p_->isPlaying)
		return;
	//if no sample then create a new bergenSample
	if (!p_->sample)
		{
		p_->sample = new bergenSample(p_->file.c_str(),server());
		p_->duration = p_->sample->duration();
		if (p_->loop)
			p_->sample->setLoop(1);
		else
			p_->sample->setLoop(0);
		}
	p_->sample->setLoop(p_->loop);
	setCurAmplitude(computeAmplitude());
	//play the sample
	p_->sample->play();
	//generate a heartbeat for the server
	heartbeat();
	if (!p_->loop)
		{
		p_->startTime = ygWorld::FrameTime;
		p_->endTime = p_->startTime + p_->duration;
		}
	p_->isPlaying = true;
	eventOccurred("start");
	p_->netPlay = true;
	netKeyChanged("play");
	}


void soundSource::stop(bool sendStopCommand)
	{
	//if a sample is playing
	if (p_->isPlaying)
		{
		//send a stop command to the server
		if ((sendStopCommand) && (p_->sample))
			{
			p_->sample->stop();
			setCurAmplitude(0.0);
			}
		p_->isPlaying = false;
		eventOccurred("stop");
		p_->netPlay = false;
		netKeyChanged("play");
		}
	}


void soundSource::app(void)
	{
	//if a sample exists
	if (p_->sample)
		{
		//generate a heartbeat for the server
		heartbeat();
		//if the sample is playing
		if (p_->isPlaying)
			{
			//if sample is past end time and not looping then stop
			if ((!p_->loop) && (p_->endTime < ygWorld::FrameTime))
				stop(false);
			//else, compute new amplitude
			else
				{
				//if update interval has passed then update sound server
				if ((ygWorld::FrameTime - p_->lastUpdate) > p_->updateInterval)
					{
					float newAmpl = computeAmplitude();
					if (fabs(newAmpl - p_->curAmplitude) > 0.005)
						setCurAmplitude(newAmpl);
					//determine position of this sound
					pfVec3 newPos;
					newPos = origin();
					pfVec3 cavePos;
					//get position relative to CAVE
					CAVENavConvertWorldToCAVE(newPos.vec,cavePos.vec);
					if (cavePos != p_->currPosition)
						{
						if (p_->sample)
							p_->sample->setPosition(cavePos.vec[0],cavePos.vec[2],-cavePos.vec[1]);
						p_->currPosition = cavePos;
						}
					p_->lastUpdate = ygWorld::FrameTime;
					}
				}
			}
		}
	ygSoundServer::app();
	}


void soundSource::acceptNetKey(const ygString& key)
	{
	if (key == "play")
		{
		if (p_->netPlay)
			play();
		else
			stop();
		}
	else if (key == "atten")
		setAttenuation(p_->attenuation);
	else if (key == "file")
		setFile(p_->netFile);
	else
		ygSoundServer::acceptNetKey(key);
	}
