// Description: creates a node that will record the microphone input to a file
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the volume of a sound is the region in which the sound will play at 100% amplitude(loudness)
//<li> falloffDistance is a region outside the volume in which the amplitude is diminished as the user moves further away from the source(volume)
//<li> recorded sounds are stored in .aiff (Audio Interchange File Format)
//<li> the sound server, snerd, must be run on the a local or remote machine
//<li> if the sound file does not exist in YG_PATH, then it will be created in the first directory in the path
//<li> sounds under a switch will turn off but must be restarted when the switch is toggled back on
//</ul>
//
// Category: Sound
// Author: Alex Hill
// Revision: 08/01/03
//
#include <pfcave.h>
#include <bergenRecord.h>
#include <ygWorld.h>
#include "ygNetKeys.h"
#include "soundRecorder.h"

using namespace std;

extern "C" ygNode* construct_soundRecorder(const char* name,bool master) { return new soundRecorder(name,master); }

struct _soundRecorderPrivateData
	{
	ygString file;
	bergenRecord * sample;
	bool isRecording;
	float curAmplitude;
	pfVec3 currPosition;
	ygString netFile;
	float updateInterval, lastUpdate;
	bool netRecording;
	};


soundRecorder::soundRecorder(const char* name,bool master) : ygSoundServer(name,master)
	{
	setClassName("soundRecorder",true);
	p_ = new struct _soundRecorderPrivateData;
	p_->sample = NULL;
	p_->isRecording = false;
	p_->curAmplitude = 0;
	p_->updateInterval = 0.1;
	p_->lastUpdate = 0;
	addNetKey("rec", &p_->netRecording, YG_NET_BOOL);
	addNetKey("file", &p_->netFile, YG_NET_STRING);
	}

soundRecorder::~soundRecorder(void)
	{
	if (p_->sample)
		delete p_->sample;
	delete p_;
	}


void soundRecorder::reset(void)
	{
	//reset recording flag to false
	p_->isRecording = false;
	//if a sample exists then delete it
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	ygSoundServer::reset();
	}


void soundRecorder::message(const ygMessage& msg)
	{
	//set the name of the sound file
	if (msg == "file")
		{
		if (msg.args.size() > 0)
			setFile(msg.args[0]);
		}
	//set the amplitude within the space
	else if (msg == "amplitude")
		setAmplitude(msg.floatArg(0));
	//set the zero amplitude distance from the space perimeter
	else if (msg == "falloffDistance")
		setFalloffDistance(msg.floatArg(0));
	//start the sound recording
	else if (msg == "record")
		{
		if (msg.args.size() > 0)
			{
			setFile(msg.args[0]);
			record();
			}
		else
			record();
		}
	//stop the recording sound
	else if (msg == "stop")
		stop();
	//set the update interval for sound server
	else if (msg == "updateInterval")
		{
		if (msg.args.size() > 0)
			p_->updateInterval = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygSoundServer::message(msg);
	}


void soundRecorder::setFile(const ygString& val)
	{
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	p_->file = val;
	p_->netFile = p_->file;
	netKeyChanged("file");
	}


void soundRecorder::setCurAmplitude(float val)
	{
	p_->curAmplitude = val;
	if (p_->sample)
		p_->sample->setAmplitude(val);
	}


bool soundRecorder::isPlaying(void)
	{
	return p_->isRecording;
	}


void soundRecorder::record(void)
	{
	if (p_->isRecording)
		return;
	//if no sample then create a new bergenRecord
	if (!p_->sample)
		p_->sample = new bergenRecord(p_->file.c_str(),server());
	//record the sample
	p_->sample->record();
	//generate a heartbeat for the server
	heartbeat();
	p_->isRecording = true;
	eventOccurred("start");
	p_->netRecording = true;
	netKeyChanged("rec");
	}


void soundRecorder::stop(bool sendStopCommand)
	{
	//if a sample is playing
	if (p_->isRecording)
		{
		//send a stop command to the server
		if ((sendStopCommand) && (p_->sample))
			p_->sample->stop();
		p_->isRecording = false;
		eventOccurred("stop");
		p_->netRecording = false;
		netKeyChanged("rec");
		}
	}


void soundRecorder::app(void)
	{
	//if a sample exists generate a heartbeat for the server
	if (p_->sample)
		{
		heartbeat();
		//if update interval has passed then update sound server
		if ((ygWorld::FrameTime - p_->lastUpdate) > p_->updateInterval)
			{
			float newAmpl = computeAmplitude();
			if (fabs(newAmpl - p_->curAmplitude) > 0.005)
				setCurAmplitude(newAmpl);
			//determine position of this sound
			pfVec3 newPos;
			newPos = origin();
			pfVec3 cavePos;
			//get position relative to CAVE
			CAVENavConvertWorldToCAVE(newPos.vec,cavePos.vec);
			if (cavePos != p_->currPosition)
				{
				if (p_->sample)
					p_->sample->setPosition(cavePos.vec[0],cavePos.vec[2],-cavePos.vec[1]);
				p_->currPosition = cavePos;
				}
			p_->lastUpdate = ygWorld::FrameTime;
			}
		}
	ygSoundServer::app();
	}


void soundRecorder::acceptNetKey(const ygString& key)
	{
	if (key == "rec")
		{
		if (p_->netRecording)
			record();
		else
			stop();
		}
	else if (key == "file")
		setFile(p_->netFile);
	else
		ygSoundServer::acceptNetKey(key);
	}
