// Description: creates any number of snow particles in a range about the user
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the range message determines the range about the user head within which snow is generated
//<li> use a switch to stop display of the snow
//</ul>
//
// Category: Geometry
// Author: Kyoung Park
//           11/01/01
// Revision: 11/05/02 Alex Hill - added network keys
//
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pf/pfDCS.h>
#include <Performer/pfdu.h>
#include <ygNetKeys.h>
#include <ygUser.h>
#include <ygWorld.h>
#include "snow.h"

extern "C" ygNode* construct_snow(const char* name,bool master) { return new snow(name,master); }

#define snowMaxSize 0.1
#define snowMinSize 0.07

snow::snow(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("snow",true);
	range_.xyz.set(-1,-1,0);
	range_.hpr.set(1,1,1);
	snow_ = NULL;
	count_ = 0;
	ground_ = 0.0;
	speed_ = 1.1;
	addNetKey("count",&count_,YG_NET_FLOAT);
	addNetKey("ground",&ground_,YG_NET_FLOAT);
	addNetKey("speed",&speed_,YG_NET_FLOAT);
	addNetKey("range",&range_,YG_NET_COORD);
	}

void snow::reset(void)
	{
	snow_ = NULL;
	//set range to (-1,-1,0) and (1,1,1)
	range_.xyz.set(-1,-1,0);
	range_.hpr.set(1,1,1);
	//set count to 0
	count_ = 0;
	//set ground to 0.0
	ground_ = 0.0;
	//set speed to 1.0
	speed_ = 1.0;
	ygNode::reset();
	}


void snow::message(const ygMessage& msg)
	{
	//set the number of snow particles
	if (msg == "count") 
		{
		count_ = msg.intArg(0);
		createGeode();	
		netKeyChanged("count");
		}
	//set the speed of decent
	else if (msg == "speed")
		{
		speed_ = msg.floatArg(0);
		netKeyChanged("speed");
		}
	//set the recycle point
	else if (msg == "ground")
		{
		ground_ = msg.floatArg(0);
		netKeyChanged("ground");
		}
	//set the minimum and maximum of the range
	else if (msg == "range")
		{
		msg.getVec3Args(range_.xyz);
		msg.getVec3Args(range_.hpr,3);
		netKeyChanged("range");
		}
	else
		ygNode::message(msg);
}

void snow::createGeode(void)
	{
	pfDCS* sphereDCS_;
	pfGeode* sphereGeode_ = makeSphere();
	pfVec3 center;
	float radius;

	snow_ = (pfVec3*)pfMalloc(count_*sizeof(pfVec3),pfGetSharedArena());

	for (int i = 0; i < count_; i++)
		{
		pfVec3 userPos = ygWorld::World->viewer()->origin(this);
		snow_[i].vec[0] = randomVal(range_.xyz[0], range_.hpr[0]) + userPos[0];
		snow_[i].vec[1] = randomVal(range_.xyz[1], range_.hpr[1]) + userPos[1];
		snow_[i].vec[2] = randomVal(range_.xyz[2], range_.hpr[2]) + userPos[2];

		center[0] = snow_[i].vec[0];
		center[1] = snow_[i].vec[1];
		center[2] = snow_[i].vec[2];
		radius = randomVal(snowMinSize,snowMaxSize);

		sphereDCS_ = new pfDCS;
		sphereDCS_->setScale(radius,radius,radius);
		sphereDCS_->setTrans(center[0],center[1],center[2]);
		sphereDCS_->addChild(sphereGeode_);
		pfnode()->addChild(sphereDCS_);
		}
	}

pfGeode* snow::makeSphere(void)
	{
	pfGeode* geode = new pfGeode;
	geode->setTravMask(PFTRAV_ISECT,0,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
	pfGeoSet* gset = pfdNewSphere(32,pfGetSharedArena());
	pfVec4* colorAttr = (pfVec4*)pfMalloc(sizeof(pfVec4),pfGetSharedArena());
	colorAttr->set(1.0,1.0,1.0,0.7);
	gset->setAttr(PFGS_COLOR4,PFGS_OVERALL,colorAttr,NULL);
	gset->setGState(new pfGeoState);

	gset->getGState()->setMode(PFSTATE_ENLIGHTING,PF_OFF);
	gset->getGState()->setMode(PFSTATE_TRANSPARENCY,PFTR_ON);
	gset->getGState()->setMode(PFSTATE_CULLFACE,PFCF_OFF);
	gset->setDrawMode(PFGS_COMPILE_GL,1);
	geode->addGSet(gset);
	return geode;
	}

void snow::app(void)
	{
	if (snow_)
		{
		pfDCS *sphereDCS_;
		pfVec3 userPos, center;

		for (int i = 0; i < count_; i++) 
			{
			// move the snow
			pfVec3 delta(0, 0, -1);
			delta *= speed_ * randomVal(0.5, 1.0) * ygWorld::FrameDeltaTime;

			snow_[i].vec[0] += delta[0];
			snow_[i].vec[1] += delta[1];
			snow_[i].vec[2] += delta[2];

			// if snow reaches ground, put it at top
			userPos = ygWorld::World->viewer()->origin(this);
			if (snow_[i].vec[2] < ground_)
				{
				pfVec3 userPos = ygWorld::World->viewer()->origin(this);
				snow_[i].vec[0] = randomVal(range_.xyz[0], range_.hpr[0]) + userPos[0];
				snow_[i].vec[1] = randomVal(range_.xyz[1], range_.hpr[1]) + userPos[1];
				snow_[i].vec[2] = randomVal(range_.xyz[2], range_.hpr[2]) + userPos[2];
				}

			center[0] = snow_[i].vec[0];
			center[1] = snow_[i].vec[1];
			center[2] = snow_[i].vec[2];
			sphereDCS_ = (pfDCS *)(pfnode()->getChild(i));
			sphereDCS_->setTrans(center[0],center[1],center[2]);
			}
		}
	ygNode::app();
}

void snow::acceptNetKey(const ygString& key)
    {
    if (key == "count")
		createGeode();
    else
		ygNode::acceptNetKey(key);
    }
