// Description: creates a rising plume of smoke
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> you must add the following to your RUN file in order to use this node:
//<ul>
//<li> setenv YG_PF_INIT_SMOKES
//<li> setenv YG_PRELOAD_CLASSES smoke
//</ul>
//</ul>
//
// Category: Geometry
// Author: Josheph Tremonti
//           11/01/01
// Revision: 10/01/04 Alex Hill - changed "type" to "mode" to avoid key problems
//
#include <Performer/pf/pfTraverser.h>
#include <Performer/pf/pfDCS.h>
#include <Performer/pf/pfGeode.h>
#include <math.h>
#include <ygNetKeys.h>
#include <ygWorld.h>
#include "smoke.h"

extern "C" ygNode* construct_smoke(const char* name,bool master) { return new smoke(name,master); }

typedef struct _smokeData
	{
	pfVec3 eye;
	} smokeData_t;

smoke::smoke(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("smoke",true);
	createSmoke();
	addNetKey("vel",&velocity,YG_NET_VEC2);
	addNetKey("den",&density,YG_NET_VEC3);
	addNetKey("col",&color,YG_NET_COORD);
	addNetKey("act",&active,YG_NET_BOOL);
	addNetKey("typ",&type,YG_NET_INT);
	reset();
	}

smoke::~smoke(void)
	{
	}

void smoke::reset(void)
	{
	density.set(1.0,1.0,0.5);
	velocity.set(3.0,0.2);
	color.xyz.set(1.0,1.0,1.0);
	color.hpr.set(1.0,1.0,1.0);
	type = PFUSMOKE_SMOKE;
	active = true;
	pfuSmokeVelocity(smoke_,velocity[1],velocity[0]);
	pfuSmokeDensity(smoke_,density[0],density[1],density[2]);
	pfVec3 origin(0,0,0);
	pfuSmokeOrigin(smoke_,origin,1.0f);
	pfVec3 direction(0,0,1);
	pfuSmokeDir(smoke_,direction);
	pfuSmokeDuration(smoke_,0xffffffff);
	pfuSmokeColor(smoke_,color.xyz,color.hpr);
	pfuSmokeType(smoke_,type);
	pfuSmokeMode(smoke_,PFUSMOKE_START);
	ygNode::reset();
	}

void smoke::message(const ygMessage& msg)
	{
	//set smoke type
	if (msg == "mode")
		{
		type = PFUSMOKE_SMOKE;
		if (msg.args[0] == "missile")
			type = PFUSMOKE_MISSILE;
		else if (msg.args[0] == "explosion")
			type = PFUSMOKE_EXPLOSION;
		else if (msg.args[0] == "fire")
			type = PFUSMOKE_FIRE;
		else if (msg.args[0] == "smoke")
			type = PFUSMOKE_SMOKE;
		else if (msg.args[0] == "dust")
			type = PFUSMOKE_DUST;
		pfuSmokeType(smoke_,type);
		netKeyChanged("typ");
		}
	//set the smoke density, dissipation, and expansion
	else if (msg == "density")
		{
		msg.getVec3Args(density);
		pfuSmokeDensity(smoke_,density[0],density[1],density[2]);
		netKeyChanged("den");
		}
	//set the smoke velocity and turbulence
	else if (msg == "velocity")
		{
		msg.getVec2Args(velocity);
		pfuSmokeVelocity(smoke_,velocity[1],velocity[0]);
		netKeyChanged("vel");
		}
	//set the smoke begining and ending color
	else if (msg == "color")
		{
		msg.getVec3Args(color.xyz);
		msg.getVec3Args(color.hpr,3);
		pfuSmokeColor(smoke_,color.xyz,color.hpr);
		netKeyChanged("col");
		}
	//turn the smoke on
	else if (msg == "on")
		{
		active = true;
		pfuSmokeMode(smoke_,PFUSMOKE_START);
		netKeyChanged("act");
		}
	//turn the smoke off
	else if (msg == "off")
		{
		active = false;
		pfuSmokeMode(smoke_,PFUSMOKE_STOP);
		netKeyChanged("act");
		}
	else
		ygNode::message(msg);
	}

void smoke::createSmoke()
	{
	pfSphere bsph;
	bsph.radius = 100.0f;
	bsph.center.set(0,0,0);
	pfGeode* geode = new pfGeode;
	geode->setBound(&bsph,PFBOUND_STATIC);
	geode->setTravFuncs(PFTRAV_DRAW,smokePreDraw,smokePostDraw);
	data_ = (smokeData_t*) pfMalloc(sizeof(smokeData_t),pfGetSharedArena());
	data_->eye.set(0,0,0);
	geode->setTravData(PFTRAV_DRAW,data_);
	geode->setTravMask(PFTRAV_ISECT,0,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
	smoke_ = pfuNewSmoke();
	pfnode()->addChild(geode);
	}

void smoke::app(void)
	{
	ygNode* viewer = ygWorld::World->viewer();
	if (viewer)
		{
		data_->eye = viewer->origin(this);
		data_->eye.normalize(); 
		}
	}
	
int smoke::smokePreDraw(pfTraverser*,void* data)
	{
	//cast the traversal data as the smokeData structure
	smokeData_t* smokeData = (smokeData_t*)data;
	glDepthMask(GL_FALSE);
	pfuDrawSmokes(smokeData->eye);
	return PFTRAV_CONT;
	}

int smoke::smokePostDraw(pfTraverser*,void* data)
	{
	glDepthMask(GL_TRUE);
	return PFTRAV_CONT;
	}

void smoke::acceptNetKey(const ygString& key)
    {
    if (key == "vel")
		pfuSmokeVelocity(smoke_,velocity[1],velocity[0]);
    else if (key == "den")
		pfuSmokeDensity(smoke_,density[0],density[1],density[2]);
    else if (key == "col")
		pfuSmokeColor(smoke_,color.xyz,color.hpr);
    else if (key == "typ")
		pfuSmokeType(smoke_,type);
    else if (key == "act")
		{
		if (active)
			pfuSmokeMode(smoke_,PFUSMOKE_START);
		else
			pfuSmokeMode(smoke_,PFUSMOKE_STOP);
		}
    else
		ygNode::acceptNetKey(key);
    }
