// Description: a transform that reflects scene geometry about an axis
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the default axis is the Y axis (0,1,0)
//<li> the scene node should neither be a transform nor have a parent transform
//<li> you must add the following to your RUN file in order to use this node:<br>
//<ul>
//<li> setenv YG_PRELOAD_CLASSES reflection
//</ul>
//</ul>
//
// Category: Transformation
// Author: Alex Hill
// Revision: 11/10/01
//
#include <ygNodeDB.h>
#include <ygNetKeys.h>
#include "reflection.h"


extern "C" ygNode* construct_reflection(const char* name,bool master) { return new reflection(name,master); }


reflection::reflection(const char* name,bool master) : ygTransform(name,master)
	{
	setClassName("reflection",true);
	pfnode()->setTravFuncs(PFTRAV_DRAW,preDrawFunc,postDrawFunc);
	reflection_.makeIdent();
	axis.set(0,1,0);
	setAxis(axis);
	dcs = new pfDCS();
	pfnode()->addChild(dcs);
	//distribute the nodeNames string
	addNetKey("node",&nodeNames,YG_NET_STRING);
	//distribute the reflection axis
	addNetKey("axis",&axis,YG_NET_VEC3);
	}

reflection::~reflection(void)
	{
	}

void reflection::reset(void)
	{
	//clear nodeNames string
	nodeNames.clear();
	setNode(nodeNames);
	//reset the reflection axis to the Y axis (0,1,0)
	axis.set(0,1,0);
	setAxis(axis);
	ygTransform::reset();
	}

void reflection::message(const ygMessage& msg)
	{
	//set the name of the node to reflect
	if (msg == "node")
		{
		if (msg.args.size() == 1)
			{
			nodeNames = msg.stringArg(0);
			setNode(nodeNames);
			netKeyChanged("node");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//add the name of the node to reflect
	else if (msg == "addNode")
		{
		if (msg.args.size() == 1)
			{
			nodeNames += " ";
			nodeNames += msg.stringArg(0);
			setNode(nodeNames);
			netKeyChanged("node");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the axis for reflection
	else if (msg == "axis")
		{
		msg.getVec3Args(axis);
		setAxis(axis);
		netKeyChanged("axis");
		}
	else
		ygTransform::message(msg);
	}

void reflection::setAxis(const pfVec3& val)
	{
	axis = val;
	if (axis[0] == 1.0)
		reflection_.setRow(0,-1,0,0,0);
	else
		reflection_.setRow(0,1,0,0,0);
	if (axis[1] == 1.0)
		reflection_.setRow(1,0,-1,0,0);
	else
		reflection_.setRow(1,0,1,0,0);
	if (axis[2] == 1.0)
		reflection_.setRow(2,0,0,-1,0);
	else
		reflection_.setRow(2,0,0,1,0);
	}
	
void reflection::app(void)
	{
	pfMatrix selfMatrix;
	getTransform(selfMatrix);
	pfMatrix inverseSelfMatrix;
	inverseSelfMatrix.invertOrtho(selfMatrix);
	pfMatrix matrix;
	matrix = inverseSelfMatrix * reflection_;
	dcs->setMat(matrix);
	ygTransform::app();
	}

void reflection::setNode(const ygString& name)
	{
	//delete all existing DCS children
	while (dcs->getNumChildren() > 0)
		dcs->removeChild(dcs->getChild(0));
	//set nodeNames to the input string
	nodeNames = name;
	ygString* nodeName;
	ygNode* node;
	int i=0;
	while (nodeName = nodeNames.nextToken(" ",&i))
		{
		//find the associated node in the database
		node = ygNodeDB::find(*nodeName);
		//if the node was found then add the associated Performer node as a child of the DCS
		if (node)
			dcs->addChild(node->pfnode());
		delete nodeName;
		}
	}

int reflection::preDrawFunc(pfTraverser* trav,void* data)
	{
	glFrontFace(GL_CW);
	return PFTRAV_CONT;
	}
	
int reflection::postDrawFunc(pfTraverser* trav,void* data)
	{
	glFrontFace(GL_CCW);
	return PFTRAV_CONT;
	}

void reflection::acceptNetKey(const ygString& key)
	{
	if (key == "node")
		setNode(nodeNames);
	else if (key == "axis")
		setAxis(axis);
	else
		ygTransform::acceptNetKey(key);
	}
