//Description: creates a rectangular surface
//
//<b>notes:</b>
//<ul>
//<li> this node currently is supported only on LINUX machines
//<li> derived nodes inherit all messages from their base classes
//</ul>
//
// Category: Geometry
// Author: Kalle Jalkanen and Jonas Westling, Interactive Institute Ume, Sweden
//           02/24/03
// Revision: 08/10/03 Helen-Nicole Kostis - documentation
//
//
#include <errno.h>
#include "rectSurface.h"


using namespace std;


extern "C" ygNode* construct_rectSurface(const char* name,bool master) 
{ 
	return new rectSurface(name,master); 
}


struct _rectSurfacePrivateData
{
	pfDCS *dcs;
	pfGeoSet *gset;				
	pfGeoState *gstate;
	pfGeode *geode;  
  	
	pfMaterial *mat;  
   	pfTexture *tex;   
	pfVec3 *coords;   
	pfVec2 *texs;		

	bool coordsLoaded;
	ygString sizeString; 
	unsigned char *img;
	int imgWidth, imgHeight;	
	int texWidth2, texHeight2;  
	int nColComp;

	bool snapToUser;
};




int log2(int x) 
{
	int a = -1;
	while (x > 0) 
	{
		x>>=1;
        	a++;
   	}
   return a;
}


rectSurface::rectSurface(const char* name, bool master) : ygNode(name, master)
{
	setClassName("rectSurface", true); 
   	_p = new struct _rectSurfacePrivateData;

	_p->dcs = new pfDCS();
	setPfNode(_p->dcs);
	
	_p->gset = NULL;				
	_p->gstate = NULL;
	_p->geode = NULL;  
  	
	_p->mat = NULL; 
   	_p->tex = NULL; 
		
	_p->coords = (pfVec3*) pfMalloc(4 * sizeof(pfVec3), pfGetSharedArena());
	_p->coords[0].set(-1, 0, 1);
	_p->coords[1].set(1, 0, 1);
	_p->coords[2].set(1, 0, 0);
	_p->coords[3].set(-1, 0, 0);
	_p->coordsLoaded = false;
	
	_p->texs = (pfVec2*) pfMalloc(4 * sizeof(pfVec2), pfGetSharedArena());	
	_p->texs[0].set(0, 0);
	_p->texs[1].set(1, 0);
	_p->texs[2].set(1, 1);
	_p->texs[3].set(0, 1);
	
	_p->sizeString = "";
	addNetKey("coords", &_p->sizeString, YG_NET_STRING);
	

	_p->imgWidth = 0;
	_p->imgHeight = 0;
	_p->img = NULL;	
	_p->nColComp = 0;

	resizeImg(128, 128, 3);	
	clear(0L);

	_p->snapToUser = false;
	addNetKey("snaptouser", &_p->snapToUser, YG_NET_BOOL);

}



rectSurface::~rectSurface()
{
	pfDelete(_p->gset);
	pfDelete(_p->gstate);
	pfDelete(_p->geode);
	pfDelete(_p->mat);
	pfDelete(_p->tex);
	pfDelete(_p->coords);
	pfDelete(_p->texs);

	if (_p->img)
   	pfDelete(_p->img);	
	
   	delete _p;
}


void rectSurface::reset()
{
	ygNode::reset();
}




bool rectSurface::resizeImg(int w, int h, int ncolc)
{
	bool _changed = false;
	
	if ((_p->imgWidth!=w) || (_p->imgHeight!=h) || (_p->nColComp!=ncolc)) {

		
		if (w*h*ncolc > 0) {
			char log2w = (char)log2(w);
			char log2h = (char)log2(h);
			if (w > (1<<log2w)) log2w++;
			if (h > (1<<log2h)) log2h++;
				
			_p->texWidth2 = 1 << log2w;
		        _p->texHeight2 = 1 << log2h;

			float s,t,s1,t1;
			s = 0;
			t = 0;
			s1 = (float)w/(float)_p->texWidth2; 
			t1 = (float)h/(float)_p->texHeight2; 
			_p->texs[0].set(s, t);
			_p->texs[1].set(s1, t);
			_p->texs[2].set(s1, t1);
			_p->texs[3].set(s, t1);

			if (_p->gset) {
				_p->gset->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, _p->texs, NULL);
			}
			
			_p->imgWidth = w;
			_p->imgHeight = h;
			_p->nColComp = ncolc;
			
			_changed = true;
		}
	}
	
	if (_changed) {	
		if (_p->img) pfDelete(_p->img);
		_p->img = (unsigned char *)pfMalloc(_p->texWidth2 * _p->texHeight2 * 
					_p->nColComp, pfGetSharedArena());
	}
	
	return _changed;
}


void rectSurface::clear(unsigned int col)
{	
	int i=0;

	if (!_p->img) return;
	if (!_p->tex) return;
	
	if (_p->nColComp == 1) {
		unsigned char _col = (unsigned char)col;
		memset(_p->img, _col, _p->texWidth2*_p->texHeight2);
	}
	if (_p->nColComp == 2) {
		unsigned short _col = (unsigned short)col;
		unsigned short *p = (unsigned short *)_p->img;
		for ( ; i<_p->texWidth2*_p->texHeight2; ++i){
			p[i] = _col;
		}
	}
	if (_p->nColComp == 3) {
		unsigned char c1 = (col>>16) & 0x000000ff;
		unsigned char c2 = (col>>8) & 0x000000ff;
		unsigned char c3 = (col) & 0x000000ff;
		for ( ; i<_p->texWidth2*_p->texHeight2; ++i){
			_p->img[3*i] = c3;
			_p->img[3*i+1] = c2;
			_p->img[3*i+2] = c1;
		}

	}
	if (_p->nColComp == 4) {
		unsigned int _col = col;
		unsigned int *p = (unsigned int *)_p->img;
		for ( ; i<_p->texWidth2*_p->texHeight2; ++i){
			p[i] = _col;
		}
	}	
	
}	



void rectSurface::setImg(unsigned char *img, int w, int h, int ncolc)
{	
	bool _sizeChanged = resizeImg(w, h, ncolc);

	if (!_p->img) return;
	if (!_p->tex) return;
	
	if (img && (_p->texWidth2==w) && (_p->texHeight2==h)) {
		_p->tex->setImage((uint *)img, _p->nColComp, _p->texWidth2, _p->texHeight2, 1);

	} else {

		loadSubImg(img, w, h, ncolc);
		_p->tex->setImage((uint *)_p->img, _p->nColComp, _p->texWidth2, _p->texHeight2, 1);
	}

	if (_p->nColComp == 3) {
		_p->tex->setFormat(PFTEX_IMAGE_FORMAT, PFTEX_RGB);
		_p->tex->setFormat(PFTEX_INTERNAL_FORMAT, PFTEX_RGB_8);
	} else if (_p->nColComp == 4) {
		_p->tex->setFormat(PFTEX_IMAGE_FORMAT, PFTEX_RGBA);
		_p->tex->setFormat(PFTEX_INTERNAL_FORMAT, PFTEX_RGBA_8);
	}
}


void rectSurface::loadSubImg(unsigned char *img, int w, int h, int ncolc)
{
	if (img && _p->img) {
		int x,y,i;
		int od, os;
		
		for (y=0; y<h; ++y) {
			od = ncolc * _p->texWidth2 * y;
			os = ncolc * w * y;
				
			for (x=0; x<w; ++x) {

				for (i=0; i<ncolc; ++i) {
					_p->img[od+i] = img[os+i];
				}

				od += ncolc;
				os += ncolc;
			}
		}

	}
}




void rectSurface::setCoords(pfVec3& c1, pfVec3& c2, pfVec3& c3, pfVec3& c4)
{

	_p->coords[0].set(c1[0], c1[1], c1[2]);
	_p->coords[1].set(c2[0], c2[1], c2[2]);
   	_p->coords[2].set(c3[0], c3[1], c3[2]);
   	_p->coords[3].set(c4[0], c4[1], c4[2]);

	if (_p->gset)
		_p->gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, _p->coords, NULL);

}


void rectSurface::message(const ygMessage& msg)
{
	if (msg == "coords") 
	{
     		pfVec3 c1, c2, c3, c4;
      		if (msg.args.size() != 12) 
		{
         		msg.error("","(wrong number of arguments -"
			" should be 'coords X1 Y1 Z1 X2 Y2 Z2 X3 Y3 Z3 X4 Y4 Z4')");
     		 } 
		else {
			msg.getVec3Args(c1, 0);
			msg.getVec3Args(c2, 3);
			msg.getVec3Args(c3, 6);
			msg.getVec3Args(c4, 9);
			setCoords(c1, c2, c3, c4);
			msg.writeString(_p->sizeString);
			netKeyChanged("coords");
			_p->coordsLoaded = true;
		}
	} 
	else if (msg == "snaptouser")
 	{
		if (msg.args.size() > 0) 
		{
			_p->snapToUser = msg.boolArg(0);
		}
		else 
		{
			_p->snapToUser = true;		
		}
		netKeyChanged("snaptouser");

		if ( !_p->snapToUser) 
		{
			pfMatrix mat;
			mat.makeIdent();
			_p->dcs->setMat( mat );
		}
		
	} 
	else 
	{
		
		ygNode::message(msg);
	}

}



void rectSurface::createrectSurface(void)
{
	_p->gset = new pfGeoSet;
	_p->gstate = new pfGeoState;
	_p->geode = new pfGeode;


	_p->mat = new pfMaterial;
	_p->mat->setColor(PFMTL_AMBIENT, 1.0f, 1.0f, 1.0f);
	_p->mat->setColor(PFMTL_DIFFUSE, 1.0, 1.0, 1.0);
	_p->mat->setColorMode(PFMTL_BOTH, PFMTL_CMODE_AD);	
	_p->tex = new pfTexture;


	if (_p->img) {
		_p->tex->setImage((uint *)_p->img, _p->nColComp, _p->texWidth2, _p->texHeight2, 1);
	}

	_p->tex->setFilter(PFTEX_MINFILTER, PFTEX_LINEAR);
	
	if (_p->nColComp == 3) {
		_p->tex->setFormat(PFTEX_IMAGE_FORMAT, PFTEX_RGB);
		_p->tex->setFormat(PFTEX_INTERNAL_FORMAT, PFTEX_RGB_8);
	} else if (_p->nColComp == 4) {
		_p->tex->setFormat(PFTEX_IMAGE_FORMAT, PFTEX_RGBA);
		_p->tex->setFormat(PFTEX_INTERNAL_FORMAT, PFTEX_RGBA_8);
	}

	_p->gstate->setAttr(PFSTATE_TEXTURE, _p->tex);
	_p->gstate->setAttr(PFSTATE_FRONTMTL, _p->mat);
	_p->gstate->setMode(PFSTATE_ENTEXTURE, PF_ON);
	_p->gstate->setMode(PFSTATE_ENLIGHTING, PF_OFF);
	_p->gstate->setMode(PFSTATE_CULLFACE, PFCF_OFF);   
	_p->gset->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, _p->coords, NULL);
	_p->gset->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, _p->texs, NULL);
	_p->gset->setPrimType(PFGS_QUADS);
	_p->gset->setNumPrims(1);	
	_p->gset->setGState(_p->gstate);   
	_p->geode->addGSet(_p->gset);	
	pfnode()->addChild(_p->geode);

}

bool rectSurface::isCreated() 
{ 
	return (_p->geode != NULL); 
}


void rectSurface::setVisibility(bool vis)
{
	if (vis) {
		pfnode()->setTravMask(PFTRAV_DRAW,0xffffffff,PFTRAV_SELF,PF_SET);
	} else {
		pfnode()->setTravMask(PFTRAV_DRAW,0,PFTRAV_SELF,PF_SET);
	}
}


 
void rectSurface::app(void)
{

	if (!_p->geode && _p->coordsLoaded) {
		createrectSurface();
		cout << "rectSurface created.\n";
	}

	
	if (_p->snapToUser) {
		ygNode* viewer = ygWorld::World->viewer();
		if (viewer) {
			pfMatrix mat;
			ygCAVEViewer *cv = (ygCAVEViewer *)viewer;
			cv->getMatrix( mat );
			_p->dcs->setMat( mat );
		}
	}

	ygNode::app();
}



void rectSurface::acceptNetKey(const ygString& key)
{
	if (key == "coords") {
		cout << "key coords\n" << endl;
		pfVec3 c1, c2, c3, c4;
		ygMessage msg;
		msg.parseString(_p->sizeString);
		if (msg.args.size() != 12) {
         msg.error("","(wrong number of arguments -"
			" should be 'coords X1 Y1 Z1 X2 Y2 Z2 X3 Y3 Z3 X4 Y4 Z4')");
      } else {
			msg.getVec3Args(c1, 0);
			msg.getVec3Args(c2, 3);
			msg.getVec3Args(c3, 6);
			msg.getVec3Args(c4, 9);
			setCoords(c1, c2, c3, c4);
			_p->coordsLoaded = true;
		}
	} else if (key == "snaptouser") {
		if ( !_p->snapToUser) {
			pfMatrix mat;
			mat.makeIdent();
			_p->dcs->setMat( mat );
		}

	} else
		ygNode::acceptNetKey(key);
}





