// Description: creates a positional sound within a space using a RAT source
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the volume of a sound is the region in which the sound will play at 100% amplitude(loudness)
//<li> falloffDistance is a region outside the volume in which the amplitude is diminished as the user moves further away from the source(volume)
//<li> the sound server, snerd, must be run on the a local or remote machine
//<li> the RAT application must be running on the same machine as the sound server
//<li> sounds under a switch will turn off but must be restarted when the switch is toggled back on
//</ul>
//
// Category: Sound
// Author: Alex Hill
// Revision: 08/01/03
//
#include <pfcave.h>
#include <bergenRATSource.h>
#include "ygWorld.h"
#include "ygNetKeys.h"
#include "ratSource.h"

using namespace std;

extern "C" ygNode* construct_ratSource(const char* name,bool master) { return new ratSource(name,master); }

struct _ratSourcePrivateData
	{
	ygString source;
	bergenRATSource * sample;
	bool isPlaying;
	float curAmplitude;
	pfVec3 currPosition;
	ygString netSource;
	float updateInterval, lastUpdate;
	bool netPlay;
	};


ratSource::ratSource(const char* name,bool master) : ygSoundServer(name,master)
	{
	setClassName("ratSource",true);
	p_ = new struct _ratSourcePrivateData;
	p_->sample = NULL;
	p_->curAmplitude = 0;
	p_->updateInterval = 0.1;
	p_->lastUpdate = 0;
	p_->isPlaying = false;
	addNetKey("play", &p_->netPlay, YG_NET_BOOL);
	addNetKey("src", &p_->netSource, YG_NET_STRING);
	}

ratSource::~ratSource(void)
	{
	if (p_->sample)
		delete p_->sample;
	delete p_;
	}


void ratSource::reset(void)
	{
	//reset playing flag to false
	p_->isPlaying = false;
	//set current amplitude to 0.0
	p_->curAmplitude = 0;
	//set update interval to 0.1
	p_->updateInterval = 0.1;
	//if a sample exists then delete it
	if (p_->sample)
		{
		stop();
		delete p_->sample;
		p_->sample = NULL;
		}
	ygSpace::reset();
	}


void ratSource::message(const ygMessage& msg)
	{
	//set the RAT source identifier (SSRC)
	if (msg == "source")
		{
		if (msg.args.size() > 0)
			setSource(msg.args[0]);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the amplitude within the space
	else if (msg == "amplitude")
		{
		if (msg.args.size() > 0)
			setAmplitude(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set the zero amplitude distance from the space perimeter
	else if (msg == "falloffDistance")
		{
		if (msg.args.size() > 0)
			setFalloffDistance(msg.floatArg(0));
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//start the RAT spatialization with source and amplitude
	else if (msg == "play")
		{
		if (msg.args.size() > 0)
			{
			setSource(msg.args[0]);
			if (msg.args.size() > 1)
				{
				setAmplitude(msg.floatArg(1));
				play();
				}
			else
				play();
			}
		else
			play();
		}
	//stop the playing sound
	else if (msg == "stop")
		stop();
	//set the update interval for sound server
	else if (msg == "updateInterval")
		{
		if (msg.args.size() > 0)
			p_->updateInterval = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygSoundServer::message(msg);
	}


void ratSource::setSource(const ygString& val)
	{
	p_->source = val;
	p_->netSource = p_->source;
	netKeyChanged("src");
	if (p_->sample)
		p_->sample->setSource(p_->source.c_str());
	}


void ratSource::setCurAmplitude(float val)
	{
	p_->curAmplitude = val;
	if (p_->sample)
		p_->sample->setAmplitude(val);
	}

bool ratSource::isPlaying(void)
	{
	return p_->isPlaying;
	}

void ratSource::play(void)
	{
	if (p_->isPlaying)
		return;
	//if no sample then create a new bergenRATSource
	if (!p_->sample)
		p_->sample = new bergenRATSource(p_->source.c_str(),server());
	setCurAmplitude(computeAmplitude());
	//play the sample
	p_->sample->set3DEnable(TRUE);
	p_->sample->play();
	//generate a heartbeat for the server
	heartbeat();
	p_->isPlaying = true;
	eventOccurred("start");
	p_->netPlay = true;
	netKeyChanged("play");
	}


void ratSource::stop(bool sendStopCommand)
	{
	//if a sample is playing
	if (p_->isPlaying)
		{
		//send a stop command to the server
		if ((sendStopCommand) && (p_->sample))
			{
			p_->sample->stop();
			setCurAmplitude(0.0);
			}
		p_->isPlaying = false;
		eventOccurred("stop");
		p_->netPlay = false;
		netKeyChanged("play");
		}
	}


void ratSource::app(void)
	{
	//if a sample exists
	if (p_->sample)
		{
		//generate a heartbeat for the server
		heartbeat();
		//if the sample is playing
		if (p_->isPlaying)
			{
			//if update interval has passed then update sound server
			if ((ygWorld::FrameTime - p_->lastUpdate) > p_->updateInterval)
				{
				float newAmpl = computeAmplitude();
				if (fabs(newAmpl - p_->curAmplitude) > 0.005)
					setCurAmplitude(newAmpl);
				//determine position of this sound
				pfVec3 newPos;
				newPos = origin();
				pfVec3 cavePos;
				//get position relative to CAVE
				CAVENavConvertWorldToCAVE(newPos.vec,cavePos.vec);
				if (cavePos != p_->currPosition)
					{
					if (p_->sample)
						p_->sample->setPosition(cavePos.vec[0],cavePos.vec[2],-cavePos.vec[1]);
					p_->currPosition = cavePos;
					}
				p_->lastUpdate = ygWorld::FrameTime;
				}
			}
		}
	ygSoundServer::app();
	}


void ratSource::acceptNetKey(const ygString& key)
	{
	if (key == "play")
		{
		if (p_->netPlay)
			play();
		else
			stop();
		}
	else if (key == "src")
		setSource(p_->netSource);
	else
		ygSoundServer::acceptNetKey(key);
	}
