// Description: creates any number of rain particles in a range about the user
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the range message determines the range about the user head within which rain is generated
//<li> use a switch to stop display of the rain
//</ul>
//
// Category: Geometry
// Author: Kyoung Park
//           11/01/01
// Revision: 11/05/02 Alex Hill - added network keys
//
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pf/pfDCS.h>
#include <Performer/pfdu.h>
#include <ygNetKeys.h>
#include <ygUser.h>
#include <ygWorld.h>
#include "rain.h"

extern "C" ygNode* construct_rain(const char* name,bool master) { return new rain(name,master); }

#define rainMaxSize 0.1
#define rainMinSize 0.07

rain::rain(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("rain",true);
	range_.xyz.set(-1,-1,0);
	range_.hpr.set(1,1,1);
	rain_ = NULL;
	count_ = 0;
	ground_ = 0.0;
	speed_ = 1.1;
	addNetKey("count",&count_,YG_NET_FLOAT);
	addNetKey("ground",&ground_,YG_NET_FLOAT);
	addNetKey("speed",&speed_,YG_NET_FLOAT);
	addNetKey("range",&range_,YG_NET_COORD);
	}

void rain::reset(void)
	{
	rain_ = NULL;
	//set range to (-1,-1,0) and (1,1,1)
	range_.xyz.set(-1,-1,0);
	range_.hpr.set(1,1,1);
	//set count to 0
	count_ = 0;
	//set ground to 0.0
	ground_ = 0.0;
	//set speed to 1.0
	speed_ = 1.0;
	ygNode::reset();
	}


void rain::message(const ygMessage& msg)
	{
	//set the number of rain particles
	if (msg == "count") 
		{
		count_ = msg.intArg(0);
		createGeode();	
		netKeyChanged("count");
		}
	//set the speed of decent
	else if (msg == "speed")
		{
		speed_ = msg.floatArg(0);
		netKeyChanged("speed");
		}
	//set the recycle point
	else if (msg == "ground")
		{
		ground_ = msg.floatArg(0);
		netKeyChanged("ground");
		}
	//set the minimum and maximum of the range
	else if (msg == "range")
		{
		msg.getVec3Args(range_.xyz);
		msg.getVec3Args(range_.hpr,3);
		netKeyChanged("range");
		}
	else
		ygNode::message(msg);
}

void rain::createGeode(void)
	{
	pfDCS* cylinderDCS_;
	pfGeode* cylinderGeode_ = makeCylinder();
	pfVec3 center;
	float radius;

	rain_ = (pfVec3*)pfMalloc(count_*sizeof(pfVec3),pfGetSharedArena());

	for (int i = 0; i < count_; i++)
		{
		pfVec3 userPos = ygWorld::World->viewer()->origin(this);
		rain_[i].vec[0] = randomVal(range_.xyz[0], range_.hpr[0]) + userPos[0];
		rain_[i].vec[1] = randomVal(range_.xyz[1], range_.hpr[1]) + userPos[1];
		rain_[i].vec[2] = randomVal(range_.xyz[2], range_.hpr[2]) + userPos[2];

		center[0] = rain_[i].vec[0];
		center[1] = rain_[i].vec[1];
		center[2] = rain_[i].vec[2];
		radius = randomVal(rainMinSize,rainMaxSize);

		cylinderDCS_ = new pfDCS;
		cylinderDCS_->setScale(0.15*radius,0.15*radius,3.5*radius);
		cylinderDCS_->setTrans(center[0],center[1],center[2]);
		cylinderDCS_->addChild(cylinderGeode_);
		pfnode()->addChild(cylinderDCS_);
		}
	}

pfGeode* rain::makeCylinder(void)
	{
	pfGeode* geode = new pfGeode;
	geode->setTravMask(PFTRAV_ISECT,0,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
	pfGeoSet* gset = pfdNewPipe(1.0,1.0,24,pfGetSharedArena());
	pfVec4* colorAttr = (pfVec4*)pfMalloc(sizeof(pfVec4),pfGetSharedArena());
	colorAttr->set(1.0,1.0,1.0,0.7);
	gset->setAttr(PFGS_COLOR4,PFGS_OVERALL,colorAttr,NULL);
	gset->setGState(new pfGeoState);

	gset->getGState()->setMode(PFSTATE_ENLIGHTING,PF_OFF);
	gset->getGState()->setMode(PFSTATE_TRANSPARENCY,PFTR_ON);
	gset->getGState()->setMode(PFSTATE_CULLFACE,PFCF_OFF);
	gset->setDrawMode(PFGS_COMPILE_GL,1);
	geode->addGSet(gset);
	return geode;
	}

void rain::app(void)
	{
	if (rain_)
		{
		pfDCS *cylinderDCS_;
		pfVec3 userPos, center;

		for (int i = 0; i < count_; i++) 
			{
			// move the rain
			pfVec3 delta(0, 0, -1);
			delta *= speed_ * randomVal(0.5, 1.0) * ygWorld::FrameDeltaTime;

			rain_[i].vec[0] += delta[0];
			rain_[i].vec[1] += delta[1];
			rain_[i].vec[2] += delta[2];

			// if rain reaches ground, put it at top
			userPos = ygWorld::World->viewer()->origin(this);
			if (rain_[i].vec[2] < ground_)
				{
				pfVec3 userPos = ygWorld::World->viewer()->origin(this);
				rain_[i].vec[0] = randomVal(range_.xyz[0], range_.hpr[0]) + userPos[0];
				rain_[i].vec[1] = randomVal(range_.xyz[1], range_.hpr[1]) + userPos[1];
				rain_[i].vec[2] = randomVal(range_.xyz[2], range_.hpr[2]) + userPos[2];
				}

			center[0] = rain_[i].vec[0];
			center[1] = rain_[i].vec[1];
			center[2] = rain_[i].vec[2];
			cylinderDCS_ = (pfDCS *)(pfnode()->getChild(i));
			cylinderDCS_->setTrans(center[0],center[1],center[2]);
			}
		}
	ygNode::app();
}

void rain::acceptNetKey(const ygString& key)
    {
    if (key == "count")
		createGeode();
    else
		ygNode::acceptNetKey(key);
    }
