// Description: creates a number of falling particles in a range about the user
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> you must add the following to your RUN file in order to use this node:
//<ul>
//<li> setenv YG_PRELOAD_CLASSES quadparts
//</ul>
//</ul>
//
// Category: Geometry
// Author: Dave Pape
//           11/01/01
// Revision: 10/25/03 Kyoung Park - added "file" message to use any texture
//
#include <Performer/pf/pfGroup.h>
#include <Performer/pf/pfGeode.h>
#include <Performer/pr/pfGeoSet.h>
#include <Performer/pr/pfTexture.h>
#include <Performer/pf/pfTraverser.h>
#include <ygUtil.h>
#include <ygWorld.h>
#include "quadparts.h"

using namespace std;

extern "C" ygNode* construct_quadparts(const char* name,bool master) { return new quadparts(name,master); }

static int zUpdateOff(pfTraverser *, void *data);
static int zUpdateOn(pfTraverser *, void *data);


quadparts::quadparts(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("quadparts");
	range_.min.set(-1, -1, 0);
	range_.max.set(1, 1, 1);
	numParticles_ = 100;
	nextParticle_ = 0;
	size_ = 1;
	tex_ = NULL;
	firstFrame_ = true;
	}

quadparts::~quadparts(void)
	{
	}


void quadparts::reset(void)
	{
	ygNode::reset();
	}


void quadparts::message(const ygMessage& msg)
	{
	if (msg == "range")
		{
		msg.getVec3Args(range_.min);
		msg.getVec3Args(range_.max,3);
		}
	else if (msg == "numparticles")
		numParticles_ = msg.intArg(0);
	else if (msg == "size")
		size_ = msg.floatArg(0);
	else if (msg == "file")
		if (msg.args.size() == 1) 
			loadTex(msg.args[0]);
	else
		ygNode::message(msg);
	}

void quadparts::loadTex(const ygString& file)
	{
        char *searchPath = ".";
        ygString fullPath;
        if (getenv("YG_PATH"))
                searchPath = getenv("YG_PATH");
        if (!ygFindFile(file, searchPath, fullPath))
        	{
                fprintf(stderr, "quadparts::loadTex: could not find file %s\n",
                        file.c_str());
                return;
        	}
 
        if (!tex_)
                tex_ = new pfTexture;
        tex_->loadFile(file.c_str());
	}

void quadparts::app(void)
	{
	if (firstFrame_)
		{
		createGeode();
		firstFrame_ = false;
		}
	int i;
	pfVec3 delta(0, 0, -1);
	if (nextParticle_ < numParticles_)
	    {
	    initParticle(nextParticle_);
	    nextParticle_++;
	    gset_->setNumPrims(nextParticle_);
	    }
	delta *= ygWorld::FrameDeltaTime;
	for (i = 0; i < nextParticle_; i++)
	    {
	    if (verts_[i*4][PF_Z] < -1)
		    initParticle(i);
	    verts_[i*4] += delta;
	    verts_[i*4+1] += delta;
	    verts_[i*4+2] += delta;
	    verts_[i*4+3] += delta;
	    }
	ygNode::app();
	}

void quadparts::createGeode(void)
{
 pfGeode *geode = new pfGeode;
 pfGeoState *gstate = new pfGeoState;
 int numVerts = numParticles_*4, i;
 pfSphere boundSphere;
 pfVec2 *texcoords;
 pfVec4 *colors;
 verts_ = (pfVec3*) pfMalloc(numVerts * sizeof(pfVec3), pfGetSharedArena());
 colors = (pfVec4*) pfMalloc(sizeof(pfVec4), pfGetSharedArena());
 colors[0].set(1, 1, 1, 1);
 texcoords = (pfVec2*) pfMalloc(numVerts * sizeof(pfVec2), pfGetSharedArena());
 for (i=0; i < numParticles_; i++)
	{
	texcoords[i*4].set(0,0);
	texcoords[i*4+1].set(0,1);
	texcoords[i*4+2].set(1,1);
	texcoords[i*4+3].set(1,0);
	}
 gset_ = new pfGeoSet;
 gset_->setPrimType(PFGS_QUADS);
 gset_->setNumPrims(0);
 gset_->setAttr(PFGS_COORD3, PFGS_PER_VERTEX, verts_, NULL);
 gset_->setAttr(PFGS_COLOR4, PFGS_OVERALL, colors, NULL);
 gstate->setMode(PFSTATE_ENLIGHTING, PF_OFF);
 gset_->setAttr(PFGS_TEXCOORD2, PFGS_PER_VERTEX, texcoords, NULL);
 gstate->setAttr(PFSTATE_TEXTURE, tex_);
 gstate->setMode(PFSTATE_ENTEXTURE, PF_ON);
 gstate->setMode(PFSTATE_TRANSPARENCY, PFTR_ON);
 gset_->setGState(gstate);
 gset_->setDrawBin(PFSORT_TRANSP_BIN);
 geode->addGSet(gset_);
 boundSphere.center.set(0, 0, 0);
 boundSphere.radius = 1000000;
 geode->setBound(&boundSphere, PFBOUND_STATIC);
 geode->setTravFuncs(PFTRAV_DRAW, zUpdateOff, zUpdateOn);
 geode->setTravMask(PFTRAV_ISECT, 0, PFTRAV_SELF|PFTRAV_DESCEND, PF_SET);
 pfnode()->addChild(geode);
}


static int zUpdateOff(pfTraverser *, void *data)
{
 glDepthMask(GL_FALSE);
 return PFTRAV_CONT;
}


static int zUpdateOn(pfTraverser *, void *data)
{
 glDepthMask(GL_TRUE);
 return PFTRAV_CONT;
}


static float randomVal(float min, float max)
{
 return min + drand48()*(max-min);
}


void quadparts::initParticle(int npart)
{
 pfVec3 pt1(1,0,0), pt2(1,0,1), pt3(0,0,1);
 float size = (drand48()+0.5f) * size_;
 verts_[npart*4][0] = randomVal(range_.min[0], range_.max[0]);
 verts_[npart*4][1] = randomVal(range_.min[1], range_.max[1]);
 verts_[npart*4][2] = randomVal(range_.min[2], range_.max[2]);
 verts_[npart*4] += ygWorld::World->viewer()->origin(this);
 pt1 *= size;
 verts_[npart*4+1] = verts_[npart*4] + pt1;
 pt2 *= size;
 verts_[npart*4+2] = verts_[npart*4] + pt2;
 pt3 *= size;
 verts_[npart*4+3] = verts_[npart*4] + pt3;
}
