// Description: a valuator that can only take positions and orientations listed in a file
//
//<b>notes:</b>
//<ul>
//<li> all parent node messages are inherited
//<li> see reset method for default settings
//</ul>
//
// Category: Math
// Author: Alex Hill
// Revision: 11/01/01
//
#include <iostream>
#include <ygUtil.h>
#include "positioner.h"

using namespace std;

extern "C" ygNode* construct_positioner(const char* name,bool master) { return new positioner(name,master); }

typedef struct pointData_
	{
	pfVec3 pos;
	pfVec3 ori;
	pfVec3 vec;
	bool occupied;
	} pointData;

struct _positionerPrivateData
	{
	map<ygString,pointList*> lists;
	};

struct _positionerPrivateData* positioner::p_ = NULL;

positioner::positioner(const char* name,bool master) : valuator(name,master)
	{
	setClassName("positioner");
	points = NULL;
	lastPoint = NULL;
	}

positioner::~positioner(void)
	{
	}

void positioner::reset(void)
	{
	//reset scale factor to 1.0
	scaleFactor = 1.0;
	//reset average distance calculation
	averageDist = 0.0;
	//clear the points list
	if (points && lastPoint)
		{
		for (int i=0; i<points->size(); i++)
			{
			pointData* currPoint = (*points)[i];
			if (currPoint->pos == lastPoint->pos)
				currPoint->occupied = false;
			}
		}
	points = NULL;
	lastPoint = NULL;
	valuator::reset();
	}

void positioner::message(const ygMessage& msg)
	{
	//load a position point file
	if (msg == "file")
		{
		if (msg.args.size() == 1)
			{
			if (points && lastPoint)
				{
				for (int i=0; i<points->size(); i++)
					{
					pointData* currPoint = (*points)[i];
					if (currPoint->pos == lastPoint->pos)
						currPoint->occupied = false;
					}
				lastPoint = NULL;
				}
			points = listReference(msg.args[0]);
			if (points)
				calcAvgDistance();
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//add a position point to the point list
	else if (msg == "point")
		{
		if (msg.args.size() == 6)
			{
			if (points)
				{
				pointData* data = new pointData;
				msg.getVec3Args(data->pos);
				msg.getVec3Args(data->ori,3);
				pfMatrix matrix;
				matrix.makeEuler(data->ori.vec[2],data->ori.vec[0],data->ori.vec[1]);
				data->vec.set(1,1,1);
				data->vec.normalize();
				data->vec.xformVec(data->vec,matrix);
				data->occupied = false;
				points->push_back(data);
				calcAvgDistance();
				}
			else
				msg.error(name(),"(no point file loaded)");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	//set relative weight of orientation
	else if (msg == "scale")
		{
		if (msg.args.size() == 1)
			scaleFactor = msg.floatArg(0);
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		valuator::message(msg);
	}

void positioner::calcAvgDistance(void)
	{
	int pointsSize = points->size();
	pointData* currPoint;
	pointData* lastPoint = (*points)[0];
	averageDist = 0;
	for (int i=1; i<pointsSize; i++)
		{
		currPoint = (*points)[i];
		float distance = currPoint->pos.distance(lastPoint->pos);
		averageDist = (averageDist*(pointsSize-1) + distance)/(float)pointsSize;
		lastPoint = currPoint;
		}
	}

void positioner::app(void)
	{
	calculateTransform();
	if (isChanged())
		{
		pfVec3 position;
		if (size >= 3)
			{
			position.vec[0] = (values[0] - startValues[0])/(endValues[0]-startValues[0]);
			position.vec[1] = (values[1] - startValues[1])/(endValues[1]-startValues[1]);
			position.vec[2] = (values[2] - startValues[2])/(endValues[2]-startValues[2]);
			}
		float x_ang, y_ang, z_ang;
		if (size >= 6)
			{
			x_ang = (values[3] - startValues[3])/(endValues[3]-startValues[3]);
			y_ang = (values[4] - startValues[4])/(endValues[4]-startValues[4]);
			z_ang = (values[5] - startValues[5])/(endValues[5]-startValues[5]);
			}
		int pointsSize = points->size();
		pfMatrix matrix;
		matrix.makeEuler(z_ang,x_ang,y_ang);
		pfVec3 vector;
		vector.set(1,1,1);
		vector.normalize();
		vector.xformVec(vector,matrix);
		float minimum = 0xffffffff;
		float euclidianDist;
		float angularDist;
		float currMeasure;
		pointData* currPoint;
		pointData* minPoint = NULL;
		for (int i=0; i<pointsSize; i++)
			{
			currPoint = (*points)[i];
			if (!currPoint->occupied || (lastPoint && currPoint == lastPoint))
				{
				euclidianDist = position.distance(currPoint->pos);
				angularDist = vector.dot(currPoint->vec);
				currMeasure = euclidianDist + ((-angularDist + 1)*scaleFactor)/(2*averageDist);
				if (currMeasure < minimum)
					{
					minimum = currMeasure;
					minPoint = currPoint;
					}
				}
			}
		if (minPoint)
			{
			if (size >= 3)
				{
				values[0] = startValues[0] + minPoint->pos.vec[0]*(endValues[0]-startValues[0]);
				values[1] = startValues[1] + minPoint->pos.vec[1]*(endValues[1]-startValues[1]);
				values[2] = startValues[2] + minPoint->pos.vec[2]*(endValues[2]-startValues[2]);
				}
			if (size >= 6)
				{
				values[3] = startValues[3] + minPoint->ori.vec[0]*(endValues[3]-startValues[3]);
				values[4] = startValues[4] + minPoint->ori.vec[1]*(endValues[4]-startValues[4]);
				values[5] = startValues[5] + minPoint->ori.vec[2]*(endValues[5]-startValues[5]);
				}
			for (int i=0; i<pointsSize; i++)
				{
				currPoint = (*points)[i];
				if (lastPoint && currPoint->pos == lastPoint->pos)
					currPoint->occupied = false;
				if (currPoint->pos == minPoint->pos)
					currPoint->occupied = true;
				}
			lastPoint = minPoint;
			}
		else
			cout << "ERROR: " << name() << " has no unoccupied points" << endl;
		}
	value::app();
	updateTransform();
	}
	
pointList* positioner::loadPointList(const ygString& file)
	{
	char *searchPath = ".",line[512];
	ygString fullPath;
	FILE *fp=NULL;
	if (getenv("YG_PATH"))
		searchPath = getenv("YG_PATH");
	if (!ygFindFile(file,searchPath,fullPath))
		{
		fprintf(stderr,"ERROR: positioner::loadPoints: could not find file '%s'\n",
			file.c_str());
		return NULL;
		}
	fp = fopen(fullPath.c_str(),"r");
	if (!fp)
		{
		fprintf(stderr,"ERROR: positioner::loadPoints could not open \"%s\"\n",
			fullPath.c_str());
		perror(fullPath.c_str());
		return NULL;
		}
	pointList* list = new pointList;
	list->clear();
	while (fgets(line,sizeof(line),fp))
		{
		pointData* data = new pointData;
		sscanf(line,"%f%f%f%f%f%f",&data->pos.vec[0],&data->pos.vec[1],&data->pos.vec[2],&data->ori.vec[0],&data->ori.vec[1],&data->ori.vec[2]);
		pfMatrix matrix;
		matrix.makeEuler(data->ori.vec[2],data->ori.vec[0],data->ori.vec[1]);
		data->vec.set(1,1,1);
		data->vec.normalize();
		data->vec.xformVec(data->vec,matrix);
		data->occupied = false;
		list->push_back(data);
		}
	fclose(fp);
	return list;
	}

	
pointList* positioner::listReference(const ygString& file)
	{
	if (!p_)
		p_ = new struct _positionerPrivateData;
	pointList* list;
	list = p_->lists[file];
	if (!list)
		{
		list = loadPointList(file.c_str());
		p_->lists[file] = list;
		}
	return list;
	}
