// Description: allows the loading and animation of poser motion file
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> an associated map file with a model for each joint is required
//<li> the set/position message takes a value between 0 and 1
//<li> the set/position message is useful for moving smoothly between all keyframes
//</ul>
//
// Category: Transformation
// Author: Shalini Venkataraman
// Revision: 11/01/01
//
#include <Performer/pf/pfDCS.h>
#include <Performer/pfdu.h>
#include <string>
#include <strstream>
#include <iostream>
#include <ygUtil.h>
#include <ygNetKeys.h>
#include <ygPFObjectCache.h>
#include "poserAnimator.h"

using namespace std;

#define MAX_STRINGLENGTH 8000

extern "C" ygNode* construct_poserAnimator(const char* name,bool master) { return new poserAnimator(name,master); }

//to define the different channel types a particular joint may have
//depending on the degree of freedom it has
#define CHANNEL_TYPE_NONE		0		// no channel
#define CHANNEL_TYPE_S			1		// scale (X Y Z) only
#define CHANNEL_TYPE_T			2		// translation only (X Y Z) 
#define CHANNEL_TYPE_RXYZ		3		// rotation (X Y Z) 
#define CHANNEL_TYPE_RXZY       4		// rotation (X Z Y) 
#define CHANNEL_TYPE_RYXZ	    5		// rotation (Y X Z) 
#define CHANNEL_TYPE_RYZX		6		// rotation (Y Z X) 
#define CHANNEL_TYPE_RZYX	    7		// rotation (Z Y X) 
#define CHANNEL_TYPE_RZXY		8		// rotation (Z X Y) 
#define CHANNEL_TYPE_TRS        9       // translation (X Z Y), (X Y Z) and scale(X Y Z)
#define CHANNEL_TYPE_TRXYZ      10      // translation (X Y Z) and rotation (X Y Z) 
#define CHANNEL_TYPE_TRXZY      11      // translation (X Y Z) and rotation(X Z Y) 
#define CHANNEL_TYPE_TRYXZ      12      // translation (X Y Z) and rotation (Y X Z) 
#define CHANNEL_TYPE_TRYZX      13      // translation (X Y Z) and rotation (Y Z X) 
#define CHANNEL_TYPE_TRZYX      14      // translation (X Y Z) and rotation (Z Y X) 
#define CHANNEL_TYPE_TRZXY      15      // translation (X Y Z) and rotation (Z X Y) 

typedef struct boneData_
	{
	pfVec3 translation;
	pfMatrix* frames;
	int channel;
	} boneData;

//the no of data required to store info corresponding to the above types
static int s_Channel_Type_Size[] = {0, 3, 3, 3, 3, 3, 3, 3, 3, 9, 6, 6, 6, 6, 6, 6};

//Read a line from a given file and parse it 
//putting all the individual tokens in words and no of tokens parsed in cnt
static void readLine(FILE *fp, char *buffer, stringArray *words, int *cnt)
	{
	words->clear(); 
	fgets(buffer, MAX_STRINGLENGTH, fp);
	//divide a given string in its words; buffer is the string given and 
	//words will contains the divided strings; cnt has the no of words parsed
	istrstream line_stream(buffer);
	ygString temp;
	string temp_string;
	*cnt = 0;
	line_stream >> ws;
	while (line_stream >> temp_string) 
		{
		line_stream >> ws;
		temp = temp_string.c_str();
		(*words).push_back(temp);
		*cnt += 1;
		}   
	}

poserAnimator::poserAnimator(const char* name,bool master) : ygNode(name,master)
	{
	setClassName("poserAnimator",true);	
	addNetKey("bvh",&bvhFile,YG_NET_STRING);	
	addNetKey("map",&mapFile,YG_NET_STRING);	
	addNetKey("frame",&frame,YG_NET_INT);	
	}

poserAnimator::~poserAnimator(void)
	{	
	}

void poserAnimator::reset(void)
	{
	frame = 0;
	numFrames = 0;
	bvhFile.clear();
	mapFile.clear();
	ygNode::reset();
	}


void poserAnimator::message(const ygMessage& msg)
	{
	if (msg == "bvhfile")
		{
		//load a poser file
		if (msg.args.size() == 1)
			{
			bvhFile = msg.args[0];
			loadPoserFile(bvhFile);
			netKeyChanged("bvh");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else if (msg == "mapfile") 
		{
		//correlate the joints to their corresponding model file
		if (msg.args.size() == 1) 
			{
			mapFile = msg.args[0];
			loadMapFile(mapFile);
			netKeyChanged("map");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else if (msg == "frame")
		{
		if (msg.args.size() == 1)
			{
			frame = msg.intArg(0);
			setFrame((pfGroup*)(pfnode()->getChild(0)),msg.intArg(0));
			netKeyChanged("frame");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else if (msg == "set" || msg == "position")
		{
		if (msg.args.size() == 1)
			{
			float position = msg.floatArg(0);
			frame = (int)(position*numFrames);
			if (frame == numFrames)
				frame = numFrames-1;
			setFrame((pfGroup*)(pfnode()->getChild(0)),frame);
			netKeyChanged("frame");
			}
		else
			msg.error(name(),"(wrong number of arguments)");
		}
	else
		ygNode::message(msg);
	}

void poserAnimator::loadMapFile(const ygString& file)
	{
	char *searchPath = ".";
	ygString fullPath;
	FILE *fp = NULL;
	if (getenv("YG_PATH"))
		searchPath = getenv("YG_PATH");
	if (!ygFindFile(file,searchPath,fullPath))
		{
		fprintf(stderr,"ERROR: poserAnimator::loadPath: could not find file '%s'\n",
			file.c_str());
		return;
		}
	if ( fp = fopen(fullPath.c_str(),"rt") )
		{
		parseMapFile((pfGroup*)(pfnode()->getChild(0)),fp);
		fclose(fp);
		}
	}
	
void poserAnimator::parseMapFile(pfGroup* joint,FILE* fp) 
	{
	//loads the parts in the map file for each of the joints that they correspond to
	stringArray words;
	int cnt = 0;
	char buffer[MAX_STRINGLENGTH];
	if (joint->isExactType(pfDCS::getClassType())) 
		{
		readLine(fp, buffer,&words, &cnt); 
		if (cnt == 5)
			{
			//only if model was defined
			pfMatrix mat;
			//mat.makeIdent();
			mat.makeTrans(-atof(words[2].c_str()),atof(words[4].c_str()),-atof(words[3].c_str()));
			pfSCS* scs = new pfSCS(mat);
			joint->addChild(scs);
			pfNode* node;
			node = ygPFObjectCache::objectReference(words[1].c_str());
			node->setTravMask(PFTRAV_ISECT,0,PFTRAV_SELF|PFTRAV_DESCEND,PF_SET);
			scs->addChild(node);
			}
  		}
	if( joint->getNumChildren() > 0 )
		{
		for(int i=0;i<joint->getNumChildren();i++)
			{
			if (joint->getChild(i)->isExactType(pfDCS::getClassType()))
				{
				//only do if posertranform
				parseMapFile((pfGroup*)joint->getChild(i),fp);
				}
			}
		}   
	}

void poserAnimator::loadPoserFile(const ygString& file)
	{
	char *searchPath = ".";
	ygString fullPath;
	FILE *fp = NULL;
	if (getenv("YG_PATH"))
		searchPath = getenv("YG_PATH");
	if (!ygFindFile(file,searchPath,fullPath))
		{
		fprintf(stderr,"ERROR: poserAnimator::loadPath: could not find file '%s'\n",
			file.c_str());
		return;
		}
    char buffer[MAX_STRINGLENGTH];
    stringArray words;
    int	cnt, index;
	if ( fp = fopen(fullPath.c_str(),"rt") ) 
		{ 
        //search for the HEIRARCHY keyword. thats where the whole body starts
        readLine(fp, buffer,&words, &cnt);
        while( words[0] != "HIERARCHY" )
        	{
            readLine(fp, buffer,&words, &cnt);
        	} 

        //now load all the hierarchy information
        handlePoserHierarchy(pfnode(),fp);

        //search for motion keyword
        readLine(fp, buffer,&words, &cnt);
        while( words[0] != "MOTION" )
        	{
            readLine(fp, buffer,&words, &cnt);
        	} 
        
        //get no of frames in Animation
        readLine(fp, buffer,&words, &cnt);
		
        if( words[0] == "Frames:" && cnt == 2)
        	{
            numFrames = atoi(words[1].c_str());
			
            // get the frame time, but it is not being used right now
            fgets(buffer, MAX_STRINGLENGTH, fp);
            // Load individual frame data
            for(int i=0; i<numFrames; i++)
            	{
                readLine(fp, buffer,&words, &cnt);
                index = 0;
                handlePoserData((pfGroup*)(pfnode()->getChild(0)), &words, &index, i);
            	}
	    	words.clear();
        	}
        else //error, can not find the animation information
        	{
            fclose(fp);
            cerr << "poser Load Error: Couldn't find Frames keyword in file " << file << endl;
            return;
        	}
		
        fclose(fp);
        return;
    	}
    else 
		//could not open the file
    	{
        cout << "poser Load Error: Couldn't open sourcefile " << file << endl;
        return;
    	}
	}

void poserAnimator::handlePoserHierarchy(pfGroup* parent,FILE *fp)
	{
	/*	
		Get the Heirarchy information and build the skeleton recursively as a tree of ygNodes
		parent is the ygNode to which nodes will be added when they are parsed
		This part is responsible for interpreting the skeleton as well as the type of 
		constraint imposed on each joint - called the channel
		the dimension of the channel will affect the amount of numbers each joint requires
		in every frame
		eg - a joint constrained to only rotation&scaling and no translation will require 6 floats
	*/
    char buffer[MAX_STRINGLENGTH];
    stringArray words;
    int cnt;
    pfDCS *newbone;
    while (1) 
		{
        readLine(fp, buffer,&words, &cnt);
		//begining of a joint. call handlePoserHierarchy on this joint after attaching it to the tree
		if(( words[0] == "ROOT") || (words[0] == "JOINT") )
			{	  
			newbone = new pfDCS;
			newbone->setName(words[1].c_str());
			boneData* data = new boneData;
			newbone->setUserData(data);
			parent->addChild(newbone);
			handlePoserHierarchy(newbone,fp);
			}
		//the offset of each joint from its parent
        else if( words[0] == "OFFSET" )
			{
			if( parent != NULL )
				{
				boneData* data;
				data = (boneData*)parent->getUserData();
				data->translation[0] = atof(words[1].c_str());
				data->translation[1] = -atof(words[3].c_str());
				data->translation[2] = atof(words[2].c_str());				
				}
			}
		//the channel information contained in each joint
        else if( words[0] == "CHANNELS" )
			{
			handlePoserChannel(&words, parent);
			}
		//end of reading all the info, do nothing
        else if( words[0] == "End" )
			{        
            handlePoserHierarchy(NULL, fp);
			}
		else if( words[0] == "}" )
			{
            return;
			}
		
		if( words[0] == "ROOT" )
			return;

		words.clear();
		}
	}


void poserAnimator::handlePoserChannel(stringArray *words, pfGroup *bone)
	{
	//handle channel information; the dimensions of the joint transformation as well as order of rotation
	boneData* data;
	data = (boneData*)bone->getUserData();
	if( (*words)[1] == "9" )
		data->channel = CHANNEL_TYPE_TRS;
	else if( (*words)[1] == "6" )
		{
		if( (*words)[5] == "Xrotation" )
			{
			if( (*words)[6] == "Yrotation" )
				data->channel = CHANNEL_TYPE_TRXYZ;
			else if( (*words)[6] == "Zrotation" )
				data->channel = CHANNEL_TYPE_TRXZY;
			}
		else if( (*words)[5] == "Yrotation" )
			{
			if( (*words)[6] == "Xrotation" )
				data->channel = CHANNEL_TYPE_TRYXZ;
			else if( (*words)[6] == "Zrotation" )
				data->channel = CHANNEL_TYPE_TRYZX;
			}
		else if( (*words)[5] == "Zrotation" )
			{
			if( (*words)[6] == "Yrotation" )
				data->channel = CHANNEL_TYPE_TRZYX;
			else if( (*words)[6] == "Xrotation" )
				data->channel = CHANNEL_TYPE_TRZXY;
			}
		}
	else if( (*words)[1] == "3" )
		{
		if( (*words)[2] == "Xrotation" )
			{
			if( (*words)[3] == "Yrotation" )
				data->channel = CHANNEL_TYPE_RXYZ;
			else if( (*words)[3] == "Zrotation" )
				data->channel = CHANNEL_TYPE_RXZY;
			}
		else if( (*words)[2] == "Yrotation" )
			{
			if( (*words)[3] == "Xrotation" )
				data->channel = CHANNEL_TYPE_RYXZ;
			else if( (*words)[3] == "Zrotation" )
				data->channel = CHANNEL_TYPE_RYZX;
			}
		else if( (*words)[2] == "Zrotation" )
			{
			if( (*words)[3] == "Yrotation" )
				data->channel = CHANNEL_TYPE_RZYX;
			else if( (*words)[3] == "Xrotation" )
				data->channel = CHANNEL_TYPE_RZXY;
			}
		else if( (*words)[2] == "Xposition" )
	  		data->channel = CHANNEL_TYPE_T;
		}
	}

void poserAnimator::handlePoserData(pfGroup *bone, stringArray *words, int *index, int curFrame)
	{	
	/*
		Reads the frame by frame data and puts it into the tree of ygNodes
		bone is the joint
		words is the array of words read in earlier
		index is index within the float array of all the transformation data
		curFrame is the cur Frame no
		numFrames is the total no of frames
	*/
	boneData* data;
	data = (boneData*)bone->getUserData();
	if( curFrame == 0 )
		data->frames = new pfMatrix[numFrames];
	float *fptr;
	fptr = new float[s_Channel_Type_Size[data->channel]];
	for (int i=0; i<s_Channel_Type_Size[data->channel];i++)
		{
		fptr[i] = atof(((*words)[*index]).c_str());
		(*index)++;
		}
	setTransformation(bone,curFrame,fptr);
	delete [] fptr;
	//go through all the children
	if( bone->getNumChildren() > 0 )
		{
		for(int i=0; i<bone->getNumChildren();i++)
			{
			if (bone->getChild(i)->isExactType(pfDCS::getClassType())) 
				{
				//only do if pfDCS
				handlePoserData((pfGroup*)bone->getChild(i),words,index,curFrame);
				}
			}
		}
	}
	
void poserAnimator::setFrame(pfGroup *bone, int curFrame)
	{	
	boneData* data;
	data = (boneData*)bone->getUserData();
	pfDCS* dcs = (pfDCS*)bone;
	dcs->setMat(data->frames[curFrame]);
	//go through all the children
	if (bone->getNumChildren() > 0)
		{
		for(int i=0;i<bone->getNumChildren();i++)
			{
			if (bone->getChild(i)->isExactType(pfDCS::getClassType())) 
				{
				//only do if pfDCS
				setFrame((pfGroup*)bone->getChild(i),curFrame);
				}
			}
		}
	}

void poserAnimator::setTransformation(pfGroup *bone, int curFrame, float* fptr)
	{
	//fptr points are given in opengl coordinate space and transformed to Performer space
	boneData* data;
	data = (boneData*)bone->getUserData();
	if( curFrame >= 0 && curFrame < numFrames )
		{
		pfMatrix mat;
		mat.makeIdent();
		switch(data->channel)
			{
			case CHANNEL_TYPE_S:
				data->frames[curFrame].makeScale(fptr[0],-fptr[2],fptr[1]);
				break;
			case CHANNEL_TYPE_T:
				data->frames[curFrame].makeTrans(fptr[0],-fptr[2],fptr[1]);
				break;
			case CHANNEL_TYPE_RXYZ:
				{
				mat.makeTrans(data->translation[0],data->translation[1],data->translation[2]);
				mat.preRot(fptr[0],1.0f,0.0f,0.0f,mat);
				mat.preRot(fptr[1],0.0f,0.0f,1.0f,mat);
				mat.preRot(fptr[2],0.0f,-1.0f,0.0f,mat); 
				data->frames[curFrame].copy(mat);  
				}   
				break;
			case CHANNEL_TYPE_RZXY:
				{			
				mat.makeTrans(data->translation[0],data->translation[1],data->translation[2]);
				mat.preRot(fptr[0],0.0f,-1.0f,0.0f,mat); //opengl Z ->performer -Y
				mat.preRot(fptr[1],1.0f,0.0f,0.0f,mat); //opengl X -> performer X
				mat.preRot(fptr[2],0.0f,0.0f,1.0f,mat); //opengl Y -> performer Z
				data->frames[curFrame].copy(mat);			
				}
				break;
			case CHANNEL_TYPE_RYZX:
				{
				mat.makeTrans(data->translation[0],data->translation[1],data->translation[2]);
				mat.preRot(fptr[0],0.0f,0.0f,1.0f,mat);
				mat.preRot(fptr[1],0.0f,-1.0f,0.0f,mat);
				mat.preRot(fptr[2],1.0f,0.0f,0.0f,mat);
				data->frames[curFrame].copy(mat);	  
				}
				break;		   
			case CHANNEL_TYPE_RZYX:
				{
				mat.makeTrans(data->translation[0],data->translation[1],data->translation[2]);
				mat.preRot(fptr[0],0.0f,-1.0f,0.0f,mat);
				mat.preRot(fptr[1],0.0f,0.0f,1.0f,mat);
				mat.preRot(fptr[2],1.0f,0.0f,0.0f,mat);
				data->frames[curFrame].copy(mat);  
				}
				break;
			case CHANNEL_TYPE_RXZY:
				{
				mat.makeTrans(data->translation[0],data->translation[1],data->translation[2]);
				mat.preRot(fptr[0],1.0f,0.0f,0.0f,mat); //x
				mat.preRot(fptr[1],0.0f,-1.0f,0.0f,mat);//z
				mat.preRot(fptr[2],0.0f,0.0f,1.0f,mat);//y
				data->frames[curFrame].copy(mat); 
				}
				break;
			case CHANNEL_TYPE_RYXZ:
				{
				mat.makeTrans(data->translation[0],data->translation[1],data->translation[2]);
				mat.preRot(fptr[0],0.0f,0.0f,1.0f,mat);//y
				mat.preRot(fptr[1],1.0f,0.0f,0.0f,mat); //x
				mat.preRot(fptr[2],0.0f,-1.0f,0.0f,mat);//z
				data->frames[curFrame].copy(mat);
				}
				break;
			case CHANNEL_TYPE_TRS:
				{
				mat.makeTrans(fptr[0],-fptr[2],fptr[1]);
				mat.preRot(fptr[3],1.0f,0.0f,0.0f,mat); //x
				mat.preRot(fptr[4],0.0f,0.0f,1.0f,mat);//y
				mat.preRot(fptr[5],0.0f,-1.0f,0.0f,mat);//z
				mat.makeScale(fptr[6],-fptr[8],fptr[7]);
				data->frames[curFrame].copy(mat);  
				} 
				break;
			case CHANNEL_TYPE_TRXYZ:
				{
				mat.makeTrans(fptr[0],-fptr[2],fptr[1]);
				mat.preRot(fptr[3],1.0f,0.0f,0.0f,mat); //x
				mat.preRot(fptr[4],0.0f,0.0f,1.0f,mat);//y
				mat.preRot(fptr[5],0.0f,-1.0f,0.0f,mat);//z
				data->frames[curFrame].copy(mat); 
				}  
				break;
			case CHANNEL_TYPE_TRZXY:
				{
				mat.makeTrans(fptr[0],-fptr[2],fptr[1]);
				mat.preRot(fptr[3],0.0f,-1.0f,0.0f,mat); //z
				mat.preRot(fptr[4],1.0f,0.0f,0.0f,mat); //x
				mat.preRot(fptr[5],0.0f,0.0f,1.0f,mat);//y
				data->frames[curFrame].copy(mat);				
				}
				break;
			case CHANNEL_TYPE_TRYZX:
				{
				mat.makeTrans(fptr[0],-fptr[2],fptr[1]);
				mat.preRot(fptr[3],0.0f,0.0f,1.0f,mat);//y
				mat.preRot(fptr[4],0.0f,-1.0f,0.0f,mat); //z
				mat.preRot(fptr[5],1.0f,0.0f,0.0f,mat); //x
				data->frames[curFrame].copy(mat);
				}
				break;
			case CHANNEL_TYPE_TRZYX:
				{
				mat.makeTrans(fptr[0],-fptr[2],fptr[1]);
				mat.preRot(fptr[3],0.0f,-1.0f,0.0f,mat); //z
				mat.preRot(fptr[4],0.0f,0.0f,1.0f,mat);//y
				mat.preRot(fptr[5],1.0f,0.0f,0.0f,mat); //x
				data->frames[curFrame].copy(mat);
				}
				break;
			case CHANNEL_TYPE_TRXZY:
				{
				mat.makeTrans(fptr[0],-fptr[2],fptr[1]);
				mat.preRot(fptr[3],1.0f,0.0f,0.0f,mat); //x
				mat.preRot(fptr[4],0.0f,-1.0f,0.0f,mat); //z
				mat.preRot(fptr[5],0.0f,0.0f,1.0f,mat);//y
				data->frames[curFrame].copy(mat);
				}
				break;
			case CHANNEL_TYPE_TRYXZ:
				{
				mat.makeTrans(fptr[0],-fptr[2],fptr[1]);
				mat.preRot(fptr[3],0.0f,0.0f,1.0f,mat);//y
				mat.preRot(fptr[4],1.0f,0.0f,0.0f,mat); //x
				mat.preRot(fptr[5],0.0f,-1.0f,0.0f,mat); //z
				data->frames[curFrame].copy(mat);
				}
				break;
			}
		if (curFrame == 0)
			{
			pfDCS* dcs = (pfDCS*)bone;
			dcs->setMat(data->frames[0]);
			}
		}
	}

void poserAnimator::acceptNetKey(const ygString& key)
    {
    if (key == "bvh")
		loadPoserFile(bvhFile);
    else if (key == "map")
		loadMapFile(mapFile);
    else if (key == "frame")
		setFrame((pfGroup*)(pfnode()->getChild(0)),frame);
    else
		ygNode::acceptNetKey(key);
    }
