// Description: a transform that moves a percentage along a path of points and orientations
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the list of points can be created by a combination of data file and/or point messages
//<li> the data file can be any number of X Y Z coordinate and X Y Z orientation triplets separated by spaces
//<li> the set/value/position message establishes the decimal percentage between the first and last points
//</ul>
//
// Category: Transformation
// Author: Alex Hill
// Revision: 11/01/02
//
#include <ygUtil.h>
#include <ygWorld.h>
#include "pointMover.h"

extern "C" ygNode* construct_pointMover(const char* name,bool master) { return new pointMover(name,master); }

class pointMoverCatmullRomCurve
	{
	public:
	 pointMoverCatmullRomCurve(void);
	 void setControlPoints(const pfVec3&,const pfVec3&,const pfVec3&,const pfVec3&);
	 pfVec3 value(float t);
	 pfVec3 tangent(float t);
	protected:
	 void computeCoefficients(void);
	 pfVec3 cpt_[4];
	 pfVec4 coeff_[3];
	};

pointMover::pointMover(const char* name,bool master) : ygSimpleTransform(name,master)
	{
	setClassName("pointMover");
	position = 0.0;
	}

pointMover::~pointMover(void)
	{
	}

void pointMover::reset(void)
	{
	//set the position to 0.0
	position = 0.0;
	//clear the points list
	points.clear();
	ygSimpleTransform::reset();
	}

void pointMover::message(const ygMessage& msg)
	{
	//load a point file
	if (msg == "file" || msg == "path")
		loadPath(msg.args[0]);
	//add a point to the point list
	else if (msg == "point")
		{
		ygPosQuat* q = new ygPosQuat;
		msg.getVec3Args(q->pos);
		pfVec3 rot;
		msg.getVec3Args(rot,3);
		pfMatrix mat;
		mat.makeEuler(rot[2],rot[0],rot[1]);
		mat.getOrthoQuat(q->quat);
		points.push_back(q);
		interpolate(position);
		}
	//set the percentage along the point path
	else if (msg == "set" || msg == "value" || msg == "position")
		{
		position = msg.floatArg(0);
		interpolate(position);
		}
	else
		ygSimpleTransform::message(msg);
	}

void pointMover::loadPath(const ygString& file)
	{
	char *searchPath = ".", line[512];
	ygString fullPath;
	FILE *fp=NULL;
	if (getenv("YG_PATH"))
		searchPath = getenv("YG_PATH");
	if (!ygFindFile(file,searchPath,fullPath))
		{
		fprintf(stderr,"ERROR: pointMover::loadPath: could not find file '%s'\n",
			file.c_str());
		return;
		}
	fp = fopen(fullPath.c_str(),"r");
	if (!fp)
		{
		fprintf(stderr,"ERROR: pointMover::loadPath could not open \"%s\"\n",
			fullPath.c_str());
		perror(fullPath.c_str());
		return;
		}
	points.clear();
	while (fgets(line,sizeof(line),fp))
		{
		ygPosQuat* q = new ygPosQuat;
		pfVec3 rot;
		sscanf(line,"%f%f%f%f%f%f",&q->pos.vec[0],&q->pos.vec[1],&q->pos.vec[2],&rot.vec[0],&rot.vec[1],&rot.vec[2]);
		pfMatrix mat;
		mat.makeEuler(rot[2],rot[0],rot[1]);
		mat.getOrthoQuat(q->quat);
		points.push_back(q);
		}
	fclose(fp);
	interpolate(position);
	}

void pointMover::interpolate(float t)
	{
	int numPoints = points.size();
	float fIndex = t * (numPoints-1);
	int index = (int)(fIndex);
	float frac = fIndex - index;
	if (index < 0)
		{
		index = 0;
		frac = 0;
		}
	else if (index > numPoints-2)
		{
		index = numPoints-2;
		frac = 1;
		}
	pointMoverCatmullRomCurve curve;
	pfVec3 cpt0,cpt1,cpt2,cpt3;
	cpt1 = (*points[index]).pos;
	cpt2 = (*points[index+1]).pos;
	if (index > 0)
		cpt0 = (*points[index-1]).pos;
	else
		cpt0 = cpt1;
	if (index < numPoints-2)
		cpt3 = (*points[index+2]).pos;
	else
		cpt3 = cpt2;
	curve.setControlPoints(cpt0,cpt1,cpt2,cpt3);
	pfVec3 pos;
	pos = curve.value(frac);

	ygPosQuat beginQuat;
	ygPosQuat endQuat;
	beginQuat = *points[index];
	endQuat = *points[index+1];

	pfMatrix mat;
	getMatrix(mat);
	pfCoord coord;
	ygPosQuat currQuat;
	mat.getOrthoCoord(&coord);
	currQuat.pos = coord.xyz;
	mat.getOrthoQuat(currQuat.quat);
	//interpolate the position
	currQuat.pos = beginQuat.pos + frac * (endQuat.pos - beginQuat.pos);
	//interpolate the orientation with quaternians
	currQuat.quat.slerp(frac,beginQuat.quat,endQuat.quat);
	mat.makeQuat(currQuat.quat);
	mat.postTrans(mat,pos[0],pos[1],pos[2]);
	//set the matrix
	setMatrix(mat);
	}

pointMoverCatmullRomCurve::pointMoverCatmullRomCurve(void)
	{
	cpt_[0].set(0,0,0);
	cpt_[1].set(0,0,0);
	cpt_[2].set(0,0,0);
	cpt_[3].set(0,0,0);
	coeff_[0].set(0,0,0,0);
	coeff_[1].set(0,0,0,0);
	coeff_[2].set(0,0,0,0);
	}

void pointMoverCatmullRomCurve::setControlPoints(const pfVec3& p0,const pfVec3& p1,const pfVec3& p2,const pfVec3& p3)
	{
	cpt_[0] = p0;
	cpt_[1] = p1;
	cpt_[2] = p2;
	cpt_[3] = p3;
	computeCoefficients();
	}

pfVec3 pointMoverCatmullRomCurve::value(float t)
	{
	pfVec4 T;
	pfVec3 val;
	T.set(t*t*t,t*t,t,1.0f);
	val[0] = T.dot(coeff_[0]);
	val[1] = T.dot(coeff_[1]);
	val[2] = T.dot(coeff_[2]);
	return val;
	}

void pointMoverCatmullRomCurve::computeCoefficients(void)
	{
	pfMatrix CatmullRom (  -0.5,  1.0, -0.5,  0.0,
				 1.5, -2.5,  0.0,  1.0,
				-1.5,  2.0,  0.5,  0.0,
				 0.5, -0.5,  0.0,  0.0 );
	pfVec4 geom;
	geom.set(cpt_[0][0], cpt_[1][0], cpt_[2][0], cpt_[3][0]);
	coeff_[0].xform(geom,CatmullRom);
	geom.set(cpt_[0][1], cpt_[1][1], cpt_[2][1], cpt_[3][1]);
	coeff_[1].xform(geom,CatmullRom);
	geom.set(cpt_[0][2], cpt_[1][2], cpt_[2][2], cpt_[3][2]);
	coeff_[2].xform(geom,CatmullRom);
	}
