// Description: a transform that moves a percentage along a path of points
//
//<b>notes:</b>
//<ul>
//<li> derived nodes inherit all messages from their base classes
//<li> see reset method for default settings
//<li> the list of points can be created by any combination of point messages and files
//<li> the point file can be any number of X Y Z corrdinate triplets separated by spaces
//<li> the orientation at a point is along the line connecting the next and last points
//<li> the set/value/position message establishes the decimal percentage between the first and last points
//</ul>
//
// Category: Transformation
// Author: Alex Hill
//           11/01/01
// Revision: 12/20/02 Alex Hill - modified to change heading and pitch
//
#include <ygUtil.h>
#include <ygWorld.h>
#include "pointFollower.h"

extern "C" ygNode* construct_pointFollower(const char* name,bool master) { return new pointFollower(name,master); }

class pointFollowerCatmullRomCurve
	{
	public:
	 pointFollowerCatmullRomCurve(void);
	 void setControlPoints(pfVec3,pfVec3,pfVec3,pfVec3);
	 pfVec3 value(float t);
	 pfVec3 tangent(float t);
	protected:
	 void computeCoefficients(void);
	 pfVec3 cpt_[4];
	 pfVec4 coeff_[3];
	};


pointFollower::pointFollower(const char* name,bool master) : ygSimpleTransform(name,master)
	{
	setClassName("pointFollower");
	position = 0.0;
	}

pointFollower::~pointFollower(void)
	{
	}

void pointFollower::reset(void)
	{
	//set the position to 0.0
	position = 0.0;
	//clear the points list
	points.clear();
	ygSimpleTransform::reset();
	}

void pointFollower::message(const ygMessage& msg)
	{
	//load a point file
	if (msg == "file" || msg == "path")
		loadPath(msg.args[0]);
	//add a point to the point list
	else if (msg == "point")
		{
		pfVec3 *v = new pfVec3;
		msg.getVec3Args(*v);
		points.push_back(v);
		interpolate(position);
		}
	//set the percentage along the point path
	else if (msg == "set" || msg == "value" || msg == "position")
		{
		position = msg.floatArg(0);
		interpolate(position);
		}
	else
		ygSimpleTransform::message(msg);
	}

void pointFollower::loadPath(const ygString& file)
	{
	char *searchPath = ".", line[512];
	ygString fullPath;
	FILE *fp=NULL;
	if (getenv("YG_PATH"))
		searchPath = getenv("YG_PATH");
	if (!ygFindFile(file,searchPath,fullPath))
		{
		fprintf(stderr,"ERROR: pointFollower::loadPath: could not find file '%s'\n",
			file.c_str());
		return;
		}
	fp = fopen(fullPath.c_str(),"r");
	if (!fp)
		{
		fprintf(stderr,"ERROR: pointFollower::loadPath could not open \"%s\"\n",
			fullPath.c_str());
		perror(fullPath.c_str());
		return;
		}
	points.clear();
	while (fgets(line,sizeof(line),fp))
		{
		pfVec3 *v = new pfVec3;
		sscanf(line,"%f%f%f",&v->vec[0],&v->vec[1],&v->vec[2]);
		points.push_back(v);
		}
	fclose(fp);
	interpolate(position);
	}

void pointFollower::interpolate(float t)
	{
	pfVec3 pos, dir;
	float fIndex, frac;
	int index, numPoints = points.size();
	fIndex = t * (numPoints-1);
	index = (int)(fIndex);
	frac = fIndex - index;
	if (index < 0)
		{
		index = 0;
		frac = 0;
		}
	else if (index > numPoints-2)
		{
		index = numPoints-2;
		frac = 1;
		}
	pointFollowerCatmullRomCurve curve;
	pfVec3 cpt0,cpt1,cpt2,cpt3;
	cpt1 = *points[index];
	cpt2 = *points[index+1];
	if (index > 0)
		cpt0 = *points[index-1];
	else
		cpt0 = cpt1;
	if (index < numPoints-2)
		cpt3 = *points[index+2];
	else
		cpt3 = cpt2;
	curve.setControlPoints(cpt0,cpt1,cpt2,cpt3);
	pos = curve.value(frac);
	dir = curve.tangent(frac);
	float heading, pitch;
	pfVec3 front;
	front.set(0,1,0);
	pfMatrix mat;
	dir.normalize();
	heading = pfArcTan2(-dir[0],dir[1]);
	pitch = pfArcSin(dir[2]);
	setPosition(pos);
	setOrientation(pitch,0,heading);
	}

pointFollowerCatmullRomCurve::pointFollowerCatmullRomCurve(void)
	{
	cpt_[0].set(0,0,0);
	cpt_[1].set(0,0,0);
	cpt_[2].set(0,0,0);
	cpt_[3].set(0,0,0);
	coeff_[0].set(0,0,0,0);
	coeff_[1].set(0,0,0,0);
	coeff_[2].set(0,0,0,0);
	}

void pointFollowerCatmullRomCurve::setControlPoints(pfVec3 p0,pfVec3 p1,pfVec3 p2,pfVec3 p3)
	{
	cpt_[0] = p0;
	cpt_[1] = p1;
	cpt_[2] = p2;
	cpt_[3] = p3;
	computeCoefficients();
	}

pfVec3 pointFollowerCatmullRomCurve::value(float t)
	{
	pfVec4 T;
	pfVec3 val;
	T.set(t*t*t,t*t,t,1.0f);
	val[0] = T.dot(coeff_[0]);
	val[1] = T.dot(coeff_[1]);
	val[2] = T.dot(coeff_[2]);
	return val;
	}

pfVec3 pointFollowerCatmullRomCurve::tangent(float t)
	{
	pfVec4 T;
	pfVec3 val;
	T.set(3.0f*t*t,2.0f*t,1.0f,0.0f);
	val[0] = T.dot(coeff_[0]);
	val[1] = T.dot(coeff_[1]);
	val[2] = T.dot(coeff_[2]);
	return val;
	}

void pointFollowerCatmullRomCurve::computeCoefficients(void)
	{
	pfMatrix CatmullRom (  -0.5,  1.0, -0.5,  0.0,
				 1.5, -2.5,  0.0,  1.0,
				-1.5,  2.0,  0.5,  0.0,
				 0.5, -0.5,  0.0,  0.0 );
	pfVec4 geom;
	geom.set(cpt_[0][0], cpt_[1][0], cpt_[2][0], cpt_[3][0]);
	coeff_[0].xform(geom,CatmullRom);
	geom.set(cpt_[0][1], cpt_[1][1], cpt_[2][1], cpt_[3][1]);
	coeff_[1].xform(geom,CatmullRom);
	geom.set(cpt_[0][2], cpt_[1][2], cpt_[2][2], cpt_[3][2]);
	coeff_[2].xform(geom,CatmullRom);
	}
